package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.entity.Field;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;

/**
 * Aolai基础框架相关服务层，基于Mybatis整合的元数据的基础操作类(CRUD)
 *
 * @author Aolai
 * @version 1.0 $Date: 2023.8.20
 * @since 1.7
 */

@Service
public class AolaiService extends GmetaService {

    private static Logger log = LoggerFactory.getLogger(AolaiService.class);

    /**
     * 从GMETA元数据表中获取元数据字段属性信息一览
     */
    public Map<String, Object> getMetaList() {
		String[] keys = { Const.TABLE };
		Map<String, Object> params = utils.jsonParams(keys);
		Object table = params.get(Const.TABLE);

        // 根据表名获取获取元数据字段
        return utils.success(getTableMeta(table));
    }

    /**
     * 增加元数据项，或修改GMETA元数据表的信息
     */
    public Map<String, String> addMetaItem() {
        // 获取GMETA元数据的主键，并判断传递参数是否存在元数据主键
        // 包含META_TABLE主键一个也不能少
		Map<String, String> metas = ConfUtil.gMETAS();
		String[] keys = metas.keySet().toArray(new String[metas.size()]);
		Map<String, Object> data = utils.jsonParams(keys);

        // GMETA无相关表数据
        Map<String, Object> cond = new HashMap<>();
        cond.put(Const.TABLE, data.get(Const.TABLE));
        if (!exists(Const.GMETA, cond))
            return utils.invalidParams();
        // 增加排序序号
        data.put(Const.SORT, count(Const.GMETA, cond));

        // GMETA已有重复数据
        Map<String, String> meta = utils.obj2Map(data);
        String field = meta.get(Const.FIELD); // USER_TEST
        // GMETA已有重复数据
        cond.put(Const.FIELD, field);
        if (exists(Const.GMETA, cond))
            throw new FailedException(ConfUtil.DUPL_DATA);

        // 无重复数据，则增加相关表字段
        Map<String, String> map = new HashMap<>();
        map.put(Const.TABLE, tableOf(meta.get(Const.TABLE)));

        // 新建表字段，并满足国际化支持：meta.remove(LANG);
        if ("1".equals(meta.get(Const.LANG))) {
            for (String locale : ConfUtil.locales()) {
                map.put(Const.FIELD, setField(field + UL + locale, meta));
                dao.addCol(map);
            }
        } else {
            map.put(Const.FIELD, setField(field, meta));
            dao.addCol(map);
        }
        // 再往GMETA表中插入一条数据，并标识为自定义字段
        data.put(Const.STYLE, "{\"" + Const.CUSTOM + "\":\"1\"}");
        return addRecord(Const.GMETA, data, true);
    }

    /**
     * 删除GMETA元数据表中的一条数据记录
     */
    public Map<String, String> delMetaItem() {
        String[] keys = {Const.TABLE, Const.ALIAS};
        Map<String, Object> params = utils.jsonParams(keys);

        // 获取删除数据
        Map<String, String> meta = findOne(Const.GMETA, params);
        if (utils.isFailed(meta))
            return utils.invalidParams();
        Map<String, String> style = utils.json2Map(meta.get(Const.STYLE));
        // 不能删除非自定义数据
        if (!"1".equals(style.get(Const.CUSTOM)))
            throw new FailedException(ConfUtil.CANT_REMOVE);

        // 获取FIELD已启用的数据
        Map<String, Object> cond = new HashMap<>();
        cond.put(Field.ORIGIN, params.get(Const.TABLE));
        cond.put(Field.ALIAS, params.get(Const.ALIAS));
        if (exists(Field.TABLE, cond))
            throw new FailedException(ConfUtil.DATA_USED);

        // 获取表中已有数据，条件例如：userTest <>'' or userTest is null
        Map<String, Object> where = new HashMap<>();
        String alias = meta.get(Const.ALIAS);
        String table = meta.get(Const.TABLE);
        if (Const.NULL.equals(meta.get(Const.DEFAULT))) {
            where.put(utils.pHolder(alias, "IS NOT NULL"), "");
        } else {
            where.put(utils.pHolder(alias, "<>"), meta.get(Const.DEFAULT));
        }
        if (exists(table, where))
            throw new FailedException(ConfUtil.DATA_USED);

        // 先删除元数据
        if (utils.isSuccess(delete(Const.GMETA, cond))) {
            // 清除缓存
            redis.clearDto(table, rdsDbid());

            // 再删除已有的表字段，并清除相关的dto缓存：m.table*
            Map<String, String> map = new HashMap<>();
            map.put(Const.TABLE, tableOf(table));
            map.put(Const.FIELD, meta.get(Const.FIELD));
            return utils.result(dao.dropCol(map) > 0);
        }
        return utils.invalidParams();
    }

    /**
     * 删除GMETA元数据表中的多条数据记录
     */
/*	public Map<String, String> delMetaItems() {
		Map<String, Object> params = utils.jsonParams();
		// 获取删除数据的条件
		List<Map<String, Object>> list = utils.obj2List(params.get(Const.ITEMS));
		if (list.isEmpty())
			throw new FailedException();
		// list中的数据为逐条删除数据的条件
		// 清除与meta信息相关的dto缓存
		return cleanDto(batchDelete(null, Const.GMETA, list, null));
	}*/

    /**
     * 为统一处理，这里清除了所有base中与GMETA信息相关的dto缓存
     */
/*	<T> Map<String, T> cleanDto(Map<String, T> result) {
		return cleanDto(result, "");
	}*/

    /**
     * 清除所有base中与FIELD信息相关的dto缓存
     */
/*	<T> Map<String, T> cleanDto(Map<String, T> result, Object base) {
		if (utils.isSuccess(result))
			cleanDto(ConfUtil.isCustomMeta() ? base : "");
		return result;
	}*/

    /**
     * 清除所有与元数据Dto相关的缓存
     */
/*	private void cleanDtoRds(Object table) {
		redis.delKeys(Const.RDS_META + table + tenant());
	}*/

    /*********************************************************
     * 分页查询：支持多张表 join关联查询：left="left"为左连接
     */
	public Map<String, Object> queryList(Object base, Map<String, Map<String, String>> tables,
			String uriKey, Map<String, Object> cond, Map<String, String> sort, String limit) {
		return queryList(base, tables, uriKey, cond, sort, LEFT, limit);
	}

    /**
     * 支持多张表 join关联查询：left=""为内连接，查询条件需放到别名列表中
     */
	public Map<String, Object> queryList(Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> cond, Map<String, String> sort, String limit) {
		return queryList(null, tables, uriKey, cond, sort, LEFT, limit);
	}

    /**
     * 支持多张表 join关联查询：left=""为内连接，查询条件需放到别名列表中
     */
	public Map<String, Object> queryList(Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> cond, Map<String, String> sort) {
		return queryList(null, tables, uriKey, cond, sort, LEFT, "0");
	}

    /**
     * 支持多张表 join关联查询：left=""为内连接，查询条件需放到别名列表中
     *
     * @param tables 两表名为key，两表的关联主键的映射，表1设为null
     * @param uriKey 前台上下文或接口名，仅缓存数据用
     * @param cond   综合多表的查询条件，分页时需要pageNo、pageSize
     * @param limit  限制每页输出行数，为“0”时不限行数
     */
	public Map<String, Object> queryList(Object base, Map<String, Map<String, String>> tables,
			String uriKey, Map<String, Object> cond, Map<String, String> sort, String left,
			String limit) {

		// 后面的 setArgsAndCond 要使用，getDtoBy(uriKey)未必包含cond要使用的字段
		Map<String, String> allDto = new HashMap<>();
//		for (String table : tables.keySet()) {
        for (Map.Entry<String, Map<String, String>> e : tables.entrySet()) {
			allDto.putAll(getDto(e.getKey()));
		}
		// 这里的table为join后多个关联的表
		String[] table = joinTables(base, allDto, tables, left);

		// 数据库查询参数：table,alias,sort,joint,limit,floor
		Map<String, String> args = queryArgs(table[1], uriKey, limit);
		// 处理args.get("alias")和查询条件
		// “别名”设为：['userLang', 'userMail', 'userName', ...]
		cond = setArgsAndCond(args, allDto, cond, sort);

        /** 增加了租户数据隔离 Jun.22,2024 */
		// { "TABLE_TENANT =": "889" }
        setDataIsolation(cond, table[0]); // 主表table[0]

        // 返回数据总行数：T1 left join T2 on T2.A=T1.B
        int total = dao.countRows(args, cond);
        log.debug("total...{}", total);

        // 返回查询结果
		Map<String, String> dto = getDtoBy(allDto, uriKey);
		// 若dto的元数据携带style信息，根据blur模糊处理敏感数据
        return splitPage(total, setBlur(dao.findList(args, cond), dto));
    }

    /**
     * 数据分页显示
     */
    private Map<String, Object> splitPage(int total, List<Map<String, String>> items) {
        Map<String, Object> result = new HashMap<>();
        result.put(Const.TOTAL, total);
        result.put(Const.ITEMS, items);
        return result;
    }

    /*********************************************************
     * 查询数据一览，限定了最大返回记录数（limit.rows）<br>
     * queryList返回全部数据总行数total
     */
	public Map<String, Object> queryList(Map<String, String> args, Map<String, Object> cond,
			Map<String, String> sort) {
        String table = args.get(Const.TABLE);
        String uriKey = args.get(Const.ALIAS);

		// 后面的 setArgsAndCond 要使用，getDtoBy(uriKey)未必包含cond要使用的字段
		Map<String, String> allDto = getDto(table);

		// 处理args.get("alias")和查询条件
		// “别名”设为：['userLang', 'userMail', 'userName', ...]
		cond = setArgsAndCond(args, allDto, cond, sort);

//		Object base = baseOf(args);
        args.put(Const.TABLE, tableOf(table));

        /** 增加了租户数据隔离 Jun.22,2024 */
        setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

        // 返回数据总行数
        int total = count(null, table, cond);

        // 返回查询结果
		Map<String, String> dto = getDtoBy(allDto, uriKey);
		// 若dto的元数据携带style信息，根据blur模糊处理敏感数据
        return splitPage(total, setBlur(dao.findList(args, cond), dto));
    }

    /**
     * 查询数据一览，限定了最大返回记录数（limit.rows）<br>
     * queryList返回全部数据总行数total
     *
     * @deprecated
     */
	@Deprecated
    public Map<String, Object> queryList(Map<String, String> args, Map<String, Object> cond) {
        return queryList(args, cond, null);
    }

    /**
     * 查询数据一览<br>
     * queryList返回全部数据总行数total，uriKey为传回前端的参数数组
     */
	Map<String, Object> queryList(Object base, String table, String uriKey,
			Map<String, Object> cond, Map<String, String> sort) {
        // 把base参数传递给
        if (base != null && !cond.containsKey(Const.BASE))
            cond.put(Const.BASE, base);
        String[] args = {null, table, uriKey};
        return queryAll(cond, sort, args);
    }

    /**
     * 查询数据一览<br>
     * queryList返回全部数据总行数total，uriKey为传回前端的参数数组
     */
	public Map<String, Object> queryList(String table, String uriKey, Map<String, Object> cond,
			Map<String, String> sort) {
		return queryList(null, table, uriKey, cond, sort);
	}

    /**
     * 查询数据一览<br>
     * queryList返回全部数据总行数total
     *
     * @deprecated
     */
	@Deprecated
	public Map<String, Object> queryList(Object base, String table, String uriKey,
			Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return queryList(base, table, uriKey, cond, null);
	}

    /**
     * 查询数据一览<br>
     * queryList返回全部数据总行数total
     */
    public Map<String, Object> queryList(String table, String uriKey, Map<String, Object> cond) {
        // uriKey为传回前端的参数数组
        return queryList(null, table, uriKey, cond, null);
    }

    /**
     * 查询数据一览<br>
     * queryList返回全部数据总行数total
     */
    public Map<String, Object> queryList(String table, Map<String, Object> cond) {
        return queryList(table, null, cond);
    }

    /**
     * 基于查询条件args：base,table,alias,limit的数据集，用以支持limit="0"的情况<br>
     * queryList返回全部数据总行数total
     */
    public Map<String, Object> queryAll(Map<String, Object> cond, Map<String, String> sort,
                                        String[] args) {
        Map<String, String> map = convertArgs(args);
        if (map == null)
            return splitPage(0, new ArrayList<>());
        else
            return queryList(map, cond, sort);
    }

    /**
     * 默认关联“多个表”查询时，检索出全部数据
     */
    public List<Map<String, String>> findAll(Object base, Map<String, Map<String, String>> tables,
                                             String uriKey, Map<String, Object> cond, Map<String, String> sort) {
        return findAll(base, tables, uriKey, cond, sort, LEFT);
    }

    /**
     * 默认关联“多个表”查询时，检索出全部数据
     */
    public List<Map<String, String>> findAll(Map<String, Map<String, String>> tables,
                                             String uriKey, Map<String, Object> cond, Map<String, String> sort) {
        return findAll(null, tables, uriKey, cond, sort, LEFT);
    }

    /**
     * 默认关联“多个表”查询时，检索出全部数据
     */
    public List<Map<String, String>> findAll(Object base, Map<String, Map<String, String>> tables,
                                             String uriKey, Map<String, Object> cond, Map<String, String> sort, String left) {
        return findList(base, tables, uriKey, cond, sort, left, "0");
    }

    /**
     * 默认关联“多个表”查询时，检索出全部数据
     */
    public List<Map<String, String>> findAll(Map<String, Map<String, String>> tables, String uriKey,
                                             Map<String, Object> cond, Map<String, String> sort, String left) {
        return findAll(null, tables, uriKey, cond, sort, left);
    }

    /**
     * 这里增加树状结构的一个节点，在父级节点下增加一个子节点
     */
    public Map<String, String> addTreeNode(Map<String, Object> item, String[] keys, int tierW) {
        /**
         * 表名：0-TABLE, 公司编号：1-COMP, 节点编号：2-ID, 父级节点：3-PID, 叶子节点：4-LEAF,
         * 层级：5-LEVEL, 当前操作用户：6-OPUID
         */
        String[] newKeys = new String[]{keys[1], keys[1], Const.I18N};
        Map<String, Object> cond = utils.sameIf(item, 0, newKeys);
        // 按节点编号升序排序
        Map<String, String> sort = new HashMap<>();
        sort.put(keys[2], "ASC");

        // 查询数据一览
        String base = keys.length > 7 ? keys[7] : null;
        String[] args = {base, keys[0], null, Const.S_0};
        List<Map<String, String>> list = findAll(cond, sort, args);

        // 生成部门编号，默认部门编号：DEFAULT_DEPT=001
        // 每层的部门编号位数 001001 类似
        // 部门数量超限
        int w = tierW;
        int limit = 1;
        while (w-- > 0)
            limit = limit * 10;
        if (list.size() == limit - 1)
            throw new FailedException(ConfUtil.OVER_COUNTS);

        // 根据父级编号，生成新节点编号
        Object pid = item.get(keys[3]); // PID
        String itemId = utils.newLeafId(pid, list, keys[2], tierW);
        // 赋值节点
        item.put(keys[2], itemId); // ID
        item.put(keys[4], Const.S_1); // LEAF
        item.put(keys[5], itemId.length() / tierW); // LEVEL
        // 增加一条新记录
        try {
            Map<String, String> result = addRecord(base, keys[0], item, true);
            if (!utils.isSuccess(result))
                utils.doFailed();
        } catch (Exception e) {
            log.error("addTreeNode...{}", item);
            utils.doFailed();
        }
        // 如果不是根节点，则需要更新父节点为“非叶子”状态
        if (!Const.S_0.equals(pid)) {
            Map<String, Object> fields = new HashMap<>();
            fields.put(keys[4], Const.S_0); // LEAF
            fields.put(keys[6], item.get(keys[6])); // OPUID
            // 把“叶子”状态，更新为“非叶子”状态
            Map<String, Object> cond1 = new HashMap<>();
            cond1.put(keys[1], item.get(keys[1])); // COMP
            cond1.put(keys[2], pid); // item.get(keys[3])); // ID\PID
            cond1.put(keys[4], Const.S_1); // LEAF
            // 更新父节点
            update(base, keys[0], fields, cond1);
        }
        return utils.success();
    }

    /**
     * 从GMETA元数据表中获取元数据的表名一览
     */
    public List<String> getMetaTable() {
        return dao.getMetaTable(tableOf(Const.GMETA));
    }

    /**
     * 判断某表名是否存在（暂未使用：暂无SQL注入风险）
     *
     * @deprecated 不支持PG，故弃用
     */
	@Deprecated
    public boolean tableExists(Object base, String table) {
        // 用法 SHOW TABLES 并无SQL注入风险
        return !dao.showTables(getTable(base, table)).isEmpty();
    }

    /*********************************************************
     * 创建表（不推荐），如果table参数从前台传入，请检查SQL注入风险
     * @deprecated
     */
	@Deprecated
    public Map<String, String> createTable(Object base, String table) {
        // 表名不符合规则
		if (!utils.isMatch(table, "\\w+")) // "[_a-zA-Z0-9]+$"
			throw new FailedException();

        // 先从META表中获取元数据
        Map<String, String> map = new HashMap<>();
        map.put(Const.TABLE, getTable(base, table));
        map.put(Const.FIELD, getMetaSql(table));
        // 动态建表
        dao.create(map);
        log.debug("createTable...{}", map);
        // 返回成功
        return utils.success();
    }

    /**
     * 根据Meta元数据整理建表Sql
     */
    private String getMetaSql(String table) {
        List<Map<String, String>> metas = getTableMeta(table);
        StringJoiner s = new StringJoiner(Const.COMMA);
        String pkey = "";

        /** 增加了租户数据隔离 Jun.22,2024 */
        if (ConfUtil.dataIsolation()) {
            pkey = tenantKey(table); // TABLE_TENANT
            s.add(pkey + " INT NOT NULL DEFAULT 0");
        }
        for (int i = 0; i < metas.size(); i++) {
            Map<String, String> meta = metas.get(i);
            // 国际化字段
            String field = meta.get(Const.FIELD);
            if (Const.S_1.equals(meta.get(Const.LANG))) {
                // 拼接国际化语言标识
                for (String locale : ConfUtil.locales()) {
                    s.add(setField(field + UL + locale, meta));
                    // 一般国际化字段为非主键
//					pkey = setPkey(pkey, field, meta);
                }
            } else {
                // 非国际化字段
                s.add(setField(field, meta));
                pkey = setPkey(pkey, field, meta);
            }
        }
        if (pkey.length() > 0)
            s.add("PRIMARY KEY (" + pkey + ")");
        return s.toString();
    }

    /**
     * 格式如：USER_NAME_ZH INTEGER NOT NULL DEFAULT '0'
     */
    private String setField(String field, Map<String, String> meta) {
        field = String.join(" ", field, meta.get(Const.TYPE));
        if (Const.S_1.equals(meta.get(Const.NULL)))
            field += " NOT NULL";
/*		String def = meta.get(DEFAULT);
		if (def.length() > 0)
			field += " DEFAULT " + def; // 默认值不可能为空
*/
        return field + " DEFAULT " + meta.get(Const.DEFAULT);
    }

    /**
     * 新建表的主键
     */
    private String setPkey(String pkey, String field, Map<String, String> meta) {
        if (Const.S_1.equals(meta.get(Const.PKEY)))
            pkey += pkey.length() > 0 ? Const.COMMA + field : field;
        return pkey;
    }

    /**
     * 从表中获取最大值+1
     */
    public int getMaxCode(Object base, String table, String field) {
        return getMaxCode(base, table, field, null);
    }

    /**
     * 从表中获取最大值+1
     */
    public int getMaxCode(String table, String field) {
        return getMaxCode(null, table, field, null);
    }

    /**
     * 从表中获取最大值+1
     */
    public int getMaxCode(Object base, String table, String field, Map<String, Object> cond) {
        return Integer.parseInt(getMaxValue(base, table, field, cond)) + 1;
    }

    /**
     * 从表中获取最大值+1
     */
    public int getMaxCode(String table, String field, Map<String, Object> cond) {
        return getMaxCode(null, table, field, cond);
    }

    /**
     * 获取当前某个表中某个字段最大值
     */
    public String getMaxValue(Object base, String table, String field) {
        return getMaxValue(base, table, field, null);
    }

    /**
     * 获取当前某个表中某个字段最大值
     */
    public String getMaxValue(String table, String field) {
        return getMaxValue(null, table, field, null);
    }

    /**
     * 获取当前某个表中某个字段最大值
     */
    public String getMaxValue(Object base, String table, String field, Map<String, Object> cond) {
        return String.valueOf(getMValue(base, table, field, cond, "MAX"));
    }

    /**
     * 获取当前某个表中某个字段最大值
     */
    public String getMaxValue(String table, String field, Map<String, Object> cond) {
        return getMaxValue(null, table, field, cond);
    }

    /**
     * 获取当前某个表中某个字段最大值
     */
    public String getMinValue(Object base, String table, String field) {
        return getMinValue(base, table, field, null);
    }

    /**
     * 获取当前某个表中某个字段最大值
     */
    public String getMinValue(String table, String field) {
        return getMinValue(null, table, field, null);
    }

    /**
     * 获取当前某个表中某个字段最小值
     */
    public String getMinValue(Object base, String table, String field, Map<String, Object> cond) {
        return String.valueOf(getMValue(base, table, field, cond, "MIN"));
    }

    /**
     * 获取当前某个表中某个字段最小值
     */
    public String getMinValue(String table, String field, Map<String, Object> cond) {
        return getMinValue(null, table, field, cond);
    }

    /**
     * 合计数据，用于数据计算
     */
    public int sumTotal(Object base, String table, String field, Map<String, Object> cond) {
        return getMValue(base, table, field, cond, "SUM");
    }

    /**
     * 合计数据，用于数据计算
     */
    public int sumTotal(String table, String field, Map<String, Object> cond) {
        return sumTotal(null, table, field, cond);
    }

    /**
     * 合计数据，用于数据计算
     */
    public int sumTotal(Object base, String table, String field) {
        return sumTotal(base, table, field, null);
    }

    /**
     * 合计数据，用于数据计算
     */
    public int sumTotal(String table, String field) {
        return sumTotal(null, table, field, null);
    }

    /**
     * 归集数据行数，用于统计和分组数据
     */
	public List<Map<String, String>> groupCount(String table, String[] keys,
			Map<String, Object> cond) {
		return groupCount(null, table, keys, cond);
	}

    /**
     * 归集数据行数，用于统计和分组数据
     */
    public List<Map<String, String>> groupCount(String table, String key, Map<String, Object> cond) {
        return groupCount(null, table, new String[]{key}, cond);
    }

}
