package cn.ps1.aolai.utils;

import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 调整输出数据的格式
 * 
 * @author Aolai
 * @since 1.8 $Date: 2024.7.20
 *
 */
public class FmtUtil {

	private static Logger LOG = LoggerFactory.getLogger(FmtUtil.class);

	/** 数值为空时 */
	private static final String NIL = "0.00";

	/** 人民币显示格式 */
	private static final String RMB = "#,##0.00";

	/**
	 * 时间（日期）格式转变
	 */
	public static String timeParse(String str, String pat0, String pat1) {
		try {
			Calendar cal = Calendar.getInstance();
			cal.setTime(new SimpleDateFormat(pat0).parse(str));
			return timeParse(cal, pat1);
		} catch (Exception e) {
			LOG.error("timeParse...{}", str);
			return str;
		}
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public static String timeParse(Calendar cal, String pattern) {
		pattern = pattern == null ? Const.DTF : pattern;
		return new SimpleDateFormat(pattern).format(cal.getTime());
	}

	/**
	 * 缩进值为指定位数的左边距
	 */
	public static String indent(String str, int num, char chr) {
		str = str == null ? "" : str;
		while ((num--) > 0) {
			str = chr + str;
		}
		return str;
	}

	/**
	 * 字符串左侧填充符号到指定长度
	 */
	public static String leftPad(String str, int size, char chr) {
		return indent(str, str == null ? size : size - str.length(), chr);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 */
	public static String commaFmt(String str, int prec) {
		return commaFmt(toNumeric(str), prec);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 */
	public static String commaFmt(String str, int prec, String def) {
		return commaFmt(toNumeric(str), prec, def);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 * 
	 * @param num 精度数据
	 * @param prec 小数点精度的位数
	 */
	public static String commaFmt(double num, int prec) {
		return commaFmt(num, prec, "-");
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 * 
	 * @param num 精度数据
	 * @param prec 小数点精度的位数
	 * @param def 默认0的返回字符串
	 */
	public static String commaFmt(double num, int prec, String def) {
		String str = parse(num, "#,##0", prec);
		return isZero(str) ? def : str;
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	public static String parse(String str, int prec) {
		return parse(toNumeric(str), prec);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	public static String parse(double num, int prec) {
		return parse(num, "#0", prec);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 * 
	 * @param num 数值
	 * @param fmt 格式
	 * @param prec 小数点精度的位数
	 */
	private static String parse(double num, String pattern, int prec) {
		pattern = prec > 0 ? pattern + Const.DOT : pattern;
		while ((prec--) > 0) {
			pattern += Const.S_0;
		}
		return parse(num, pattern);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	private static String parse(double num, String pattern) {
		return new DecimalFormat(pattern).format(num);
	}

	/**
	 * 把符号“0”、“0.0+”转为空字符串
	 */
	public static String ifZero(String str, String def) {
		return (str == null || isZero(str)) ? def : str;
	}

	/**
	 * 判断字符串是否为“0”、“0.0+”
	 */
	public static boolean isZero(Object obj) {
		return Pattern.matches("(-?0)(\\.0+)?", String.valueOf(obj));
	}

	/**
	 * 字符串转换为Double类型
	 */
	private static double toNumeric(String str) {
		try {
			return isNumeric(str) ? Double.parseDouble(str) : 0.0;
		} catch (Exception e) {
			LOG.error("toNumeric...{}", str);
			return 0.0;
		}
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public static String toYuan(String str, String def) {
		str = parse(toNumeric(str) / 100, RMB);
		return def != null && NIL.equals(str) ? def : str;
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public static String toRmb(String money) {
		return isInteger(money) ? toRmb(Long.parseLong(money)) : "";
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public static String toRmb(long num) {
		String[] digit = { "零", "壹", "贰", "叁", "肆", "伍", "陆", "柒", "捌", "玖" };
		String[][] units = { { "分", "角", "元", "万", "亿", "万" },
				{ "", "拾", "佰", "仟" } };
		String s = "", head = num < 0 ? "负" : "";
		num = Math.abs(num);
		for (int i = 0; i < 2 && num > 0; i++) {
			s = digit[(int) (num % 10)] + units[0][i] + s;
			num = num / 10;
		}
		for (int i = 2; i < units[0].length && num > 0; i++) {
			String p = "";
			for (int j = 0; j < units[1].length && num > 0; j++) {
				p = digit[(int) (num % 10)] + units[1][j] + p;
				num = num / 10;
			}
			p = p.replaceAll("(零.)*零$", "").replaceAll("^$", "零");
			s = p + units[0][i] + s;
		}
		s = s.replaceAll("(零.)*零元", "元").replaceAll("(零.)+", "零");
		return head + s.replaceAll("零$", "整");
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 */
	public static boolean isInteger(String str) {
		return str != null && Pattern.matches("-?\\d+", str);
	}

	/**
	 * 判断是否为浮点数：包含整数、小数、负数
	 */
	public static boolean isNumeric(String str) {
		return str != null && Pattern.matches("(-?\\d+)(\\.\\d+)?", str);
	}

}
