package cn.ps1.aolai.utils;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 登录页面处理图形验证码相关业务处理
 * 
 * @author Aolai
 * @since 1.8 $Date: 2023.5.1
 * @version 1.0
 */
public class Captcha {

	private static Logger LOG = LoggerFactory.getLogger(Captcha.class);

	// 默认图形模板文件类型
	private static final String PNG = "png";
	private static final String EXT = ".png";

	private static int WIDTH;
	private static int HEIGHT;

	/**
	 * 从资源图上根据模板尺寸切图
	 */
	public static List<String> imageCutout(File templateFile, File originFile,
			File blockFile) throws Exception {

		List<String> imgList = new ArrayList<>();

		// 标准模板图：{562, 342}
		BufferedImage templateImage = ImageIO.read(templateFile);
		WIDTH = templateImage.getWidth();
		HEIGHT = templateImage.getHeight();

		// 从资源图上根据模板尺寸切图
		// 随机切取的目标区域: Point{X,Y} >> imgSize={WIDTH,HEIGHT}
		BufferedImage target = cutTargetArea(originFile);

		// 抠图用的随机滑块（如：五边形、心形、星形等）
		BufferedImage block = ImageIO.read(blockFile);

		// 随机切点
		Point pt = getCutPoint(block, 3, 5);
		// 首先，参考block的尺寸生成遮罩图案
		BufferedImage newImage = getMaskImage(target, block, pt);

		// 目标区域上抠图后的图像
		target = cutoutByBlock(target, block, pt);

		// 仅测试用：saveToFile(target, "ori"); saveToFile(newImage, "cut");

		imgList.add(imgEncode(toByteArray(target)));
		imgList.add(imgEncode(toByteArray(newImage)));
		imgList.add(String.valueOf(pt.y));

		return imgList;
	}

	/**
	 * 从资源图上根据模板尺寸切图，获取目标区域
	 */
	private static BufferedImage cutTargetArea(File originFile)
			throws Exception {
		// 资源大图
		BufferedImage originImage = ImageIO.read(originFile);
		int ORI_W = originImage.getWidth();
		int ORI_H = originImage.getHeight();

		// 随机抠图的位置坐标，注意保持X轴距离右端、Y轴距离底部的偏移
		Point point = randomPoint(ORI_W - WIDTH, ORI_H - HEIGHT);
		Rectangle rec = new Rectangle(point.x, point.y, WIDTH, HEIGHT);

		// 资源图的数据流
		InputStream fis = new FileInputStream(originFile);
		ImageInputStream iis = ImageIO.createImageInputStream(fis);
		Iterator<ImageReader> imageReaderList = ImageIO
				.getImageReadersByFormatName(PNG);
		ImageReader imageReader = imageReaderList.next();
		// 输入源中的图像将只按顺序读取
		imageReader.setInput(iis, true);
		ImageReadParam param = imageReader.getDefaultReadParam();
		// 设定目标区域大小
		param.setSourceRegion(rec);
		// 返回数据
		return imageReader.read(0, param);
	}

	/**
	 * 随机生成一个切点
	 */
	private static Point getCutPoint(BufferedImage blockImage, int top, int left) {
		// 随机抠图的坐标点，注意保持X轴距离右端、Y轴距离底部的偏移
		int x = WIDTH - blockImage.getWidth() - WIDTH / left;
		int y = HEIGHT - blockImage.getHeight() - top; // 顶部预留3px距离
		// 随机坐标点
		Point point = randomPoint(x, y);
		x = point.x + WIDTH / left; // 位置略偏右
		y = point.y + top; // 顶部预留3px距离
		return new Point(x, y);
	}

	/**
	 * 随机抠图处理：在背景图上随机位置处，把原图透明化处理
	 */
	private static BufferedImage cutoutByBlock(BufferedImage origin,
			BufferedImage block, Point p) throws Exception {
		// 图像处理
		Graphics2D g2D = origin.createGraphics();
		// 图像叠加合成：0.7f为透明度，值从0-1.0，依次变得不透明
		g2D.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP,
				0.7f));
		// 在背景图上随机位置处，叠加block透明水印
		g2D.drawImage(block, p.x, p.y, block.getWidth(), block.getHeight(),
				null);
		// 在背景图上随机位置处，把原图透明化处理
		// 透明化处理：setWaterMark(origin, block, p.x, p.y, 100);

		g2D.dispose(); // 处置
		return origin;
	}

	/**
	 * 生成遮罩图案
	 */
	public static BufferedImage getMaskImage(BufferedImage origin,
			BufferedImage block, Point p) {
		// 生成最终遮罩图案
		int width = block.getWidth();
		int height = block.getHeight(); // 图片高度

		// 这里高度用了origin.getHeight()
		BufferedImage newImage = new BufferedImage(width, origin.getHeight(),
				origin.getType());

		// 重构水印的遮罩图案
		for (int i = 0; i < width; i++) { // 从左到右，每一列
			for (int j = 0; j < height; j++) { // 从上到下，每一行
				// 如果图像当前像素点有图案色(!=0)，则复制源文件信息到目标图片中
				// 黑色像素对应值"-16777216"，蓝色像素"-16755216"
				if (block.getRGB(i, j) < 0) {
					newImage.setRGB(i, p.y + j, origin.getRGB(p.x + i, p.y + j));
				}
			}
		}
		return newImage;
	}

	/**
	 * 增加透明水印（未使用）
	 */
	public static void setWaterMark(BufferedImage origin, BufferedImage block,
			int x, int y, int alpha) {
		// 添加水印图像
		for (int i = 0; i < block.getWidth(); i++) {
			// 图片高度
			for (int j = 0; j < block.getHeight(); j++) {
				// 如果图像当前像素点有图案色(!=0)，则叠加到源文件信息到目标图片中
				// 黑色像素对应值"-16777216"，蓝色像素"-16755216"
				if (block.getRGB(i, j) < 0) {
					Color color = new Color(origin.getRGB(x + i, y + j)); // 源像素
					Color newC = new Color(color.getRed(), color.getGreen(),
							color.getBlue(), alpha);
					origin.setRGB(x + i, y + j, newC.getRGB());
				}
			}
		}
	}

	/**
	 * 图片数据格式转换
	 */
	public static String imgEncode(byte[] bytes) {
		// 获取编码器
		Base64.Encoder encoder = Base64.getEncoder();
		String base64 = encoder.encodeToString(bytes);
		return "data:image/" + PNG + ";base64," + base64;
	}

	/**
	 * 图片数据格式转换：解码为字节数组
	 */
	public static byte[] imgDecode(String base64Str) {
		Base64.Decoder decoder = Base64.getDecoder();
		String[] arr = base64Str.split(",");
		base64Str = arr.length == 1 ? arr[0] : arr[1];
		return decoder.decode(base64Str); // 解码为字节数组
	}

	/**
	 * 图片数据格式转换
	 */
	public static BufferedImage toImage(String base64) {
		Base64.Decoder decoder = Base64.getMimeDecoder();
		try {
			String[] arr = base64.split(",");
			byte[] bytes = decoder.decode(arr[1]);
			InputStream bais = new ByteArrayInputStream(bytes);
			return ImageIO.read(bais);
		} catch (Exception e) {
			LOG.error("toImage...{}", e.getMessage());
			return null;
		}
	}

	/**
	 * 随机生成位置坐标点
	 */
	public static Point randomPoint(int widthDiff, int heightDiff) {
		Random random = new Random();
		int X = widthDiff <= 0 ? 0 : random.nextInt(widthDiff);
		int Y = heightDiff <= 0 ? 0 : random.nextInt(heightDiff);
		return new Point(X, Y);
	}

	/**
	 * 把图片转换为二进制数据
	 */
	private static byte[] toByteArray(BufferedImage bufImg) throws Exception {
		ByteArrayOutputStream os = new ByteArrayOutputStream(); // 新建流
		// 利用ImageIO类提供的write方法，以png图片的数据模式写入流
		ImageIO.write(bufImg, PNG, os);
		// 从流中获取数据数组
		byte[] bytes = os.toByteArray();
		// 写随机文件：String f = Digest.uuid8() + EXT;
		// 写入流：os.writeTo(new FileOutputStream(f));
		os.close();
		return bytes;
	}

	/**
	 * 轮廓边线比对，根据透明度计算像素位移
	 */
	public static int getBlockShift(String shadeStr, String blockStr, int Y) {
		return getBlockShift(toImage(shadeStr), toImage(blockStr), Y);
	}

	/**
	 * 轮廓边线比对，根据透明度计算像素位移
	 */
	public static int getBlockShift(BufferedImage shade, BufferedImage block,
			int Y) {
		List<Point> points = getBlockSide(block);
		int W = shade.getWidth() - block.getWidth();
		int L = 0, X = 0;
		for (int x = 1; x < W; x++) {
			int l = 0;
			for (Point p : points) {
				if (alphaDiff(shade, x + p.x, Y + p.y)) {
					l++; // 透明度不同
				}
			}
			if (L < l) {
				L = l;
				X = x;
			}
		}
		return X;
	}

	/**
	 * 轮廓边线比对，根据颜色值计算像素位移
	 */
	public static int getBlockShift(String shadeStr, String blockStr, int Y,
			int diff) {
		return getBlockShift(toImage(shadeStr), toImage(blockStr), Y, diff);
	}

	/**
	 * 轮廓边线比对，根据颜色值计算像素位移
	 */
	public static int getBlockShift(BufferedImage shade, BufferedImage block,
			int Y, int diff) {
		List<Point> points = getBlockSide(block);
		int pixel = shade.getWidth() - block.getWidth();
		int L = 0, X = 0;
		for (int x = 1; x < pixel; x++) {
			int l = 0;
			for (Point p : points) {
				if (colorDiff(shade, x + p.x, Y + p.y, diff)) {
					l++; // 找到相似度最低的边界
				}
			}
			if (L < l) {
				L = l;
				X = x;
			}
		}
		return X;
	}

	/**
	 * 获取block边缘的像素点
	 */
	private static List<Point> getBlockSide(BufferedImage block) {
		List<Point> list = new ArrayList<>();
		for (int y = 0; y < block.getHeight(); y++) {
			for (int x = 0; x < block.getWidth(); x++) {
				if (block.getRGB(x, y) < 0) {
					list.add(new Point(x, y));
					break;
				}
			}
		}
		return list;
	}

	/**
	 * 获颜色的差值
	 */
	private static boolean colorDiff(BufferedImage shade, int x, int y, int d) {
		try {
			Color c0 = new Color(shade.getRGB(x, y));
			Color c1 = new Color(shade.getRGB(x - 1, y - 1));
			Color c2 = new Color(shade.getRGB(x + 1, y + 1));
			if (Math.abs(c1.getRed() - c0.getRed()) > d
					|| Math.abs(c1.getGreen() - c0.getGreen()) > d
					|| Math.abs(c1.getBlue() - c0.getBlue()) > d
					|| Math.abs(c2.getRed() - c0.getRed()) > d
					|| Math.abs(c2.getGreen() - c0.getGreen()) > d
					|| Math.abs(c2.getBlue() - c0.getBlue()) > d)
				return true;
		} catch (Exception e) {
			LOG.error("colorDiff...{}", e.getMessage());
		}
		return false;
	}

	/**
	 * 获取透明度的差值
	 */
	private static boolean alphaDiff(BufferedImage shade, int x, int y) {
		int diff = 0;
		try {
			Color c0 = new Color(shade.getRGB(x, y), true);
			Color c1 = new Color(shade.getRGB(x - 1, y - 1), true);
			Color c2 = new Color(shade.getRGB(x + 1, y + 1), true);
			diff = Math.abs(c1.getAlpha() - c0.getAlpha());
			diff += Math.abs(c2.getAlpha() - c0.getAlpha());
		} catch (Exception e) {
			LOG.error("alphaDiff...{}", e.getMessage());
		}
		return diff > 0;
	}

	/** * * 以下为测试内容 * * */

	public static void main(String[] args) {
		try {
			/**
			 * 可以用以下测试文件：bag.png，origin.png，block.png<br>
			 * 模板：File tempFile = new File("D:\\javaApp\\doyea\\bag.png");<br>
			 * 滑块：File blockFile = new File("D:\\javaApp\\doyea\\block.png");<br>
			 * 源：File originFile = new File("D:\\javaApp\\doyea\\origin.png");<br>
			 * 分割：imageCutout(templateFile, originFile, blockFile);
			 */

			File f0 = new File("D:\\javaApp\\doyea\\block1.png");
			File f1 = new File("D:\\javaApp\\doyea\\shade2.png");
			BufferedImage block = ImageIO.read(f0);
			BufferedImage shade = ImageIO.read(f1);
			/**
			 * 挖孔：cutoutByBlock(shade, block, new Point(80, 47));<br>
			 * 点：Point pt = new Point(80, 47);<br>
			 * 新图：BufferedImage newImg = getMaskImage(shade, block, pt);<br>
			 * 保存：saveToFile(newImg, "newImg");
			 */
			getBlockShift(shade, block, 47);

			LOG.debug("sucess!");
		} catch (Exception e) {
			LOG.error("test...{}", e.getMessage());
		}
	}

	/**
	 * 仅测试用
	 */
	public static void saveToFile(BufferedImage bufImg, String name) {
		ByteArrayOutputStream os = new ByteArrayOutputStream(); // 新建流
		try {
			// 利用ImageIO类提供的write方法，以png图片的数据模式写入流
			ImageIO.write(bufImg, PNG, os);
			os.writeTo(new FileOutputStream(randFileName(name)));
			os.close();
		} catch (Exception e) {
			LOG.error("saveToFile...{}", e.getMessage());
		}
	}

	/**
	 * 仅测试用
	 */
	public static void saveToFile(byte[] bytes, String name) throws Exception {
		File file = new File(randFileName(name));
		FileOutputStream fos = new FileOutputStream(file);
		BufferedOutputStream bos = new BufferedOutputStream(fos);
		bos.write(bytes);
		fos.close();
	}

	/**
	 * 生成一个随机文件名
	 */
	private static String randFileName(String name) {
		return name + System.currentTimeMillis() + EXT;
	}

}
