package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.dao.AolaiDao;
import cn.ps1.aolai.entity.*;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;

/**
 * Aolai基础框架相关服务层，基于Mybatis整合的元数据的基础操作类(CRUD)
 * 
 * @author Aolai
 * @version 1.0 $Date: 2019.7.6
 * @since 1.7
 */

@Service
public abstract class GmetaService {

	private static Logger LOG = LoggerFactory.getLogger(GmetaService.class);
	// 元数据表
	static final String META = "GMETA";
	// 排序字段
	static final String META_SORT = "META_SORT";
	/** 租户ID */
	static final String TENANT = "TENANT";

	/** 数据库中空字符 */
	static final String NUL = "''";

	/** 数字 */
	static final Pattern P_DIGIT = Pattern.compile("\\d+"); // 数字
	/** 匹配普通的 {fieldKey} 关键字字段 */
	static final Pattern P_QUOTE = Pattern.compile("\\{ *\\w+ *\\}"); // 引用
	/** 固定数值占位符{V} */
	static final Pattern P_PLACE = Pattern.compile("\\{ *V* *\\}");
	// 正则匹配JSON查询条件，只支持2层："{userDuty}->>'$.appCode.way' ="
	static final Pattern P_JSON = Pattern.compile("^,'\\$(\\.\\w+)+'\\)$");

	// 这里base为映射中的主键名
//	static final String TYPE = "type";
	static final String PKEY = "pkey";
	static final String DUPL = "dupl";

	static final String M_PKEY = "!p"; // mark_Pkey
	static final String M_I18N = "!i"; // mark_I18n
	static final String M_TYPE = "!t"; // mark_Type,VARCHAR(32)
	static final String NULL = "null";
	static final String LEFT = "left";
	static final String DEFT = "default";
	static final String WHERE = "where";

	static final String _S = "_";
	static final String _W = "\\w+";
	static final String CAST = "CAST(";
	static final String R_PAREN = "')";

	@Autowired
	AolaiDao dao;
	@Autowired
	RedisService redis;
	@Autowired
	UtilsService utils;
	@Autowired
	HttpServletRequest req; // 单实例可以注入
	@Autowired
	private ApplicationContext appCtx;

	/** 元数据的元数据 */
	static Map<String, String> gmeta = new HashMap<>();
	static {
		gmeta.put(Const.FIELD, "META_FIELD");
		gmeta.put(Const.ALIAS, "META_ALIAS");
		gmeta.put(Const.NAME, "META_NAME");
		gmeta.put(Const.TYPE, "META_TYPE");
		gmeta.put(NULL, "META_NULL");
		gmeta.put(Const.I18N, "META_I18N");
		gmeta.put(PKEY, "META_PKEY");
		gmeta.put(Const.WIDTH, "META_WIDTH");
		gmeta.put(DEFT, "META_DEFAULT");
		gmeta.put(Const.SORT, META_SORT);
	}

	/**
	 * 当前数据源是否为PostgreSQL
	 */
	boolean isPostgres() {
		// 注意区分数据库驱动，这里与AolaiDao.xml中的databaseId命名保持一致
		// 在加载配置文件，注册SqlSessionFactoryBean时获取配置属性
		return appCtx.containsBean("dsPostgres");
	}

	/**
	 * 设置多租户数据分离查询条件<br>
	 * {"USER_NAME_ZH like":"张%", "TABLE_TENANT =": "889"}
	 */
	void setDataIsolation(Map<String, Object> cond, String table) {
		/** 增加了租户数据隔离 Jun.22,2024 */
		if (ConfUtil.dataIsolation())
			cond.put(tenantKey(table) + Const.EQU, req.getAttribute(ConfUtil.dbid()));// base);
	}

	/**
	 * 租户ID
	 */
	String tenantKey(String table) {
		return table + _S + TENANT;
	}

	/*********************************************************
	 * 往“table”表中增加一条记录
	 */

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 * 
	 * @param table 表名
	 * @param dto 字段映射
	 * @param data 数据
	 * @param dupl 更新标记
	 * @return result 返回状态
	 */
	Map<String, String> addRecord(Object base, String table,
			Map<String, String> dto, Map<String, Object> data, boolean dupl) {
		// 也可通过data携带base参数
		base = base == null ? data.get(Const.BASE) : base;

		// 获取键值对映射：{USER_NAME_ZH : "张三"}
		data = setFieldData(dto, data, true);
		/** 增加了租户数据隔离 Jun.22,2024 */
		// {USER_NAME_ZH : "张三", TABLE_TENANT : "889"}
//		setDataIsolation(data, base, table);
		if (ConfUtil.dataIsolation())
			data.put(tenantKey(table), req.getAttribute(ConfUtil.dbid()));//base);

		Map<String, String> map = duplCheck(base, table, dto, dupl);
		// 返回执行结果
		LOG.debug("addRecord...{}", data);
		return utils.result(dao.addOne(map, data) > 0);
	}

	/**
	 * 删除重复数据，为了兼容PostgreSQL，识别出table的主键
	 */
	private Map<String, String> duplCheck(Object base, String table,
			Map<String, String> dto, boolean dupl) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// TODO：有些数据库不支持ON DUPLICATE KEY UPDATE语法
		// 思考：如果已有数据存在，则先删除已有数据？？
		if (dupl) {
			// 为了兼容PostgreSQL，识别出table的主键
			map.put(DUPL, getDuplPks(table, dto));
		}
		return map;
	}

	/**
	 * 识别出单表table的主键pkey，以“逗号”分隔
	 */
	private String getDuplPks(String table, Map<String, String> dto) {
		if (!isPostgres())
			return "";
		/** 增加了租户数据隔离 Jun.22,2024 */
		return getMetaPks(dto, ConfUtil.dataIsolation() ? tenantKey(table) : "");
	}

	/**
	 * 识别出单表table的主键pkey，以“逗号”分隔
	 */
	private String getMetaPks(Map<String, String> dto, String pk) {
		StringJoiner jt = new StringJoiner(Const.COMMA);
		if (pk.length() > 0)
			jt.add(pk);
		for (Map.Entry<String, String> e : dto.entrySet()) {
			// 获取元数据中table的约定的主键名（pkey）
			if (e.getKey().contains(M_PKEY) && Const.S_1.equals(e.getValue())) {
				// 根据截取的userId 获取 USER_ID
				jt.add(dto.get(e.getKey().split("!")[0]));
			}
		}
		return jt.toString();
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(base, table, getDto(table), data, dupl);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(null, table, data, dupl);
	}

	/**
	 * 往基础表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table, Map<String, Object> data) {
		return addRecord(null, table, data, false);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table,
			Map<String, Object> data) {
		return addRecord(base, table, data, false);
	}

	/*********************************************************
	 * 批量往业务表中插入多条数据
	 */
	private Map<String, String> batchAdd(Object base, String table,
			Map<String, String> dto, List<Map<String, Object>> list,
			Object lang, boolean dupl) {
		if (list.size() == 0)
			throw new FailedException();

		// 取第一行数据作为参数
		Map<String, Object> data = list.get(0);
		// 也可通过data携带base参数
		base = base == null ? data.get(Const.BASE) : base;

		data.put(Const.I18N, lang);
		// 设置别名映射格式：{USER_NAME_ZH : "userName"}
		data = setFieldData(dto, data, false);
		/** 增加了租户数据隔离 Jun.22,2024 */
/*		if (ConfUtil.dataIsolation()) {
			// 这里"TENANT"为映射别名
			data.put(tenantKey(table), TENANT);
			for (Map<String, Object> item : items)
				item.put(TENANT, base); // { "TENANT" : "889" }
		}
*/
		// 这里补充参数"TENANT"为映射的别名：{ "XXX_TENANT": TENANT }
		if (ConfUtil.dataIsolation())
			data.put(tenantKey(table), TENANT);
		for (Map<String, Object> item : list) {
			for (Map.Entry<String, Object> e : item.entrySet()) {
				validationRules(dto, e);
			}
			// { "TENANT" : "889" }
			if (ConfUtil.dataIsolation())
				item.put(TENANT, req.getAttribute(ConfUtil.dbid()));//base);
		}

		// 数据库执行参数
		Map<String, String> map = duplCheck(base, table, dto, dupl);
		// 返回执行结果
		LOG.debug("batchAdd...{}", map);
		return utils.result(dao.batchAdd(map, data, list) > 0);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(Object base, String table,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		return batchAdd(base, table, getDto(table), items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Map<String, Object> map,
			boolean dupl) {
		return batchAdd(map.get(Const.BASE), table, items, i18n(map), dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Map<String, Object> map) {
		return batchAdd(table, items, map, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		return batchAdd(null, table, items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Object lang) {
		return batchAdd(null, table, items, lang, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items) {
		return batchAdd(table, items, new HashMap<>());
	}

	/**
	 * 批量往业务表中插入多条数据：注意参数items、dupl（可选）
	 */
	public Map<String, String> batchAdd(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数：items、dupl
		List<Map<String, Object>> list = utils.json2List(map.get("items"));
		boolean dupl = map.containsKey(DUPL); // 重复则更新
		return batchAdd(getBase(map), getTable(map), list, i18n(map), dupl);
	}

	/*********************************************************
	 * 删除表中的数据
	 */
	Map<String, String> delete(Object base, String table,
			Map<String, String> dto, Map<String, Object> cond, Object joint) {
		// 梳理删除条件
		cond = setCondition(dto, cond, joint);
		// 删除条件不能为空（全部删除）
		if (cond.isEmpty())
			throw new FailedException();

		// 也可通过data携带base参数
		base = base == null ? cond.get(Const.BASE) : base;
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));

		/** 增加了租户数据隔离 Jun.22,2024 */
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		return utils.result(dao.delete(map, cond) > 0);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table,
			Map<String, Object> cond, Object joint) {
		return delete(base, table, getDto(table), cond, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> cond,
			Object joint) {
		return delete(null, table, cond, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> cond) {
		return delete(null, table, cond, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table,
			Map<String, Object> cond) {
		return delete(base, table, cond, null);
	}

	/**
	 * 删除表中的数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> delete(Map<String, String> params) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> cond = utils.json2Map(params.get(WHERE));
		return delete(getBase(params), getTable(params), cond);
	}

	/*********************************************************
	 * 批量删除数据
	 */

	/**
	 * 批量删除数据，其中list中的数据为逐条删除数据的条件
	 */
	Map<String, String> batchDelete(Object base, String table,
			Map<String, String> dto, List<Map<String, Object>> list,
			String joint) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));

		// 遍历数据并逐条删除
		int count = 0;
		for (int i = 0; i < list.size(); i++) {
			// list中的数据为逐条删除数据的条件：{"USER_NAME_ZH like":"张%"}
			Map<String, Object> cond = setCondition(dto, list.get(i), joint);
			if (!cond.isEmpty()) {
				/** 增加了租户数据隔离 Jun.22,2024 */
				setDataIsolation(cond, table); // {"TABLE_TENANT =": "889"}
				count += dao.delete(map, cond); // 计数
			}
		}
		// if (count == list.size()) // 数据一致性
		return utils.result(count > 0);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(Object base, String table,
			List<Map<String, Object>> list) {
		return batchDelete(base, table, getDto(table), list, null);
	}

	/**
	 * 批量删除表中的数据，其中cond中的数据为删除数据的条件
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> batchDelete(Object base, String table,
			Object cond) {
		String whr = String.valueOf(cond);
		return batchDelete(base, table, utils.json2List(whr));
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(String table,
			List<Map<String, Object>> list) {
		return batchDelete(null, table, list);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(Object base, String table,
			List<Map<String, Object>> list, String joint) {
		return batchDelete(base, table, getDto(table), list, joint);
	}

	/**
	 * 批量删除表中的数据，其中cond中的数据为删除数据的条件
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> batchDelete(String table, Object cond) {
		return batchDelete(null, table, cond);
	}

	/**
	 * 批量删除表中的数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> batchDelete(Map<String, Object> params) {
		if (params == null)
			return utils.result(false);
		String table = String.valueOf(params.get(Const.TABLE));
		return batchDelete(params.get(Const.BASE), table, params.get(WHERE));
	}

	/*********************************************************
	 * 更新表中的数据
	 */
	private Map<String, String> update(Object base, String table,
			Map<String, String> dto, Map<String, ?> fields,
			Map<String, Object> cond, Object joint) {
		cond = setCondition(dto, cond, joint);
		fields = setCondition(dto, fields, null);
		if (cond.isEmpty() || fields.isEmpty())
			throw new FailedException();
		// 也可通过data携带base参数
		base = base == null ? cond.get(Const.BASE) : base;

		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));

		/** 增加了租户数据隔离 Jun.22,2024 */
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		// 返回执行结果
		return utils.result(dao.update(map, fields, cond) > 0);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table,
			Map<String, ?> fields, Map<String, Object> cond, Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// DTO = { uid:'USER_ID', uid!t:'INTEGER', uid!i:'1', uid!p:'1' }
		return update(base, table, getDto(table), fields, cond, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table,
			Map<String, ?> fields, Map<String, Object> cond) {
		return update(base, table, fields, cond, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, ?> fields,
			Map<String, Object> cond, Object joint) {
		return update(null, table, fields, cond, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, ?> fields,
			Map<String, Object> cond) {
		return update(null, table, fields, cond, null);
	}

	/**
	 * 更新数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> update(Map<String, String> map, Object lang) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> fields = utils.json2Map(map.get("fields"));
		Map<String, Object> cond = utils.json2Map(map.get(WHERE));
		fields.put(Const.I18N, lang);
		cond.put(Const.I18N, lang);
		return update(getBase(map), getTable(map), fields, cond);
	}

	/*********************************************************
	 * 支持多张表 join关联查询：left="left"为左连接
	 */
	public <T> List<Map<String, String>> findList(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> cond, Map<String, String> sort, String limit) {
		return findList(base, tables, uriKey, cond, sort, LEFT, limit);
	}

	/**
	 * 支持多张表 join关联查询：left=""为内连接
	 * 
	 * @param tables 两表名为key，两表的关联主键的映射，表1设为null
	 * @param uriKey 前台上下文或接口名，仅缓存数据用
	 * @param cond 综合多表的查询条件
	 * @param limit 限制每页输出行数，为“0”时不限行数
	 */
	public <T> List<Map<String, T>> findList(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> cond, Map<String, String> sort, String left,
			String limit) {
		// 也也可通过cond携带base参数
		base = base == null ? cond.get(Const.BASE) : base;

		// 这里的dto为查询多个表相关联的字段
		Map<String, String> dto = getMetaDto(base, uriKey);
		String[] table = joinTables(base, dto, tables, left);
		if (table == null)
			return new ArrayList<>();

		// 数据库查询参数：table,alias,sort,joint,limit,floor
		Map<String, String> args = new HashMap<>();
		args.put(Const.TABLE, table[1]);
		args.put(Const.ALIAS, uriKey);
		args.put(Const.LIMIT, limit);
		// 处理args.get("alias")和查询条件
		cond = setArgsAndCond(args, dto, cond, sort);

		/** 增加了租户数据隔离 Jun.22,2024 */
		setDataIsolation(cond, table[0]); // { "TABLE_TENANT =": "889" }
		
		// 返回执行结果
		return dao.findList(args, cond);
	}

	/**
	 * 联合多表查询，这里的dto为查询多个表相关联的字段
	 */
	String[] joinTables(Object base, Map<String, String> dto,
			Map<String, Map<String, String>> tables, String left) {
		if (tables == null)
			return null;
		String dbid = utils.sqlVal(req.getAttribute(ConfUtil.dbid()));
		String table = null, T0 = null, T = null;
		StringJoiner jt = new StringJoiner(" ", " ", "");
		// 以表名为主键，其中第一个表为空，第二个表与第一个表关联，第三个表与第二个表关联
		// { Table01:<null>, Table02<Table02.colA=Table01.colB> }
//		int t = 1; // 必须从1开始
		for (String key : tables.keySet()) {
			// 根据表名table获取左连接的字段映射join关系
			Map<String, String> cond = tables.get(key);
			// 表的全名，如：base=“public.base899_”\“base899.”\“899”
			table = getTable(base, key);// + " AS ";
			if (utils.isEmpty(cond)) {
				// 第一个主表的关联条件必为空，后面不能为空
				T0 = table;// + "t0"; // 主表
				T = key;
			} else {
//				T = "t" + (t++);
				// 左连接关联表
				if (!utils.isEmpty(left))
					jt.add(left);
				// 两个表关联的关系条件：t0 left join table AS t2 on t2.TENANT='899'
//				jt.add("join").add(table + T).add("on");
				jt.add("join").add(table).add("on");
				/** 增加了租户数据隔离 Jun.22,2024 */
				if (ConfUtil.dataIsolation()) {
					// t2.TENANT='899'
//					jt.add(T + Const.DOT + TENANT + Const.EQU + utils.sqlVal(base));
					jt.add(tenantKey(key) + Const.EQU + dbid);//utils.sqlVal(base));
				}
				int n = 0;
				for (Map.Entry<String, String> e : cond.entrySet()) {
					String val = e.getValue();
					if (val == null)
						continue;
					if (n++ > 0 || ConfUtil.dataIsolation())
						jt.add("and");

					// 两个表关联的对应主键，如果右侧的元数据主键不存在，直接使用右侧的值
					// 多个表关联特殊情况时，可以返回一个纯粹带引号的“字符串”参数
					// 使用单引号以支持PGSQL，其中保证key不能有SQL注入非法字符
					if (dto.containsKey(val)) {
						val = i18n(dto, val, null);
					} else {
						val = utils.isMatch(val, _W) ? utils.sqlVal(val) : NUL;
					}
					// 用等号（=）拼接两表的关联主键：and USER_COMP=COMP_ID
					jt.add(i18n(dto, e.getKey(), null) + Const.EQU + val);
				}
			}
		}
//		return T == null ? null : T0 + jt.toString();
		return T == null ? null : new String[] { T, T0 + jt.toString() };
	}

	/**
	 * 限定了最大返回记录数（limit.rows）
	 */
	private void setLimitRows(Map<String, String> args,
			Map<String, Object> cond) {
		String[] keys = {"pageNo","pageSize"};
		if (utils.availParams(cond, keys)) {
			// 页面号码、及每页展示的行数
			int pNo = utils.getInt(cond, keys[0]);
			int pSize = utils.getInt(cond, keys[1]);
/*
			String pageNo = String.valueOf(cond.get("pageNo"));
			String pageSize = String.valueOf(cond.get("pageSize"));
			if (utils.isInteger(pageNo) && utils.isInteger(pageSize)) {
				long pNo = Long.parseLong(pageNo);
				long pSize = Long.parseLong(pageSize);
*/
				args.put("floor", String.valueOf((pNo - 1) * pSize));
				args.put(Const.LIMIT, String.valueOf(pSize));
				return;
//			}
		}
		String limit = args.get(Const.LIMIT);
		// 当limit==null或''，不限制数据行数
		if (!utils.isInteger(limit)) {
			// 如果limit.rows=''，默认不限制数据，有可能仅limit.rows=''
			args.put(Const.LIMIT, ConfUtil.limitRows());
		} else if ("0".equals(limit)) {
			// 如果limit==0，删除参数，设置为空不限行数，其他时保留
			args.remove(Const.LIMIT);
		}
	}

	/**
	 * 为了适配处理返回分页信息，把查询条件集中处理：args, dto, cond, sort
	 */
	Map<String, Object> setArgsAndCond(Map<String, String> args,
			Map<String, String> dto, Map<String, Object> cond,
			Map<String, String> sort) {

		// 数据库查询参数：table,alias,sort,joint,limit,floor
		Object lang = i18n(cond);

		// 传回前端的数据映射的主键参数数组
		args.put(Const.ALIAS, setAlias(dto, args.get(Const.ALIAS), lang)); // 无SQL注入风险
		// 根据“数据转换对象DTO”，整理排序字段
		args.put(Const.ORDER, orderBy(dto, sort, lang));
		// 限定了最大返回记录数，设置参数：limit,floor
		setLimitRows(args, cond);

		// 拼接查询条件
		return setCondition(dto, cond, args.get(Const.JOINT));
	}

	/*********************************************************
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 * <p>
	 * 数据库查询参数：base,table,alias,sort,joint,limit,floor
	 */
	public <T> List<Map<String, T>> findList(Map<String, String> args,
			Map<String, Object> cond, Map<String, String> sort) {
		String table = getTable(args);
		Map<String, String> dto = getDto(table);
		// 租户隔离
		Object base = getBase(args);
		// 也也可通过cond携带base参数
		base = base == null ? cond.get(Const.BASE) : base;

		// 替换表名
		args.put(Const.TABLE, getTable(base, table));

		// 整合处理args.get("alias")和查询条件
		cond = setArgsAndCond(args, dto, cond, sort);

		/** 增加了租户数据隔离 Jun.22,2024 */
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		// 返回查询结果
		return dao.findList(args, cond);
	}

	/**
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public <T> List<Map<String, T>> findList(Map<String, String> args,
			Map<String, Object> cond) {
		return findList(args, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findList(Object base, String table,
			String uriKey, Map<String, Object> cond, Map<String, String> sort) {
		// uriKey为传回前端的参数数组
		String db = base == null ? null : String.valueOf(base);
		return findAll(cond, sort, new String[] { db, table, uriKey });
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findAll(Object base, String table,
			String uriKey, Map<String, Object> cond, Map<String, String> sort) {
		// uriKey为传回前端的参数数组
		String db = base == null ? null : String.valueOf(base);
		String[] args = { db, table, uriKey, Const.S_0 };
		return findAll(cond, sort, args);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findList(Object base, String table,
			String uriKey, Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findList(base, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findAll(Object base, String table,
			String uriKey, Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findAll(base, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findList(String table, String uriKey,
			Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findList(null, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findAll(String table, String uriKey,
			Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findAll(null, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findList(String table,
			Map<String, Object> cond) {
		return findList(table, null, cond);
	}

	/**
	 * 查询数据一览
	 */
	public <T> List<Map<String, T>> findAll(String table,
			Map<String, Object> cond) {
		return findAll(table, null, cond);
	}

	/**
	 * 转换参数：把参数数组转换为映射
	 */
	Map<String, String> convertArgs(String[] params) {
		// 无效参数
		if (params == null || params.length < 2)
			return null;
		// 有效参数传递
		String[] keys = { Const.BASE, Const.TABLE, Const.ALIAS, Const.LIMIT };
		Map<String, String> args = new HashMap<>();
		for (int i = 0; i < params.length && i < keys.length; i++) {
			args.put(keys[i], params[i]);
		}
		return args;
	}

	/**
	 * 基于条件的查询结果数据集，用以支持limit="0"的情况<br>
	 * 查询条件args：base,table,alias,limit
	 */
	public <T> List<Map<String, T>> findAll(Map<String, Object> cond,
			Map<String, String> sort, String[] params) {
		Map<String, String> args = convertArgs(params);
		return args == null ? new ArrayList<>() : findList(args, cond, sort);
	}

	/*********************************************************
	 * 查出表中的一条记录，限定了只返回一条记录，也可通过cond携带base参数
	 */
	private Map<String, String> findOne(Object base, String table,
			Map<String, String> dto, Map<String, Object> cond, String uriKey) {
		// 也可通过cond携带base参数
		base = base == null ? cond.get(Const.BASE) : base;
		// 从数据库查询所需的参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 这里的dto是基于一张表的元数据映射对象
		map.put(Const.ALIAS, setAlias(dto, uriKey, i18n(cond))); // 无SQL注入风险

		/** 增加了租户数据隔离 Jun.22,2024 */
		cond = setCondition(dto, cond, null);
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		// 限定了只返回一条记录
		map = dao.findOne(map, cond);
		// 返回执行结果
		return utils.isEmpty(map) ? utils.result(false) : map;
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table,
			Map<String, Object> cond, String uriKey) {
		return findOne(base, table, getDto(table), cond, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table,
			Map<String, Object> cond) {
		return findOne(base, table, cond, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 * 
	 * @param table 表名
	 * @param cond 查询条件
	 * @param uriKey 返回值的别名
	 */
	public Map<String, String> findOne(String table, Map<String, Object> cond,
			String uriKey) {
		return findOne(null, table, cond, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> cond) {
		return findOne(null, table, cond, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> findOne(Map<String, String> map, String uriKey) {
		Map<String, Object> cond = utils.json2Map(map.get(WHERE));
		return findOne(getBase(map), getTable(map), cond, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Map<String, String> map) {
		return findOne(map, null);
	}

	/*********************************************************
	 * 判断表中是否有数据，也可通过cond携带base参数
	 */
	private boolean exists(Object base, String table, Map<String, String> dto,
			Map<String, Object> cond, Object joint) {
		// 也可通过cond携带base参数
		base = base == null ? cond.get(Const.BASE) : base;
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));

		/** 增加了租户数据隔离 Jun.22,2024 */
		cond = setCondition(dto, cond, joint);
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		// 返回执行结果
		return dao.countRows(map, cond) > 0;
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> cond,
			Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// DTO = { uid:'USER_ID', uid!t:'INTEGER', uid!i:'1', uid!p:'1' }
		return exists(base, table, getDto(table), cond, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> cond) {
		return exists(base, table, cond, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> cond, Object joint) {
		return exists(null, table, cond, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> cond) {
		return exists(null, table, cond, null);
	}

	/**
	 * 判断表中是否有数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public boolean exists(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> cond = utils.json2Map(map.get(WHERE));
		return exists(getBase(map), getTable(map), cond);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(String table, Map<String, Object> cond) {
		return count(null, table, cond, null);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(Object base, String table, Map<String, Object> cond) {
		return count(base, table, cond, null);
	}

	/**
	 * 根据条件获取记录的行数，也可通过cond携带base参数
	 */
	public int count(Object base, String table, Map<String, Object> cond,
			String joint) {
		// 也可通过cond携带base参数
		base = base == null ? cond.get(Const.BASE) : base;
		Map<String, String> args = new HashMap<>();
		args.put(Const.TABLE, getTable(base, table));

		// 注意这里的条件转换：{"USER_NAME_ZH like":"张%"}
		cond = setCondition(getDto(table), cond, joint);
		/** 增加了租户数据隔离 Jun.22,2024 */
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		return dao.countRows(args, cond);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 根据用户的应用、公司、岗位（dutyId）获取页面上的功能事件控制权限
	 */
	private Map<String, String> cacheActionRole(Map<String, Object> map) {

		// 获取各关联表的相关的字段
		map.put(Const.ALIAS, setAlias(Action.TABLE, null)); // 无SQL注入风险
		// 查询条件：应用、公司、岗位、与页面路由相关
		List<Map<String, String>> list = dao.getActionRole(map);

		// 只要actUri有一次权限，其他页面actRoute都有权访问
		// 或则actUri有一无许可，则其他页面都无许可
		// 这里 actState='1'
		Map<String, String> actMap = utils.list2Map(list, Action.URI,
				Action.STATE);
		// 必须包含的数据信息
		if (!actMap.isEmpty()) {
			// 根据第一行判断参数是否有效
			// 注意这里key需要与third.trustedRole()中一致
			Object[] keys = { map.get(ConfUtil.APPCODE),
					map.get(ConfUtil.COMPID), map.get(Duty.ID) };
			String key = utils.join(keys, Const.DOT);
			// 缓存信息
			redis.setActionRole(key, actMap);
			LOG.debug("cacheActionRole...{}", actMap);
		}
		return actMap;
	}

	/**
	 * 根据应用（appCode）公司（compId）岗位（dutyId）获取页面菜单的控制权限（ROLE_PAGE、ROLE_ACTION）
	 * 
	 * @param params 参数中包含了appCode、compId、dutyId、i18n信息
	 */
	public List<Map<String, String>> getMenuRole(Map<String, Object> params) {
		// 关联应用(appCode)公司（compId）岗位(dutyId)信息
		if (utils.isEmpty(params))
			return new ArrayList<>();

		// 缓存鉴权信息：此处缓存所有页面上的功能事件控制权限
		cacheActionRole(params);

		// 获取各关联表的相关的字段
		params.put(Const.ALIAS, getAlias("getMenuRole", i18n(params)));

		// 查询条件：应用、公司、岗位
		// pageId;pageName;pageRoute;pageCmpt;pageType;pageGroup;pageAction;pageIcon
		// ;roleId;(rolePage;)roleAction
		List<Map<String, String>> list = dao.getMenuRole(params);

		// 根据第一行判断参数是否有效，是否包含必须的数据信息
		mergeMenuRole(list);

		return list;
	}

	/**
	 * 梳理用户岗位：依赖 params中的 appCode，userComp、userDuty
	 */
	public Map<String, Object> getRoleParams(Map<String, Object> params,
			Map<String, String> user) {
		Map<String, Object> map = utils.newCond(params);
		Object appCode = params.get(ConfUtil.APPCODE);
		if (appCode != null) {
			// 从当前登录用户获取信息: "compId"、"dutyId"
			map.put(ConfUtil.COMPID, user.get(User.COMP));
//			map.put(Const.I18N, i18n(params));

			// 在登录的时候，已经处理过userDuty，并把具体某个账套中的岗位信息写入userDuty中。
			// 所以，在此用户岗位信息必须从userDuty中获取，如果无岗位信息则框架鉴权就过不来
			Map<String, String> uDuty = utils.json2Map(user.get(User.DUTY));
			// 正常应该有权限
			if (uDuty.containsKey(appCode)) {
				map.put(Duty.ID, uDuty.get(appCode));
				map.put(ConfUtil.APPCODE, appCode);
			}
		}
		return map;
	}

	/**
	 * 合并多角色的菜单权限，应用、公司、岗位
	 */
	private Map<String, String> mergeMenuRole(List<Map<String, String>> list) {
		String[] keys = new String[] { Page.ID, Page.ROUTE, Role.ACTION };
		if (list.size() == 1)
			return utils.list2Map(list, keys[1], keys[2]);
		// 遍历数据，合并多角色role的多个action控制权限
		Map<String, String> menu = new HashMap<>();
		for (int i = list.size() - 1; i > 0; i--) {
			// 当前行，如：“P01”、当前行的前一行，如：“P00”
			Map<String, String> tmp0 = list.get(i);
			Map<String, String> tmp1 = list.get(i - 1);
			// 当前行，如：“P01”、当前行的前一行，如：“P00”
			String page0 = tmp0.get(keys[0]); // "pageId"
			String page1 = tmp1.get(keys[0]); // "pageId"

			// 取得当前行roleAction的值
			int ra = Integer.valueOf(tmp0.get(keys[2])); // "roleAction"
			if (page1.equals(page0)) {
				ra = Integer.valueOf(tmp1.get(keys[2])) | ra; // "roleAction"
				// 合并权限后，给前一行（重复行）赋新值
				tmp1.put(keys[2], String.valueOf(ra)); // "roleAction"
				// 删除当前（重复的）行
				list.remove(i);
			} else {
				// 相邻的两行pageId不同时，处理当前有效行（仅仅当前行的权限或是合并后的权限）
				// menu.put(page0, String.valueOf(ra));
				menu.put(tmp0.get(keys[1]), tmp0.get(keys[2])); // "roleAction"
			}
			// 单独处理最后一行（有效行）
			if (i == 1) {
				// 这里使用了"pageRoute"，如总账页面：'/generalLedger'
				menu.put(tmp1.get(keys[1]), tmp1.get(keys[2])); // "roleAction"
			}
		}
		return menu;
	}

	/**
	 * 设置或修改配置参数：{baseId;paramKey;paramVal}
	 */
	public Map<String, String> setParam(Map<String, Object> params) {
		Map<String, Object> cond = utils.newCond(params);
//		cond.put(Const.I18N, params.get(Const.I18N));
		cond.put(Param.BB, params.get(Const.KMBB));
		cond.put(Param.KEY, params.get(Param.KEY));
		return update(params.get(Const.BASE), Param.TABLE, params, cond);
	}

	/**
	 * 获取系统配置参数
	 * 
	 * @param base 帐套名
	 * @param kmbb 科目版本
	 * @param key 参数主键
	 */
	private String getParam(Object base, Object kmbb, Object key) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Param.KEY, key);
		cond.put(Param.BB, kmbb == null ? "00" : kmbb);
		// 获取系统配置参数：uriKey="getParamVal"
		return findOne(base, Param.TABLE, cond, "").get(Param.VAL);
	}

	/**
	 * 获取系统配置参数
	 */
	public String getParam(Map<String, ?> cond, Object key) {
		return getParam(getBase(cond), cond.get(Const.KMBB), key);
	}

	/**
	 * 获取全部系统配置参数
	 */
	public List<Map<String, String>> getParams(Map<String, Object> cond) {
//		// 其中“备注”支持多语言
//		Map<String, Object> cond = new HashMap<>();
//		cond.put(Const.I18N, params.get(Const.I18N));
//		cond.put(Param.BB, params.get(Const.KMBB));
		return findList(getBase(cond), Param.TABLE, "", cond);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 增删改查，字段自动支持国际化
	 */
	private String i18n(Map<String, String> dto, String key, Object lang) {
		String field = dto.get(key); // 注意这里field != null
		// 根据约定（i18n）判断是否支持国际化
		if (dto.containsKey(key + M_I18N))// && !utils.isEmpty(lang))
//			field += _S + lang; // 获取clientLang，添加“_”符号
			return utils.isEmpty(lang) ? NUL : field + _S + lang;
		return field;
	}

	/**
	 * 当前查询语言
	 */
	private <T> Object i18n(Map<String, T> params) {
		return params == null ? null : params.get(Const.I18N);
	}

	/**
	 * 查询排序条件：根据“数据转换对象DTO”，拼接排序条件
	 */
	private String orderBy(Map<String, String> dto, Map<String, String> sort,
			Object lang) {

		// 如果排序条件为空，默认按主键pkey（!p）排序
		if (utils.isEmpty(sort))
			return getMetaPks(dto, "");

		// 按传递的排序参数（sort）排序
		StringJoiner jt = new StringJoiner(Const.COMMA);
		for (Map.Entry<String, String> e : sort.entrySet()) {
			// 支持国际化，如：USER_NAME_ZH ASC
			if (dto.containsKey(e.getKey())) {
				String key = i18n(dto, e.getKey(), lang);
				String val = e.getValue();
				jt.add(utils.isEmpty(val) ? key : key + " " + val);
			}
		}
		return jt.toString();
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	private <T> Map<String, T> setCondition(Map<String, String> dto,
			Map<String, T> map, Object joint) {
		Map<String, T> cond = new LinkedHashMap<>();
		if (map == null)
			return cond;
		Object lang = i18n(map);
		for (Map.Entry<String, T> e : map.entrySet()) {
			T val = e.getValue();
			if (val == null) // 注意：若传递参数有空值，则跳过处理
				continue;
			String key = e.getKey();
			// 这里的key只能是单纯的“fieldKey”
			if (dto.containsKey(key)) {
				// 这里增加数据校验
				validationRules(dto, e);
				key = i18n(dto, key, lang);
				key += joint == null ? Const.EQU : " " + joint;
				// 如更新值: FIELD_KEY="889"
				cond.put(key, val);
			} else if (P_QUOTE.matcher(key).find()) {
				/**
				 * 条件匹配参数如：“{fieldKey} <>” 之类参数<br>
				 * 转换后的格式如：{“USER_NAME_ZH <>”, “张三”}<br>
				 * 这里需要注意防SQL注入，如：{userId}='1';select * from t1;<br>
				 * 这里val必须是数组参数，支持同时传递多个 “数值”={"aaa","bbb"}
				 */
				// 携带变量参数{V}时的特殊处理
				boolean noneVar = P_PLACE.matcher(key).find();
				// 带 “{fieldKey}”的一般也携带joint=“”参数
				// 用“数值”={"aaa","bbb"}置换key里的多个{V}
				key = getFieldKey(dto, key, val, lang);
				if (key != null)
					cond.put(key, noneVar ? null : val);
			}
		}
		LOG.debug("setCondition...{}", utils.obj2Str(cond));
		return cond;
	}

	/**
	 * 拼接SQL语句的查询条件
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public String getWhere(Map<String, String> dto, String expr, Object lang) {
		return getFieldKey(dto, expr, new String[] {}, lang);
	}

	/**
	 * 把字符串中以“{}”标识出来的关键词替换成“表字段”名和数值{V}
	 * <p>
	 * 替换条件“{key}>={V}”中的key和V（可能有多个），校验表达式避免SQL注入风险
	 * <p>
	 * 注意预防SQL注入，如：{userId}='1';select * from t1;
	 * 
	 * @param <T>
	 */
	private <T> String getFieldKey(Map<String, String> dto, String expr,
			T valArr, Object lang) {
		// 例如：expr=“( {key} IS NULL OR {key} in ( {V} )”
		// 如：{ userLock }='0' and { userId } in ( '111','222' ) and ...
		String[] arr = expr.split("\\{ *"); // 分割“{key} in (”
		int v = 0;
		boolean isArr = valArr.getClass().isArray();
		String[] vArr = isArr ? (String[]) valArr : new String[] {};
		// arr.length>1时，一般arr[0]=""为有效表达式，如：arr[]=["","userId} in ("]
		// 一般arr[0]=""，应从 arr[1]开始处理
		if (arr.length > 1 && isValidExpr(arr[0])) {
			// 从 arr[1]开始处理，如：arr[1] = "userId} in ("
			for (int i = 1; i < arr.length; i++) {
				// 从arr[1]再分析，如：key[] = ["userId"," in ("]
				String[] key = arr[i].split(" *\\}");
				// 字段名{key[0]}必须存在、赋值无SQL注入风险，则继续处理。如：“ in (”为有效
				if (key.length == 0) { // arr[1]=="}"时
					if (vArr.length > v)
						arr[i] = vArr[v++];

				} else if (dto.containsKey(key[0])) {
					// 替换：userId =>>USER_ID
					arr[i] = i18n(dto, key[0], lang);
					if (key.length > 1 && isValidExpr(key[1]))
						arr[i] += key[1];

				} else if (vArr.length > v) {
					// 约定{V}用""或大写字母V标识
					if ("".equals(key[0]) || "V".equals(key[0]))
						arr[i] = vArr[v++] + (key.length > 1 ? key[1] : "");
				}
			}
			return utils.join(arr);
		}
		return null;
	}

	/**
	 * 检查SQL注入，不在规定范围内的表达式无效
	 * 
	 * @param expr 已剔除空格的表达式，关键字请用小写字母
	 */
	private boolean isValidExpr(String expr) {
		if (expr.length() > 0) {
			// 从表达式拆分识别出关键字，如：“ in (”
			String[] arr = expr.split(" +"); // 以空格“”分割
			for (String str : arr) {
				// 必须能匹配规定范围内的表达式，或满足JSON类型操作的表达式格式："->>'$.inspur.eyun'"
				if (str.length() == 0
						|| utils.findIn(ConfUtil.SQL_EXPR, str.toLowerCase())
						|| P_JSON.matcher(str).find())
					continue;
				// 若有未匹配的表达式，则返回无效
				LOG.error("invalid expr...{}", str);
				throw new FailedException();
			}
		}
		return true;
	}

	/**
	 * 添加数据时：字段名对应数值的键值对，根据isValue判断是键值对、还是别名映射
	 */
	private Map<String, Object> setFieldData(Map<String, String> dto,
			Map<String, Object> params, boolean isValue) {
		// 无效参数：添加数据时，数据不能为空
		if (params != null) {
			Map<String, Object> data = new HashMap<>();
			Object lang = i18n(params);
			for (Map.Entry<String, Object> e : params.entrySet()) {
				if (!dto.containsKey(e.getKey())) // || e.getValue() == null
					continue;
				// 注意此刻的key!=e.getKey()
				String key = i18n(dto, e.getKey(), lang);
				if (isValue && validationRules(dto, e)) // type不可能为null
					data.put(key, e.getValue()); // {"USER_NAME_ZH":"张三"}
				else
					data.put(key, e.getKey()); // {"USER_NAME_ZH":"userName"}
			}
			if (!data.isEmpty())
				return data;
		}
		// 无效参数：添加数据时，数据不能为空
		throw new FailedException();
	}

	/**
	 * 输入字符边界校验规则
	 */
	private boolean validationRules(Map<String, String> dto, Map.Entry<String, ?> e) {
		if (ConfUtil.avoidRules())
			return true;
		// 未避开数据库校验规则，则执行如下校验规则
		String type = dto.get(e.getKey() + M_TYPE);
		if (type == null)
			return false;
		String str = String.valueOf(e.getValue());
		// VARCHAR(32)、CHAR(1)、TIMESTAMP、SMALLINT等
		Matcher matcher = P_DIGIT.matcher(type);
		if (matcher.find()) {
			// VARCHAR(32)\CHAR(1)
			if (Integer.parseInt(matcher.group()) >= str.length())
				return true;
		} else if (type.contains("IN")) {
			// INTEGER\BIGINT\SMALLINT\TINYINT
			if (utils.isInteger(str))
				return true;
		} else if (type.contains("J")) {
			// JSON
			return true;
		} else if (type.contains("T")) {
			// DATE+\TIME+timestamp
			if (utils.isDatetime(str, Const.DTF))
				return true;
		} else if (type.contains("L")) {
			// DOUBLE\DECIMAL\REAL
			if (utils.isNumeric(str))
				return true;
		} else {
			return true;
		}
		// 数据校验参数无效
		String key = String.join("_", ConfUtil.INVD_PARAMS, e.getKey());
		throw new FailedException(key);
	}

	/**
	 * 根据table表名获取元数据字段的别名
	 */
	public String setAlias(String table, Object lang) {
		return setAlias(table, null, lang);
	}

	/**
	 * 根据table表名获取元数据字段的别名
	 */
	public String setAlias(String table, String uriKey, Object lang) {
		return setAlias(getDto(table), uriKey, lang);
	}

	/**
	 * 根据转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String uriKey, Object lang) {
		// 增加空置判断，为空则返回全部dto字段（2022.5.2）
		// 如果支持自定义设置标签的隐藏和显示的数量，则返回全部dto字段（2023.8.27）
		// 自定义设置标签IS_CUSTOM_COLS：其中fieldShow="1"的数据，在返给前台后再处理
		if (ConfUtil.isCustomCols() || utils.isEmpty(uriKey))
			return fieldAsAlias(dto, lang);

		// 首先从配置文件读取参数，未定义时默认返回空（""）
		String val = ConfUtil.getParam(uriKey);
		// 如果返回“”，则返回全部表字段的别名
		if (val.length() == 0)
			return fieldAsAlias(dto, lang);
		// 返回字段与别名的映射：USER_NAME as userName, USER_MAIL as userMail
		return fieldAsAlias(dto, val.split(ConfUtil.COMMA), lang);
	}

	/**
	 * 根据转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String fieldAsAlias(Map<String, String> dto, String[] arr,
			Object lang) {
		StringJoiner jt = new StringJoiner(Const.COMMA);
		for (String key : arr) {
			jt.add(fieldAsAlias(dto, key, lang));
		}
		// 返回字段与别名的映射
		return jt.toString();
	}

	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String fieldAsAlias(Map<String, String> dto, Object lang) {
		StringJoiner jt = new StringJoiner(Const.COMMA);
		// 为约定区分type\i18n\pkey，标注为!t,!i,!p
		for (String key : dto.keySet()) {
			// 忽略标注了“!”的主键
			if (!key.contains("!"))
				jt.add(fieldAsAlias(dto, key, lang));
		}
		return jt.toString();
	}

	/**
	 * 拼接别名（无SQL注入风险），注意支持的数据类型，注意日期格式勿用TIME
	 */
	private String fieldAsAlias(Map<String, String> dto, String key, Object lang) {
		String field = dto.get(key);
		// 约定为区分数据类型type、国际化语言i18n，分别标注为!t,!i
		String type = dto.get(key + M_TYPE);
		String keyStr = utils.quote(key, "\""); // "\"" + key + "\"";
		if (type != null) {
			// 根据 META_FIELD的不同数据类型进行处理
			// 增加了支持JSON格式，"IN"为了TINYINT类型
//			if (type.matches("[IDBSJ].+") || type.matches("TIN.+")) {
			if (type.matches("[IJSBD].+") || type.contains("IN")) {
				// INTEGER\DOUBLE\DECIMAL\DATE+\BIGINT\SMALLINT\TINYINT\JSON
				// 补充了适配 PG(PostgreSQL)
				// mysql中的ifnull()函数对应postgresql的COALESCE
				if (isPostgres()) {
					field = CAST + field + " as TEXT)"; // 暂不处理NULL
				} else {
					field = CAST + field + " as CHAR)"; // 暂不处理NULL
				}
			} else if (type.matches("T.+")) { // TIMESTAMP(3)\勿用TIME
				// 追加PG(PostgreSQL) 适配
				if (isPostgres()) {
					field = "to_char(" + field + ",'yyyy-MM-dd hh24:mi:ss"
							+ (type.length() > 9 ? ".ms" : "") + R_PAREN;
				} else {
					field = "DATE_FORMAT(" + field + ",'%Y-%m-%d %T"
							+ (type.length() > 9 ? ".%f" : "") + R_PAREN;
				}
			} else if (dto.containsKey(key + M_I18N)) {// && !utils.isEmpty(lang)) {
				// 只有字符串（VARCHAR\CHAR）格式，才有可能支持国际化
//				field += _S + lang;
				field = utils.isEmpty(lang) ? NUL : field + _S + lang;
			}
		} else if (field == null) {
			// 当携带的fieldKey不存在时，直接使用参数本身key
			// field不存在时，使用单引号空字符串，PG不支持双引号
			field = NUL; //keyStr;
		}
		// 这里为了兼容PgSQL使用双引号
		//return val + " as " + keyStr;
		// 返回字段与别名的映射，如：USER_NAME_ZH as userName
		return String.join(" ", field, Const.AS, keyStr);
	}

	/**
	 * 数据库与前台交互传递的参数，根据请求key获取字段及别名映射串
	 */
	public String getAlias(String uriKey, Object lang) {
		return getAlias(null, uriKey, lang);
	}

	/**
	 * 数据库与前台交互传递的参数，根据请求key获取字段及别名映射串
	 */
	private String getAlias(Object base, String uriKey, Object lang) {
		// 这里把元数据转换为字段的映射
		// 如：“USER_ID as "userId", USER_LANG as "userLang"”
		return fieldAsAlias(getMetaDto(base, uriKey), lang); // 无SQL注入风险
	}

	/**
	 * 数据库与前台交互传递的参数，与getAlias的区别是参数为table
	 */
	public String getField(String table, Object lang) {
		return fieldAsAlias(getDto(table), lang);
	}

	/**
	 * 从数据库中查询并返回别名与字段的映射：{ userName: USER_NAME_ZH }
	 */
	private Map<String, String> getFields(String table, String[] keyArr,
			Object lang) {
		Map<String, String> map = new HashMap<>();
		if (keyArr != null) {
			Map<String, String> dto = getDto(table);
			// 例如key=userName，转为为{ userName: USER_NAME_ZH }
			for (String key : keyArr) {
				if (dto.containsKey(key))
					map.put(key, i18n(dto, key, lang));
			}
		}
		return map;
	}

	/**
	 * 数据库与前台交互传递的参数
	 */
//	public String getFieldAlias(List<Map<String, String>> metaList, Object lang) {
//		return fieldAsAlias(getDto(metaList), lang);
//	}

	/**
	 * 获取元数据对象：“别名”与“字段”的映射关系、类型、国际化
	 * <p>
	 * 如：{ uid:'USER_ID', uid!p:'1', uid!i:'1', uid!t:'VARCHAR(18)' }
	 */
	Map<String, String> getDto(String table) {
		String rk = Const.RDS_META + table;
		// 注意：缓存dto时有序排列，再次获取的Map为无序排列
/*		Map<String, String> dto = redis.hmget(rk);
		if (dto == null) {
			dto = getDto(getTableMeta(table));
			// 缓存数据请求对象（缓存10小时缩为2小时）
			redis.hmset(rk, dto, Const.TWO_HH);
		}
		return dto;*/

		// 这里为了保持数据有序
		String dtoStr = redis.get(rk);
		if (dtoStr == null) {
			// 缓存的dto数据为有序排列
			dtoStr = utils.obj2Str(getDto(getTableMeta(table)));
			// 缓存数据请求对象（缓存10小时缩为2小时）
			redis.set(rk, dtoStr, Const.TWO_HH);
		}
		return utils.json2Map(dtoStr);
	}

	/**
	 * 获取元数据对象：“别名”与“字段”的映射关系、类型、国际化
	 * <p>
	 * 注意：getDto按表名查询，getMetaDto按主键查询，且有可能跨表查询
	 */
	Map<String, String> getMetaDto(Object base, String uriKey) {
		boolean isCustom = ConfUtil.isCustomCols();
		// 按每个base和uriKey缓存数据
		String rk = Const.RDS_META + (isCustom ? base : "") + uriKey;
		Map<String, String> dto = redis.hmget(rk);
		if (dto == null) {
			// 如果支持自定义设置标签的隐藏和显示的数量
			if (isCustom) {
				// 先从FIELD表中获取fieldList，再从meta表中查询数据
				dto = getDto(getMeta(getFieldList(base, uriKey)));
			} else {
				dto = getDto(getMeta(uriKey));
			}
			// 缓存数据请求对象（缓存10小时）
			redis.hmset(rk, dto, Const.TEN_HH);
		}
		return dto;
	}

	/**
	 * 获取FIELD表中的元数据字段信息，根据fieldUri查询fieldAlias所有列名<br>
	 */
	public List<Map<String, String>> getFieldList(Object base, String uriKey) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);
		// 按序号排列
		Map<String, String> sort = new HashMap<>();
		sort.put(Field.SORT, "ASC");
		// 查询全部元数据的字段，具体展示格式放在前端处理
		return findList(base, Field.TABLE, null, cond, sort);
	}

	/**
	 * 元数据对象：约定数据类型type(!t)\主键pkey(!p)\国际化i18n(!i)
	 */
	private Map<String, String> getDto(List<Map<String, String>> metaList) {
		Map<String, String> dto = new LinkedHashMap<>();
		for (int i = 0; i < metaList.size(); i++) {
			Map<String, String> meta = metaList.get(i);
			// field:UDER_ID,pkey:'1',i18n:'1',alias:'userId'
			String alias = meta.get(Const.ALIAS); // userId
			dto.put(alias, meta.get(Const.FIELD)); // userId:UDER_ID
			dto.put(alias + M_TYPE, meta.get(Const.TYPE));
			dto.put(alias + M_PKEY, meta.get(PKEY));
			if (Const.S_1.equals(meta.get(Const.I18N)))
				dto.put(alias + M_I18N, meta.get(Const.I18N));
		}
		if (dto.isEmpty())
			throw new FailedException();
		return dto;
	}

	/**
	 * 根据 META_ALIAS从GMETA表中获取Meta元数据一览
	 */
	public List<Map<String, String>> getMeta(String uriKey) {
		String val = ConfUtil.getParam(uriKey);
		// 查询条件为多个字段的别名：[ 'userLang', 'userMail', 'userName', ...]
		String[] keys = val.length() == 0 ? null : val.split(ConfUtil.COMMA);
		return getMeta(keys);
	}

	/**
	 * 前面先从FIELD表中获取fieldList，再从meta表中查询数据<br>
	 * 获取元数据字段信息，根据uriKey查询FIELD、GMETA关联数据
	 */
	private List<Map<String, String>> getMeta(
			List<Map<String, String>> fieldList) {
		String[] keys = new String[fieldList.size()];
		// 转换格式：[ userId, userLang, userName, userMail, ...]
		for (int i = 0; i < fieldList.size(); i++)
			keys[i] = fieldList.get(i).get(Field.ALIAS);
		return getMeta(keys);
	}

	/**
	 * 根据“别名”获取字段一览
	 */
	private List<Map<String, String>> getMeta(String[] keys) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(null, META));
		map.put(Const.ALIAS, fieldAsAlias(gmeta, null)); // 无SQL注入风险
		// 查询结果：[{field:UDER_ID,pkey:'1',i18n:'1',alias:'userId',...}, ...]
		return dao.getMeta(map, keys);
	}

	/**
	 * 为兼容旧版单独加的处理，后期会弃用（用getTableMeta替代）
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public List<Map<String, String>> getMetaList(String table) {
		return getTableMeta(table);
	}

	/**
	 * 根据表名table从GMETA表中获取Meta元数据全部信息一览
	 * <p>
	 * 这里表名称table无SQL注入风险
	 */
	public List<Map<String, String>> getTableMeta(Object table) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(null, META));
		map.put(Const.ALIAS, fieldAsAlias(gmeta, null)); // 无SQL注入风险
		map.put(Const.ORDER, META_SORT);
		Map<String, Object> cond = new HashMap<>();
		cond.put("META_TABLE=", table);
		return dao.findList(map, cond);
	}

	/**
	 * 账套（base）及表名组合拼接符号移到UtilsService的jsonParams处理
	 */
	public String getTable(Object base, String table) {
		// 如果base为空，补充PG数据库时的信息
		if (isPostgres()) {
			/** 增加了租户数据隔离 Jun.22,2024 */
			if (base != null && !ConfUtil.dataIsolation()) {
				// base == base123.\base123_
				table = base + table;
				// 返回 base123.USER
				String db = String.valueOf(base);
				if (db.charAt(db.length() - 1) == '.')
					return table;
				// 继续处理base123_USER的情况
			}
			// table == USER\base123_USER
			String[] keys = { ConfUtil.baseMain(), table };
			// public.USER 或 public.base123_USER
			return String.join(Const.DOT, keys);
//		} else if (ConfUtil.dataIsolation()) {
			// 增加了仅数据隔离的情况
//			return table;
		} else {
			// 此处可兼容数据隔离的情况，base=""
			// 这里 base携带了拼接符号：base123. 或 base123_
			return base == null ? table : base + table;
		}
	}

	private String getTable(Map<String, String> params) {
		return params.get(Const.TABLE);
	}

	private <T> Object getBase(Map<String, T> params) {
		return params.get(Const.BASE);
	}

	/**
	 * 从表中获取最大值+1
	 */
	public int getMaxCode(Object base, String table, String field) {
		return getMaxCode(base, table, field, null);
	}

	/**
	 * 从表中获取最大值+1
	 */
	public int getMaxCode(Object base, String table, String field,
			Map<String, Object> cond) {
		String mv = getMaxValue(base, table, field, cond);
		return Integer.parseInt(mv) + 1;
	}

	/**
	 * 获取当前某个表中某个字段最大值
	 */
	public String getMaxValue(Object base, String table, String field) {
		return getMaxValue(base, table, field, null);
	}

	/**
	 * 获取当前某个表中某个字段最大值
	 */
	public String getMaxValue(Object base, String table, String field,
			Map<String, Object> cond) {
		return getMValue(base, table, field, cond, "MAX(");
	}

	/**
	 * 获取当前某个表中某个字段最大值
	 */
	public String getMinValue(Object base, String table, String field) {
		return getMinValue(base, table, field, null);
	}

	/**
	 * 获取当前某个表中某个字段最小值
	 */
	public String getMinValue(Object base, String table, String field,
			Map<String, Object> cond) {
		return getMValue(base, table, field, cond, "MIN(");
	}

	/**
	 * 合计数据，用于数据计算
	 */
	public String sumTotal(Object base, String table, String field,
			Map<String, Object> cond) {
		return getMValue(base, table, field, cond, "SUM(");
	}

	/**
	 * 合计数据，用于数据计算
	 */
	public String sumTotal(Object base, String table, String field) {
		return getMinValue(base, table, field, null);
	}

	/**
	 * 获取当前某个表中某个字段最大值或最小值
	 */
	private String getMValue(Object base, String table, String field,
			Map<String, Object> cond, String max) {
		Map<String, String> dto = getDto(table);
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// max(CAST(${map.field} AS integer)+1)
		map.put(Const.FIELD, "CAST(" + max + dto.get(field) + ") AS CHAR)");

		/** 增加了租户数据隔离 Jun.22,2024 */
		cond = setCondition(dto, cond, null);
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		return dao.getMValue(map, cond);
	}

	/**
	 * 归集数据行数，用于统计和分组数据
	 */
	public List<Map<String, String>> groupCount(Object base, String table,
			String[] keys, Map<String, Object> cond) {
		Map<String, String> dto = getDto(table);
		// 返回别名与字段的映射：{ userName: USER_NAME_ZH }
		Map<String, String> map = getFields(table, keys, null);
		/** 增加了租户数据隔离 Jun.22,2024 */
		cond = setCondition(dto, cond, null);
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }
		// 返回数据归集结果
		return dao.groupCount(getTable(base, table), map, cond);
	}

}
