package cn.ps1.aolai.service;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.http.HttpEntity;
import org.apache.http.HttpMessage;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

/**
 * 通过远程 http服务调用的业务操作
 *
 * @author Aolai
 * @version 1.0
 * @since 1.7 $Date: 2019.12.21
 */

@Service
public class HttpService {

	private static Logger LOG = LoggerFactory.getLogger(HttpService.class);

	private static Map<String, String> headers = new HashMap<>();

	static {
		headers.put("Accept", "*/*");
		headers.put("Accept-Encoding", "gzip, deflate, br");
		headers.put("Accept-Language", "zh-CN, zh, en-US");
		headers.put("Cache-Control", "no-cache");
		headers.put("Connection", "keep-alive");
		headers.put("User-Agent", "Chrome/97.0.4692.71");
	}
	// 禁止了自动重定向
	private static RequestConfig CONF = RequestConfig.custom()
			.setRedirectsEnabled(false).setConnectTimeout(Const.TIMEOUT)
			.setSocketTimeout(Const.TIMEOUT).build();
	private static String APP_KEY = null;

	@Autowired
	private UtilsService utils;

	/**
	 * Http请求
	 */
	public String doRequest(HttpUriRequest httpUri) {
		CloseableHttpClient http = HttpClientBuilder.create().build();
		CloseableHttpResponse resp = null;
		String result = null;
		try {
			// 发送请求
			resp = http.execute(httpUri);
			// 响应结果
			result = getResponse(resp);
		} catch (Exception e) {
			LOG.error("-> doRequest...");
			e.printStackTrace();
		} finally {
			utils.close(http, resp);
		}
		return result;
	}

	/**
	 * Http 处理Get请求
	 */
	public String httpGet(String url, Map<String, String> params) {
		// 拼接Get参数
		HttpGet httpGet = new HttpGet(setUrl(url, params));
		httpGet.setConfig(CONF);
		setHeaders(httpGet, url);
		return doRequest(httpGet);
	}

	/**
	 * Http 处理Post请求
	 */
	public String httpPost(String url, Map<String, String> params) {
		HttpEntity entity = null;
		try {
			entity = new UrlEncodedFormEntity(setParams(params), Const.UTF8);
		} catch (Exception e) {
			LOG.error("-> doPost..." + e.getMessage());
			e.printStackTrace();
		}
		HttpPost httpPost = new HttpPost(url);
		httpPost.setConfig(CONF);
		httpPost.setEntity(entity); // 添加参数
		// 设置信息头
		setHeaders(httpPost, url);
		
		return doRequest(httpPost);
	}

	/**
	 * Http 携带Cookies转发的Post请求
	 */
	public String httpPost(String url, String cookieHeader,
			Map<String, String> params) {
		// String cookieHeader = req.getHeader("Cookie");
		if (cookieHeader != null)
			headers.put("Cookie", cookieHeader);
		return httpPost(url, params);
	}

	/**
	 * 拼接请求参数
	 *
	 * @param url
	 * @param params
	 */
	private String setUrl(String url, Map<String, String> params) {
		if (params != null && params.size() > 0) {
			String args = "";
			for (Map.Entry<String, String> e : params.entrySet()) {
				args += "&" + e.getKey() + "=" + Digest.urlEncode(e.getValue());
			}
			if (args.length() > 0)
				url += args.substring(1);
		}
		return url;
	}

	private void setHeaders(HttpMessage msg, String url) {
		msg.addHeader("Referer", utils.getConf("app.core"));
		msg.addHeader("Host", utils.getHost(url)[0]); // 域名及端口，或IP地址及端口号
		for (Map.Entry<String, String> e : headers.entrySet()) {
			msg.addHeader(e.getKey(), e.getValue());
		}
	}

	/**
	 * 创建参数队列
	 */
	private List<NameValuePair> setParams(Map<String, String> params)
			throws UnsupportedEncodingException {
		List<NameValuePair> nvpList = new ArrayList<NameValuePair>();
		if (params != null) {
			for (Map.Entry<String, String> e : params.entrySet()) {
				String key = e.getKey();
				String val = e.getValue();
				if (!utils.isEmpty(val))
					nvpList.add(new BasicNameValuePair(key, val));
			}
		}
		return nvpList;
	}

	/**
	 * 实现HTTP摘要认证
	 */
	private CloseableHttpClient digestHttpClient(String user, String pass) {
		// String[] arr = utils.getHost(url);
		// int port = arr.length > 1 ? Integer.parseInt(arr[1])
		// : AuthScope.ANY_PORT;

		// 摘要认证参数
		UsernamePasswordCredentials creds = new UsernamePasswordCredentials(
				user, pass);
		CredentialsProvider credsProvider = new BasicCredentialsProvider();
		// credsProvider.setCredentials(new AuthScope(arr[0], port), creds);
		credsProvider.setCredentials(AuthScope.ANY, creds);
		HttpClientBuilder http = HttpClientBuilder.create();
		http.setDefaultCredentialsProvider(credsProvider);

		return http.build();
	}

	/**
	 * 设置请求参数
	 */
	private RequestBuilder setBuilder(HttpEntity entity, String method) {
		RequestBuilder request = null;
		if ("POST".equals(method)) {
			request = RequestBuilder.post();
		} else if ("GET".equals(method)) {
			request = RequestBuilder.get();
		} else if ("PUT".equals(method)) {
			request = RequestBuilder.put();
		} else if ("DELETE".equals(method)) {
			request = RequestBuilder.delete();
		}
		// 添加参数
		return request.setEntity(entity);
	}

	/**
	 * HTTP响应
	 */
	private String getResponse(HttpResponse rsp) throws Exception {
		String result = null;
		HttpEntity entity = rsp.getEntity();
		int statusCode = rsp.getStatusLine().getStatusCode();
		if (statusCode == 200 && entity != null) {
			result = EntityUtils.toString(entity, Const.UTF8);
			int len = result.length();
			LOG.debug(len > 200 ? result.substring(0, 200) : result);
		} else {
			LOG.error("-> getResponse..." + rsp.getStatusLine());
		}
		
		return result;
	}

	/**
	 * 基于摘要认证的HTTP请求
	 *
	 * @param userName
	 * @param password
	 */
	public String digestRequest(String userName, String password,
			HttpUriRequest uriRequest) {
		CloseableHttpClient http = digestHttpClient(userName, password);
		CloseableHttpResponse rsp = null;
		String result = null;
		try {
			// 请求、响应
			rsp = http.execute(uriRequest);
			result = getResponse(rsp);

		} catch (Exception e) {
			LOG.error("-> digestRequest...");
			e.printStackTrace();
		} finally {
			utils.close(http, rsp);
		}
		return result;
	}

	/**
	 * 基于摘要认证的HTTP请求
	 * 
	 * @param userName
	 * @param password
	 * @param url
	 * @param method
	 */
	public String digestJsonRequest(String userName, String password,
			String url, String json, String method) {
		LOG.info(json);
		StringEntity entity = new StringEntity(json, Const.UTF8);
		entity.setContentType("application/json");
		entity.setContentEncoding(Const.UTF8);
		RequestBuilder builder = setBuilder(entity, method);
		HttpUriRequest request = builder.setUri(url).build();
		json = digestRequest(userName, password, request);
		LOG.info(json);
		return json;
	}

	public boolean digestDownload(String userName, String password, String url,
			String filePath) {
		CloseableHttpClient http = digestHttpClient(userName, password);
		CloseableHttpResponse rsp = null;
		boolean result = false;
		try {
			// 请求、响应
			HttpGet request = new HttpGet(url);
			rsp = http.execute(request);
			HttpEntity entity = rsp.getEntity();
			InputStream ins = entity.getContent();
			FileOutputStream fos = new FileOutputStream(filePath, false);
			int data = ins.read();
			while (data != -1) {
				fos.write(data);
				data = ins.read();
			}
			fos.close();
			EntityUtils.consume(entity);
			return true;
		} catch (Exception e) {
			LOG.error("-> digestDownload...");
			e.printStackTrace();
		} finally {
			utils.close(http, rsp);
		}
		return result;
	}

	/**
	 * 读取输入流，返回字符串
	 *
	 * @param in
	 */
	private String respAsString(InputStream in) {
		StringBuilder strBuf = new StringBuilder();
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new InputStreamReader(in, Const.UTF8));
			String line;
			while ((line = reader.readLine()) != null) {
				strBuf.append(line);
			}
		} catch (IOException e) {
			LOG.error("-> respAsString..." + e.getMessage());
			e.printStackTrace();
		} finally {
			utils.close(reader);
		}
		return strBuf.toString();
	}

	private HttpURLConnection setProperty(HttpURLConnection conn) {
		conn.setConnectTimeout(Const.TIMEOUT); // 设置相应请求时间：120秒
		conn.setReadTimeout(Const.TIMEOUT); // 设置读取超时时间：120秒
		conn.setUseCaches(false); // 默认是true
		conn.setDoOutput(true); // 默认是false
		conn.setDoInput(true); // 默认是true
		for (Map.Entry<String, String> e : headers.entrySet()) {
			conn.setRequestProperty(e.getKey(), e.getValue());
		}
		return conn;
	}

	/**
	 * 通过HTTP方法请求数据
	 */
	public String urlConnect(String url, byte[] bytes, String method) {
		String result = "";
		HttpURLConnection conn = null;
		try {
			LOG.debug("-> urlConnect..." + url);
			URL httpUrl = new URL(url);
			conn = (HttpURLConnection) httpUrl.openConnection();
			setProperty(conn).setRequestMethod(method);

			conn.getOutputStream().write(bytes); // 输出流

			// PrintWriter out = new PrintWriter(conn.getOutputStream());
			// out.print(s); // 发送请求参数
			// out.flush(); // 输出流的缓冲

			// 发送请求
			if (conn.getResponseCode() == 200 || conn.getResponseCode() == 204) {
				result = respAsString(conn.getInputStream());
			} else {
				result = respAsString(conn.getErrorStream());
				LOG.error("-> urlConnect..." + result);
			}
		} catch (Exception e) {
			LOG.error("-> urlConnect...");
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * 向指定 URL发送GET请求
	 */
	public String urlConnect(String url, Map<String, String> params) {
		return urlConnect(setUrl(url, params), "".getBytes(), "GET");
	}

	/**
	 * 向指定 URL发送POST请求
	 */
	public String urlConnect(String url, String str, String method) {
		String result = "";
		if (str != null)
			result = urlConnect(url, str.getBytes(Const.CHARSET), method);
		return result;
	}

	/** 以下通过远程调用业务逻辑，获取信息 */

	/**
	 * 构造函数（为了测试用）
	 */
	public HttpService() {
		utils = new UtilsService();
	}

	/**
	 * 远程post调用，传递json参数的请求，增加了校验，不加校验直可接用httpPost
	 * 
	 * @param url
	 *            服务地址
	 * @param jsonStr
	 *            参数
	 * @return String
	 */
	public String doPost(String url, String jsonStr, String key) {
		LOG.info("-> doPost..." + url);
		if (key == null)
			key = (APP_KEY == null) ? utils.getConf("app.key") : APP_KEY;

		Map<String, String> map = new HashMap<>();
		map.put("k", key); // 如：www.ps1.cn\ZxCrOplK
		map.put(Const.JSON_STR, jsonStr);
		
		return httpPost(url, map);
	}

	/**
	 * 远程post调用，传递json参数的HttpPost请求
	 * 
	 * @param url
	 *            服务地址
	 * @param jsonStr
	 *            参数
	 * @return String
	 */
	public String doPost(String url, String jsonStr) {
		return doPost(url, jsonStr, null);
	}

	/**
	 * 远程post调用，传递json参数的请求，增加了校验，不加校验直可接用httpPost
	 * 
	 * @param url
	 *            服务地址
	 * @param params
	 *            参数
	 * @return Map jsonStr
	 */
	public <T> Map<String, T> doPost(String url, Map<String, T> params) {
		String jsonStr = doPost(url, utils.obj2Str(params));
		return utils.json2Map(jsonStr);
	}

	/**
	 * 单点SSO请求
	 * 
	 * @param ws
	 *            服务
	 * @param jsonStr
	 *            字符串
	 * @return String
	 */
	public String ssoPost(String ws, String jsonStr) {
		// 单点登录的远程服务地址：http://sso.ps1.cn/aolai/s
		final String ssoUrl = utils.getConf("sso.url");
		return doPost(ssoUrl + ws, jsonStr);
	}

	/**
	 * 单点SSO请求
	 * 
	 * @param ws
	 *            服务
	 * @param params
	 *            参数
	 * @return String
	 */
	public <T> Map<String, T> ssoPost(String ws, Map<String, T> params) {
		// 单点登录的远程服务地址：http://sso.ps1.cn/aolai/s
		final String ssoUrl = utils.getConf("sso.url");
		return doPost(ssoUrl + ws, params);
	}

	/**
	 * 转发对SSO服务的请求，如：远程调用用户验证
	 */
	public Map<String, Object> forward(HttpServletRequest req) {
		String ws = utils.getRequestURI(req); // 如：uri="wsTest1"
		return ssoPost(ws, utils.jsonParams(req));
	}

	/**
	 * 单纯的转发：携带原来的 Cookies和参数转发原请求，重新查询获取数据
	 */
	public Map<String, Object> forward(String url, HttpServletRequest req) {
		Map<String, String> params = new HashMap<>();
		String str = req.getParameter(Const.JSON_STR);
		if (str != null)
			params.put(Const.JSON_STR, str); // 转发参数
		// 转发处理
		str = httpPost(url, req.getHeader("Cookie"), params);
		return utils.json2Map(str);
	}

	/**
	 * 以下调用当前应用管理服务，获取信息
	 */
	public <T> Map<String, T> corePost(String ws, Map<String, T> params) {
		final String mstUri = utils.getConf("app.core")
				+ utils.getConf("app.main"); // + "/w";
		return doPost(mstUri + ws, params);
	}

}
