package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.dao.AolaiDao;
import cn.ps1.aolai.utils.Const;

/**
 * Aolai基础框架相关服务层，数据库基础操作类(CRUD)
 * 
 * @author Aolai
 * @version 1.0 $Date: 2019.7.6
 * @since 1.7
 */

@Service
public class AolaiService {

	private static Logger LOG = LoggerFactory.getLogger(AolaiService.class);
	private static final String META = "GMETA";
	private static String DOT = null;

	@Autowired
	private AolaiDao aolaiDao;
	@Autowired
	private RedisService redis;
	@Autowired
	private UtilsService utils;

	/** 元数据的元数据 */
	private static Map<String, String> gmeta = new HashMap<>();
	static {
		gmeta.put("field", "META_FIELD");
		gmeta.put("alias", "META_ALIAS");
		gmeta.put("name", "META_NAME");
		gmeta.put("type", "META_TYPE");
		gmeta.put("null", "META_NULL");
		gmeta.put("i18n", "META_I18N");
		gmeta.put("pkey", "META_PKEY");
		gmeta.put("width", "META_WIDTH");
		gmeta.put("default", "META_DEFAULT");
		gmeta.put("sort", "META_SORT");
	}

	/*********************************************************
	 * 往“table”表中增加一条记录
	 */

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 * 
	 * @param table 表名
	 * @param dto 字段映射
	 * @param data 数据
	 * @param dupl 更新标记
	 * @return result 返回状态
	 */
	private Map<String, String> addRecord(String table, Map<String, String> dto,
			Map<String, Object> data, boolean dupl) {
		// 根据table表名获取“别名”“字段”的映射关系
		data = setFieldData(dto, data, true);
		if (utils.isEmpty(data))
			utils.result("2");

		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		if (dupl)
			map.put("dupl", "1");

		// 返回执行结果
		LOG.info("-> addRecord..." + data);
		return utils.result(aolaiDao.addOne(map, data) > 0);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(table, getDto(table), data, dupl);
	}

	/**
	 * 往基础表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table, Map<String, Object> data) {
		return addRecord(table, data, false);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(getTable(base, table), getDto(table), data, dupl);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table,
			Map<String, Object> data) {
		return addRecord(getTable(base, table), getDto(table), data, false);
	}

	/*********************************************************
	 * 批量往业务表中插入多条数据
	 */
	private Map<String, String> batchAdd(String table, Map<String, String> dto,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		if (items.size() == 0)
			return utils.result("2");
		Map<String, Object> data = items.get(0);
		data.put("i18n", lang);
		data = setFieldData(dto, data, false);
		if (utils.isEmpty(data))
			utils.result("2");

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		if (dupl)
			map.put("dupl", "1");

		// 返回执行结果
		LOG.info("->batchAdd..." + map);
		return utils.result(aolaiDao.batchAdd(map, data, items) > 0);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(Object base, String table,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		Map<String, String> dto = getDto(table);
		return batchAdd(getTable(base, table), dto, items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		return batchAdd(table, getDto(table), items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Object lang) {
		return batchAdd(table, items, lang, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items) {
		return batchAdd(table, items, null);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数：fields,duplte
		List<Map<String, Object>> list = utils.json2List(map.get("fields"));
		return batchAdd(map.get("base"), map.get("table"), list,
				map.get("i18n"), map.containsKey("duplte"));
	}

	/*********************************************************
	 * 删除表中的数据
	 */
	private Map<String, String> delete(String table, Map<String, String> dto,
			Map<String, Object> where, Object joint) {
		// 梳理删除条件
		Map<String, Object> cond = setCondition(dto, where, joint);
		if (cond.isEmpty())
			return utils.result("2");
		LOG.info("-> delete..." + cond);

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		return utils.result(aolaiDao.delete(map, cond) > 0);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> where,
			Object joint) {
		return delete(table, getDto(table), where, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> where) {
		return delete(table, getDto(table), where, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table,
			Map<String, Object> where, Object joint) {
		return delete(getTable(base, table), getDto(table), where, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table,
			Map<String, Object> where) {
		return delete(base, table, where, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Map<String, String> params) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> where = utils.json2Map(params.get("where"));
		return delete(params.get("base"), params.get("table"), where);
	}

	/*********************************************************
	 * 批量删除
	 */
	private Map<String, String> batchDelete(String table,
			Map<String, String> dto, List<Map<String, Object>> list) {
		int count = 0;
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		for (int i = 0; i < list.size(); i++) {
			cond = setCondition(dto, list.get(i), null);
			if (cond.isEmpty())
				continue;
			map.put("table", table);
			// 计数
			count += aolaiDao.delete(map, cond);
		}
		// if (count == list.size())
		return utils.result(count > 0);
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(Object base, String table,
			List<Map<String, Object>> list) {
		return batchDelete(getTable(base, table), getDto(table), list);
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(Object base, String table,
			Object where) {
		String str = String.valueOf(where);
		return batchDelete(base, table, utils.json2List(str));
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(String table,
			List<Map<String, Object>> list) {
		return batchDelete(table, getDto(table), list);
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(String table, Object where) {
		String str = String.valueOf(where);
		return batchDelete(table, utils.json2List(str));
	}
	
	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(Map<String, Object> params) {
		String table = String.valueOf(params.get("table"));
		return batchDelete(table, params.get("where"));
	}

	/*********************************************************
	 * 更新表中的数据
	 */
	private Map<String, String> update(String table, Map<String, String> dto,
			Map<String, Object> fields, Map<String, Object> where, Object joint) {
		Map<String, Object> cond = setCondition(dto, where, joint);
		Map<String, Object> values = setCondition(dto, fields, null);
		if (cond.isEmpty() || values.isEmpty())
			return utils.result("2");
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		// 返回执行结果
		return utils.result(aolaiDao.update(map, values, cond) > 0);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table,
			Map<String, Object> fields, Map<String, Object> where, Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// {uid:USER_ID,uid!t:INTEGER,uid!i}
		Map<String, String> dto = getDto(table);
		return update(getTable(base, table), dto, fields, where, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table,
			Map<String, Object> fields, Map<String, Object> where) {
		return update(base, table, fields, where, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> fields,
			Map<String, Object> where, Object joint) {
		return update(null, table, fields, where, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> fields,
			Map<String, Object> where) {
		return update(null, table, fields, where, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Map<String, String> map, Object lang) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> fields = utils.json2Map(map.get("fields"));
		Map<String, Object> where = utils.json2Map(map.get("where"));
		fields.put("i18n", lang);
		where.put("i18n", lang);
		return update(map.get("base"), map.get("table"), fields, where);
	}
	
	/*********************************************************
	 * 支持 join多表查询：left="left"为左连接
	 */
	public List<Map<String, String>> findList(Object base,
			Map<String, Map<String, String>> tables, String alias,
			Map<String, Object> where, Map<String, String> order, String limit) {
		return findList(base, tables, alias, where, order, "left", limit);
	}

	/**
	 * 支持 join多表查询：	left=""为内连接
	 */
	public List<Map<String, String>> findList(Object base,
			Map<String, Map<String, String>> tables, String alias,
			Map<String, Object> where, Map<String, String> order, String left,
			String limit) {
		Map<String, String> dto = getMetaDto(alias);
		String table = joinTable(base, dto, tables, left);
		if (utils.isEmpty(table))
			return new ArrayList<>();

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		Object lang = i18nLang(where);
		map.put("table", table);
		map.put("alias", setAlias(dto, alias, lang));
		map.put("order", getOrder(dto, order, lang));
		setLimitRows(map, limit); // 限定了最大返回记录数

		// 返回执行结果
		return aolaiDao.findList(map, setCondition(dto, where, null));
	}

	/*********************************************************
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public List<Map<String, String>> findList(Map<String, String> args,
			Map<String, Object> where, Map<String, String> order) {
		String table = args.get("table");
		Map<String, String> dto = getDto(table);
		Object lang = i18nLang(where);
		String base = args.get("base");
		String alias = args.get("alias");
		String joint = args.get("joint");
		String limit = args.get("limit"); // 注意：limit=0表示不限行数

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		// 传回前端的参数数组
		map.put("alias", setAlias(dto, alias, lang)); // 无SQL注入风险

		// 根据“数据转换对象DTO”，整理排序字段
		map.put("order", getOrder(dto, order, lang));

		// 限定了最大返回记录数
		setLimitRows(map, limit);

		// 返回执行结果
		return aolaiDao.findList(map, setCondition(dto, where, joint));
	}

	/**
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public List<Map<String, String>> findList(Map<String, String> args,
			Map<String, Object> where) {
		return findList(args, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(Object base, String table,
			String alias, Map<String, Object> where) {
		Map<String, String> args = new HashMap<>();
		if (base != null)
			args.put("base", String.valueOf(base));
		args.put("table", table);
		args.put("alias", alias); // 传回前端的参数数组
		return findList(args, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table, String alias,
			Map<String, Object> where) {
		Map<String, String> args = new HashMap<>();
		args.put("table", table);
		args.put("alias", alias); // 传回前端的参数数组
		return findList(args, where, null);
	}

	/*********************************************************
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	private Map<String, String> findOne(String table, Map<String, String> dto,
			Map<String, Object> where, String keys) {
		Map<String, Object> cond = setCondition(dto, where, null);
		if (cond.isEmpty())
			return utils.result("2");
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		map.put("alias", setAlias(dto, keys, i18nLang(where))); // 无SQL注入风险

		// 返回执行结果，限定了只返回一条记录
		map = aolaiDao.findOne(map, cond);
		return map.isEmpty() ? utils.result(false) : map;
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table,
			Map<String, Object> where, String keys) {
		return findOne(getTable(base, table), getDto(table), where, keys);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table,
			Map<String, Object> where) {
		return findOne(getTable(base, table), getDto(table), where, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 * 
	 * @param table 表名
	 * @param where 查询条件
	 * @param keys 返回值的别名
	 */
	public Map<String, String> findOne(String table, Map<String, Object> where,
			String keys) {
		return findOne(null, table, where, keys);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> where) {
		return findOne(null, table, where, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Map<String, String> map, String keys) {
		Map<String, Object> where = utils.json2Map(map.get("where"));
		return findOne(map.get("base"), map.get("table"), where, keys);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Map<String, String> map) {
		return findOne(map, null);
	}

	/**
	 * 自定义组合条件检索数据
	 */
	private Map<String, String> findOne(String table, Map<String, String> dto,
			String alias, Map<String, Object> where) {
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		map.put("alias", alias);
		// 返回执行结果
		return aolaiDao.findOne(map, setCondition(dto, where, null));
	}

	/**
	 * 自定义组合条件检索数据
	 */
	public Map<String, String> findOne(Object base, String table, String alias,
			Map<String, Object> where) {
		return findOne(getTable(base, table), getDto(table), alias, where);
	}

	/**
	 * 自定义组合条件检索数据，命名queryOne避免与findOne冲突
	 */
	public Map<String, String> queryOne(String table, String alias,
			Map<String, Object> where) {
		return findOne(table, getDto(table), alias, where);
	}
	
	/*********************************************************
	 * 自定义组合条件检索数据
	 */
	@Deprecated
	public List<Map<String, String>> query(String sql) {
		// 返回执行结果
		return aolaiDao.query(sql);
	}

	/*********************************************************
	 * 判断表中是否有数据
	 */
	private boolean exists(String table, Map<String, String> dto,
			Map<String, Object> where, Object joint) {
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		// 返回执行结果
		return aolaiDao.exists(map, setCondition(dto, where, joint)) > 0;
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> where,
			Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// {uid:USER_ID,uid!t:INTEGER,uid!i}
		Map<String, String> dto = getDto(table);
		return exists(getTable(base, table), dto, where, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> where) {
		return exists(base, table, where, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> where, Object joint) {
		return exists(table, getDto(table), where, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> where) {
		return exists(table, getDto(table), where, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> where = utils.json2Map(map.get("where"));
		return exists(map.get("base"), map.get("table"), where);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(String table, Map<String, Object> where) {
		return count(null, table, where, null);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(Object base, String table, Map<String, Object> where) {
		return count(base, table, where, null);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(Object base, String table, Map<String, Object> where,
			String joint) {
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		// 注意这里的条件转换
		where = setCondition(getDto(table), where, joint);
		return aolaiDao.exists(map, where);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/*********************************************************
	 * 根据用户的岗位（dutyId）获取某个应用（pageApp）的菜单控制权限
	 * 
	 * @param map
	 * @param args
	 * @return
	 */
	@Deprecated
	private List<Map<String, String>> getMenuRole(Map<String, Object> map,
			String keys, String table) {
		Map<String, String> dto = getDto(table); // "PAGE"
		map.put("alias", setAlias(dto, keys, i18nLang(map))); // 无SQL注入风险
		// 关联岗位、权限、菜单表中信息
		return aolaiDao.getMenuRole(map);
	}

	/**
	 * 根据用户的岗位（dutyId）获取某个应用（pageApp）的菜单控制权限
	 */
	@Deprecated
	public List<Map<String, String>> getMenuRole(Map<String, Object> map) {
		// 数据库执行参数："getMenuRole"
		return getMenuRole(map, "getMenuRole", "PAGE");
	}

	/**
	 * 根据用户的岗位（dutyId）获取某个账套（baseId）页面（pageRoute）的按钮控制权限
	 */
	@Deprecated
	public List<Map<String, String>> getActionRole(Map<String, Object> map) {
		// 关联岗位、权限、菜单表中信息
		return getActionRole(map, "getActionRole", "ROLE"); // 取消了ACTION表
	}
	
	/**
	 * 根据用户的岗位（dutyId）获取某个账套（baseId）页面（pageRoute）的按钮控制权限
	 */
	@Deprecated
	private List<Map<String, String>> getActionRole(Map<String, Object> map,
			String keys, String table) {
		Map<String, String> dto = getDto(table); // "ROLE"
		map.put("alias", setAlias(dto, keys, i18nLang(map))); // 无SQL注入风险
		// 关联岗位、权限、菜单表中信息
		return aolaiDao.getActionRole(map);
	}

	/**
	 * 获取系统配置参数
	 * 
	 * @param base 帐套名
	 * @param paramKey 参数主键
	 * @return String 返回值
	 */
	public String getParam(Object base, String paramKey) {
		Map<String, Object> where = new HashMap<>();
		Map<String, String> result = new HashMap<>();
		// 获取系统配置参数
		where.put("paramKey", paramKey);
		result = findOne(base, "PARAM", where, "getParamVal");
		return result.get("paramVal");
	}
	
	public String getParam(Map<String, Object> params, String paramKey) {
		return getParam(params.get("base"), paramKey);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 增删改查，字段自动支持国际化
	 */
	private String i18n(Map<String, String> dto, String key, Object lang) {
		String field = dto.get(key);
		// 判断是否支持国际化
		if (lang != null && dto.containsKey(key + "!i"))
			field += "_" + lang; // 获取clientLang，添加“_”符号
		return field;
	}

	/**
	 * 当前查询语言
	 */
	private <T> Object i18nLang(Map<String, T> params) {
		return params == null ? null : params.get("i18n");
	}

	/**
	 * 限定了最大返回记录数（limit.rows）
	 */
	private void setLimitRows(Map<String, String> map, String limit) {
		if (utils.isEmpty(limit) || !utils.isInteger(limit)) {
			map.put("limit", utils.getConf("limit.rows")); // 仅limit=''
		} else if (!"0".equals(limit)) { // 如果limit!=0,不限行数
			map.put("limit", limit);
		}
	}
	
	/**
	 * 查询排序条件：根据“数据转换对象DTO”，拼接排序条件
	 */
	private String getOrder(Map<String, String> dto, Map<String, String> order,
			Object lang) {
		String str = "", key, field;

		// 默认按主键（pkey）排序
		if (utils.isEmpty(order)) {
			for (Map.Entry<String, String> e : dto.entrySet()) {
				key = e.getKey();
				if (key.indexOf("!p") > 0 && "1".equals(e.getValue())) {
					if (str.length() > 0)
						str += ",";
					key = key.substring(0, key.indexOf("!p"));
					str += dto.get(key);
				}
			}
			return str;
		}

		// 按传递的排序参数（order）排序
		for (Map.Entry<String, String> e : order.entrySet()) {
			key = e.getKey();
			if (dto.containsKey(key)) {
				if (str.length() > 0)
					str += ",";
				field = i18n(dto, key, lang); // 支持国际化
				str += field + " " + e.getValue();
			}
		}
		return str;
	}

	/**
	 * 查询条件：字段名对应的数值
	 * @param <T>
	 */
	private <T> Map<String, T> setCondition(Map<String, String> dto,
			Map<String, T> where, Object joint) {
		Map<String, T> cond = new HashMap<>();
		if (!utils.isEmpty(where)) {
			Object lang = i18nLang(where);
			for (Map.Entry<String, T> e : where.entrySet()) {
				String key = e.getKey();
				if (dto.containsKey(key)) { // 不能是“{key}>”、只能是“key”
					joint = joint == null ? "=" : " " + joint;
					key = i18n(dto, key, lang) + joint;
					// 格式如：“USER_NAME_ZH =”, “张三”
					cond.put(key, e.getValue());
				} else if(Pattern.compile("\\{").matcher(key).find()) {
					// 注意预防SQL注入，如：{userId}='1';select * from t1;
					key = getWhere(dto, key, lang);
					// 格式如：“USER_NAME_ZH >= ”, “张三”
					if (key != null) {
						//cond.put(key + joint, e.getValue());
						cond.put(key, e.getValue()); // 带 “{}”的，都带joint=“”
					}
				}
			}
		}
		return cond;
	}

	/**
	 * 替换查询条件的“{key}>=”替换
	 */
	public String getWhere(Map<String, String> dto, String whr) {
		return getWhere(dto, whr, null);
	}

	/**
	 * 替换查询条件的“{key}>=”替换，验证有效的表达式避免SQL注入风险
	 * <p>
	 * 注意预防SQL注入，如：{userId}='1';select * from t1;
	 */
	public String getWhere(Map<String, String> dto, String expr, Object lang) {
		// 如：{userLock}='0' and {userId} in ( '111','222' ) and ...
		String[] arr = expr.split("\\{"); // 分割“{key} in (”
		// 有效表达式如：arr=[“”,“userId} in (”]，arr.length=2，arr[0]=“”时有效
		if (arr.length > 1 && isValidExpr(arr[0])) {
			for (int i = 1; i < arr.length; i++) {
				String[] key = arr[i].split("\\}"); // 如：key=["userId"," in ("]
				// 如果字段名{key}不存在，或赋值有SQL注入风险
				if (!dto.containsKey(key[0]) || !isValidExpr(key[1])) // “ in (”有效
					return null;
				key[0] = i18n(dto, key[0], lang); // 替换：userId>>USER_ID
				arr[i] = key[0] + key[1];
			}
			return utils.arr2Str(arr);
		}
		return null;
	}

	/**
	 * 检查SQL注入，表达式是否有效
	 * @param expr
	 * @return
	 */
	private boolean isValidExpr(String expr) {
		expr = expr.trim();
		if (expr.length() > 0) {
			String[] arr = expr.toUpperCase().split(" "); //如：“ in (”
			for (String str : arr) {
				// 在规定表达式中没有找到
				if (str.length() > 0 && !utils.findIn(Const.SQL_EXPR, str))
					return false;
			}
		}
		return true;
	}

	/**
	 * 把字符串中以“{}”标识出来的关键词替换成“表字段”名
	 * @param <T>
	 */
	/*public <T> Map<String, T> getWhere(Map<String, String> dto,
			Map<String, T> where, Object lang) {
		//if (utils.isEmpty(where))
		//	return where;
		Map<String, T> cond = new HashMap<>();
		for (Map.Entry<String, T> e : where.entrySet()) {
			String key = e.getKey();
			if (Pattern.compile("\\{").matcher(key).find()) {
				String key0 = key.split("\\{")[1].split("\\}")[0];
				if (dto.containsKey(key0)) {
					String key1 = i18n(dto, key0, lang);
					key0 = "\\{" + key0 + "\\}";
					key = key.replaceAll(key0, key1);
					cond.put(key, e.getValue());
				}
			}
		}
		return cond;
	}*/

	/**
	 * 添加数据时：字段名对应数值的键值对
	 */
	private Map<String, Object> setFieldData(Map<String, String> dto,
			Map<String, Object> params, boolean isVal) {
		if (utils.isEmpty(params))
			return null;
		Object lang = i18nLang(params);
		Map<String, Object> data = new HashMap<>();
		for (Map.Entry<String, Object> e : params.entrySet()) {
			String key = e.getKey();
			if (!dto.containsKey(key))
				continue;
			// if (checkDataType(dto, key, val))
			// return null;
			key = i18n(dto, key, lang);
			data.put(key, isVal ? e.getValue() : e.getKey());
		}
		return data;
	}
	
	/**
	 * 根据args参数设置table表中字段的别名
	 */
	public String setAlias(Object table, Object lang) {
		return setAlias(getDto(table), null, lang);
	}
	
	/**
	 * 根据args参数设置table表中字段的别名
	 */
	public String setAlias(Object table, String keys, Object lang) {
		return setAlias(getDto(table), keys, lang);
	}
	
	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String keys, Object lang) {
		// 增加空置判断，为空则返回全部字段（2022.5.2）
		if (utils.isEmpty(keys))
			return setAlias(dto, lang);
		
		// 从配置文件读取参数
		String val = utils.getArgs(keys);
		if (val.length() == 0) // 未定义时默认返回空（""）
			return setAlias(dto, lang);
		
		String[] arr = val.split(";");
		String alias = "";
		for (int i = 0; i < arr.length; i++) {
			if (i > 0)
				alias += ",";
			alias += setAlias(dto, arr[i], dto.get(arr[i]), lang);
		}
		
		return alias;
	}

	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, Object lang) {
		String alias = "", key;
		for (Map.Entry<String, String> e : dto.entrySet()) {
			key = e.getKey();
			if (key.contains("!")) // 为区分type、i18n，标注为!t,!i
				continue;
			if (alias.length() > 0)
				alias += ",";
			alias += setAlias(dto, key, e.getValue(), lang);
		}
		return alias;
	}

	/**
	 * 拼接别名（无SQL注入风险），注意支持的数据类型，注意日期格式勿用TIME
	 */
	private String setAlias(Map<String, String> dto, String key, String val,
			Object lang) {
		String type = dto.get(key + "!t"); // 为区分type、i18n，标注为!t,!i
		if (type != null) { // META_FIELD || VARCHAR
			if (type.matches("[IDBS].+") || type.matches("TIN.+")) {
				// INTEGER\DOUBLE\DECIMAL\DATE+\BIGINT\SMALLINT\TINYINT
				val = "CAST(" + val + " AS CHAR)";
			} else if (type.matches("T.+")) { // TIMESTAMP\勿用TIME
				val = "DATE_FORMAT(" + val + ",'%Y-%m-%d %T')";
			} else if (dto.containsKey(key + "!i")) { // VARCHAR
				val = val + "_" + lang;
			}
		}
		return val + " as '" + key + "'";
	}

	/**
	 * 数据库与前台交互传递的参数
	 */
	public String getFieldAlias(List<Map<String, String>> meta, Object lang) {
		return setAlias(getDto(meta), lang); // 无SQL注入风险
	}

	/**
	 * 元数据对象：获取表的“别名”与“字段”的映射关系，以及数据类型及是否国际化 {uid:USER_ID}
	 */
	private Map<String, String> getDto(Object table) {
		String key = Const.RDS_META + table;
		Map<String, String> dto = redis.hmget(key);
		if (dto == null) {
			dto = getDto(getMetaList(table));
			if (dto != null)
				redis.hmset(key, dto, Const.TEN_HH); // 至少缓存10小时
		}
		return dto;
	}

	/**
	 * 元数据对象
	 */
	private Map<String, String> getDto(List<Map<String, String>> metaList) {
		if (metaList.size() == 0)
			return null;
		Map<String, String> dto = new HashMap<>();
		for (int i = 0; i < metaList.size(); i++) {
			Map<String, String> meta = metaList.get(i);
			String alias = meta.get("alias");
			dto.put(alias, meta.get("field"));
			dto.put(alias + "!t", meta.get("type"));
			dto.put(alias + "!p", meta.get("pkey"));
			if ("1".equals(meta.get("i18n")))
				dto.put(alias + "!i", meta.get("i18n"));
		}
		return dto;
	}

	/**
	 * 仅为基础维护校验参数时用
	 * <p>
	 * 直接获取Meta对象的元数据的属性keys数组
	 * <p>
	 * [“alias”,“name”,“type”,“null”,“i18n”,“pkey”,“width”,“default”,“sort”,“table”]
	 */
	public String[] getDtoKeys() {
		return getDtoKeys(gmeta);
	}

	/**
	 * 仅为基础维护校验参数时用
	 * <p>
	 * 根据Meta先获取“数据转换对象DTO”，再获取元数据的属性keys数组
	 */
	public String[] getDtoKeys(Map<String, String> map) {
		String[] keys = new String[map.size() + 1];
		map.keySet().toArray(keys);
		keys[map.size()] = "table"; // 扩展一个表名‘table’字段
		return keys;
	}

	/**
	 * 仅为基础维护校验参数时用
	 * <p>
	 * 根据tableName先获取“数据转换对象DTO”，再获取元数据的属性keys数组
	 */
	public String[] getDtoKeys(String table) {
		List<Map<String, String>> list = getMetaList(table);
		Map<String, String> map = new HashMap<>();
		for (Map<String, String> meta : list) {
			if ("1".equals(meta.get("pkey")))
				map.put(meta.get("alias"), meta.get("field"));
		}
		return getDtoKeys(map);
	}

	/**
	 * 根据 表名table从GMETA表中获取Meta元数据一览
	 * 
	 * @param table 表名
	 * @return list 元数据一览
	 */
	public List<Map<String, String>> getMetaList(Object table) {
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		map.put("table", META);
		map.put("alias", setAlias(gmeta, null)); // 无SQL注入风险
		map.put("order", "META_SORT");
		cond.put("META_TABLE=", table);
		return aolaiDao.findList(map, cond);
	}

	/**
	 * 根据 META_ALIAS从GMETA表中获取Meta元数据一览
	 */
	public List<Map<String, String>> getMeta(String alias) {		
		String val = utils.getArgs(alias);
		Map<String, String> map = new HashMap<>();
		map.put("alias", setAlias(gmeta, null)); // 无SQL注入风险
		String[] cond = val.length() == 0 ? null : val.split(";");
		return aolaiDao.getMetaInfo(map, cond);
	}

	private Map<String, String> getMetaDto(String alias) {
		String key = Const.RDS_META + alias;
		Map<String, String> dto = redis.hmget(key);
		if (dto == null) {
			dto = getDto(getMeta(alias));
			if (dto != null)
				redis.hmset(key, dto, Const.TEN_HH);
		}
		return dto;
	}

	/**
	 * 联合多表查询
	 */
	private String joinTable(Object base, Map<String, String> dto,
			Map<String, Map<String, String>> tables, String left) {
		if (utils.isEmpty(tables))
			return null;
		String s = "", t = "";
		// {TB1:<null>,TB2<TB2.A=TB1.B>}
		for (String tableName : tables.keySet()) {
			Map<String, String> map = tables.get(tableName);
			tableName = getTable(base, tableName);
			if (utils.isEmpty(map)) {
				if (s.length() == 0)
					s = tableName;
				continue;
			} else {
				int n = 0;
				t += " " + left + " join " + tableName + " on ";
				for (Map.Entry<String, String> ent : map.entrySet()) {
					if (n > 0)
						t += " and ";
					t += i18n(dto, ent.getKey(), null) + "=";
					t += i18n(dto, ent.getValue(), null);
					n++;
				}
			}
		}
		if (s.length() == 0)
			return null;
		return (t.length() == 0) ? s : s + t;
	}

	/**
	 * 账套（base）及表名组合
	 */
	public String getTable(Object base, String table) {
		if (utils.isEmpty(base))
			return table;
		if (DOT == null) {
			String dot = utils.getConf("base.dot");
			DOT = utils.isEmpty(dot) ? "." : dot;
		}
		return base + DOT + table;
	}

	/** * * * * * * * 以下对GMETA的基本操作 * * * * * * * **/

	/**
	 * 仅操作GMETA表的元数据“键值对”
	 */
	private Map<String, String> getMetaDto() {
		Map<String, String> map = new HashMap<>();
		map.putAll(gmeta);
		map.put("table", "META_TABLE");
		return map;
	}

	/**
	 * 往GMETA表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addMeta(Map<String, Object> data,
			boolean dupl) {
		return addRecord(META, getMetaDto(), data, dupl);
	}

	/**
	 * 从Meta表中批量删除数据
	 */
	public Map<String, String> delMeta(List<Map<String, Object>> list) {
		return batchDelete(META, getMetaDto(), list);
	}

	/**
	 * 从Meta表中获取元数据表名信息（暂停使用）
	 */
	public List<Map<String, String>> getMetaTable() {
		return aolaiDao.getMetaTable();
	}

	/**
	 * 判断某库、某表名是否存在（暂未使用：暂无SQL注入风险）
	 */
	public boolean tableExists(Object base, String table) {
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		cond.put("TABLE_SCHEMA", base);
		cond.put("TABLE_NAME", table);
		map.put("table", "information_schema.TABLES");
		return aolaiDao.exists(map, cond) > 0;
	}

	/**
	 * 判断某库、某表、某列名是否存在（暂未使用：暂无SQL注入风险）
	 */
	public boolean columnExists(Object base, String table) {
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		cond.put("TABLE_SCHEMA=", base);
		cond.put("COLUMN_NAME=", table);
		map.put("table", "information_schema.COLUMNS");
		return aolaiDao.exists(map, cond) > 0;
	}

	/*********************************************************
	 * 删除表
	 * 
	 * @param table
	 */
	@Deprecated
	public Map<String, String> dropTable(Object base, String table) {
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		// 动态删表
		aolaiDao.drop(map);
		redis.del(Const.RDS_META + table);
		// 返回成功
		return utils.result(true);
	}

	/*********************************************************
	 * 创建表
	 */
	@Deprecated
	public Map<String, String> createTable(Object base, String table) {
		// 先从META表中获取元数据
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		map.put("fields", getMetaSql(getMetaList(table)));
		// 动态建表
		LOG.debug("-> createTable..." + map);
		aolaiDao.create(map);
		// 返回成功
		return utils.result(true);
	}

	/**
	 * 根据Meta元数据整理建表Sql
	 */
	private String getMetaSql(List<Map<String, String>> meta) {
		String sql = "", field, pkey = "";
		String[] lang = utils.getLocales(); // 国际化支持
		for (int i = 0; i < meta.size(); i++) {
			if (i > 0)
				sql += ",";
			Map<String, String> map = meta.get(i);
			// 支持国际化的字段
			if ("1".equals(map.get("i18n"))) {
				for (int j = 0; j < lang.length; j++) {
					if (j > 0)
						sql += ",";
					field = map.get("field") + "_" + lang[j];
					sql += field + " " + map.get("type");
					if (map.get("null").equals("1"))
						sql += " NOT NULL";
					if (map.get("pkey").equals("1")) {
						if (pkey.length() > 0)
							pkey += ",";
						pkey += field;
					}
					if (map.get("default").length() > 0)
						sql += " DEFAULT " + map.get("default");
				}
			} else {
				field = map.get("field");
				sql += field + " " + map.get("type");
				if (map.get("null").equals("1"))
					sql += " NOT NULL";
				if (map.get("pkey").equals("1")) {
					if (pkey.length() > 0)
						pkey += ",";
					pkey += field;
				}
				if (map.get("default").length() > 0)
					sql += " DEFAULT " + map.get("default");
			}

		}
		if (pkey.length() > 0)
			sql += ",PRIMARY KEY (" + pkey + ")";
		return sql;
	}

	/**
	 * 从表中获取最大值
	 */
	public int getMaxCode(Object base, String table, String field) {
		return getMaxCode(base, table, field, null);
	}

	/**
	 * 从表中获取最大值
	 */
	public int getMaxCode(Object base, String table, String field,
			Map<String, Object> where) {
		Map<String, String> dto = getDto(table);
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		map.put("field", dto.get(field));
		return aolaiDao.getMaxCode(map, setCondition(dto, where, null));
	}

}
