package cn.ps1.aolai.utils;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Random;
import java.util.UUID;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;

import org.apache.commons.codec.digest.DigestUtils;

/**
 * 根据自定义字符串对任意字节数据进行类似Base64编码的简单算法
 * 
 * @author Fulin
 * @version 1.0
 *
 */
public class Digest {

	static final String CHARS64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
	public static String KEY = "DFJKMQSVWXZ"; // dfjkmqsvwxz

	static ScriptEngineManager sem = new ScriptEngineManager();
	static ScriptEngine se = sem.getEngineByExtension("js");

	/**
	 * 任意随机数
	 */
	public static Random rand = new Random();

	/**
	 * 任意随机生成 N位数字
	 * 
	 * @param len 长度
	 * @return
	 */
	public static String randInt(int len) {
		String str = "";
		for (int i = 0; i < len; i++) {
			str += String.valueOf(rand.nextInt(10));
		}
		return str;
	}

	/**
	 * 根据11位字符串随机排序生成KeyId
	 * 
	 * @return String 11位字符串（如：‘dfjkmqsvwxz’）
	 */
	public static String randKey() {
		return randStr(KEY, 11);
	}

	/**
	 * 任意生成的包含26位字母的随机字符串
	 * 
	 * @return String 随机26位字母
	 */
	public static String randStr() {
		return randStr(CHARS64, 26);
	}

	/**
	 * 任意字符串随机打乱顺序
	 * 
	 * @return String 打乱顺序后的字符串
	 */
	public static String randStr(String str) {
		return randStr(str, str.length());
	}

	/**
	 * 把一个N位字符串随机排序
	 * @param str 原始字符串
	 * @param len 保留的长度
	 * @return String 随机排序的N位字符串
	 */
	public static String randStr(String str, int len) {
		char[] charArr = str.toCharArray();
		for (int i = 0; i < len; i++) {
			int idx = rand.nextInt(len);
			if (idx == i)
				continue;
			char chr = charArr[idx];
			if (idx % 3 == 0 && chr < 91)
				chr = (char) (chr + 32); // 字母随机变个小写
			charArr[idx] = charArr[i]; // 序列变换
			charArr[i] = chr;
		}
		return new String(Arrays.copyOf(charArr, len));
	}

	/**
	 * 根据certId和certKey拼接解码证书串
	 * 
	 * @return
	 */
	public static String certStr(String str, String key) {
		String cert = str + key;
		return cert.toUpperCase() + cert.toLowerCase()
				+ CHARS64.substring(52);
	}

	/**
	 * 根据certId拼接解码证书串
	 * 
	 * @param certId
	 * @return
	 */
	public static String certStr(String key) {
		if (key == null) {
			return CHARS64;
		} else if (key.length() == 26) {
			return certStr(key, "");
		} else if (key.length() > 64) {
			return key;
		}
		return certStr("uncopyrightable", key); // 默认证书
	}

	/**
	 * 对字符串进行编码
	 */
	public static String encrypt(String str) {
		return encrypt(str, CHARS64);
	}

	/**
	 * 对字符数组编码
	 */
	public static String encrypt(byte[] bytes) {
		return encrypt(bytes, CHARS64);
	}

	/**
	 * 用指定证书，对字符串进行编码
	 */
	public static String encrypt(String str, String cert) {
		if (str == null)
			return str;
		return encrypt(str.getBytes(Const.CHARSET), cert);
	}

	/**
	 * 对字符数组进行编码
	 */
	public static String encrypt(byte[] bytes, String cert) {
		cert = certStr(cert);
		// 汉字三个字节，英文一个字符，如： 字符a的ASCII码是97，字符o的ASCII码是111
		byte[] keys = cert.getBytes(Const.CHARSET);
		byte[] out = new byte[((bytes.length + 2) / 3) * 4];
		for (int i = 0, idx = 0; i < bytes.length; i += 3, idx += 4) {
			boolean[] trip = { true, false, false };
			int bit = (0xFF & bytes[i]);
			for (int b = 1; b < 3; b++) {
				bit <<= 8; // 先向左移一个字节
				if (i < bytes.length - b) {
					bit |= (0xFF & bytes[i + b]); // 再拼接一个字节
					trip[b] = true;
				}
			}
			for (int b = 0; b < 3; b++) {
				out[idx + 3 - b] = keys[(trip[2 - b] ? (bit & 0x3F) : 64)];
				bit >>= 6;
			}
			out[idx] = keys[bit & 0x3F];
		}
		return new String(out);
	}

	/**
	 * 对编码过的字符串进行解码
	 */
	public static String decrypt(String str) {
		return decrypt(str, CHARS64);
	}

	/**
	 * 对编码过的字符数组进行解码
	 */
	public static byte[] decrypt(char[] arr) {
		return decrypt(arr, CHARS64);
	}

	/**
	 * 用指定证书，对编码过的字符串进行解码
	 */
	public static String decrypt(String str, String cert) {
		if (str == null)
			return str;
		byte[] bytes = decrypt(str.toCharArray(), cert);
		return new String(bytes, Const.CHARSET);
	}

	/**
	 * 用指定证书，对编码过的字符数组进行解码
	 */
	public static byte[] decrypt(char[] arr, String cert) {
		cert = certStr(cert);
		char end = cert.charAt(cert.length() - 1);
		int len = arr.length * 6 / 8;
		for (int i = arr.length - 1; arr[i] == end; i--)
			len--;
		byte[] raw = new byte[len];
		int[] quad = new int[4];
		int rawIdx = 0;
		for (int i = 0; i < arr.length; i += 4) {
			int block = 0;
			for (int idx = 0; idx < 4; idx++) {
				char chr = arr[i + idx];
				quad[idx] = chr == end ? 0 : cert.indexOf(chr);
				block += (quad[idx] << 6 * (3 - idx));
			}
			for (int b = 0; b < 3 && rawIdx + b < raw.length; b++)
				raw[rawIdx + b] = (byte) ((block >> (8 * (2 - b))) & 0xFF);
			rawIdx += 3;
		}
		return raw;
	}

	/**
	 * 用于浏览器cookie编码
	 * 
	 * @param src
	 * @return
	 */
	public static String escape(String src) {
		try {
			Object obj = se.eval("escape('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}

	/**
	 * 用于浏览器cookie解码
	 * 
	 * @param src
	 * @return
	 */
	public static String unescape(String src) {
		try {
			Object obj = se.eval("unescape('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}
	
	/**
	 * 用于浏览器cookie编码
	 * 
	 * @param src
	 * @return
	 */
	public static String encodeURIComponent(String src) {
		try {
			Object obj = se.eval("encodeURIComponent('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}

	/**
	 * 用于浏览器cookie解码
	 * 
	 * @param src
	 * @return
	 */
	public static String decodeURIComponent(String src) {
		try {
			Object obj = se.eval("decodeURIComponent('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}

	/**
	 * 读取图片字节数组
	 * 
	 * @param imgFile
	 * @return
	 */
	@SuppressWarnings("restriction")
	public static String imgEncode(String imgFile) {
		byte[] buf = null;
		try {
			InputStream in = new FileInputStream(imgFile);
			buf = new byte[in.available()];
			in.read(buf);
			in.close();
		} catch (IOException e) {
			e.printStackTrace();
			return "";
		}
		return new sun.misc.BASE64Encoder().encode(buf);
	}

	/**
	 * 还原图片
	 * 
	 * @param imgFile
	 * @return
	 */
	@SuppressWarnings("restriction")
	public static boolean imgDecode(String imgStr, String imgFile) {
		try {
			byte[] buf = new sun.misc.BASE64Decoder().decodeBuffer(imgStr);
			// for (int i = 0; i < buf.length; i++) {
			// if (buf[i] < 0) {
			// buf[i] += 256;
			// }
			// }
			OutputStream out = new FileOutputStream(imgFile);
			out.write(buf);
			out.flush();
			out.close();
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	public static String urlEncode(String url) {
		try {
			return URLEncoder.encode(url, Const.UTF8);
		} catch (Exception e) {
			e.printStackTrace();
			return url;
		}
	}

	public static String urlDecode(String url) {
		try {
			return URLDecoder.decode(url, Const.UTF8);
		} catch (Exception e) {
			e.printStackTrace();
			return url;
		}
	}

	/**
	 * 生成通用唯一识别码
	 */
	public static String uuid() {
		return UUID.randomUUID().toString();
	}

	/**
	 * 生成通用唯一识别码
	 */
	/*private static String uuid8(String uuid) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < 8; i++) {
			String str = uuid.substring(i * 4, i * 4 + 4);
			int x = Integer.parseInt(str, 16);
			// 如果是 chars62，则是x%62
			buf.append(CHARS64.charAt(x % 36));
		}
		return buf.toString();
	}*/

	/**
	 * 根据时间戳生成8位唯一识别码
	 */
	public static String uuid8() {
		// return uuid8(uuid().replace("-", ""));
		return enBase64(System.nanoTime() / 100);
	}

	/**
	 * 把十进制的数字转换为64进制编码
	 */
	public static String enBase64(long num) {
		StringBuffer buf = new StringBuffer();
		char c;
		while (num > 0) {
			c = CHARS64.charAt(new Long(num % 64).intValue());
			buf.append(c);
			num = num / 64;
		}
		return buf.reverse().toString();
	}

	/**
	 * 把64进制字符解码为十进制数字
	 */
	public static long deBase64(String str) {
		// 基数
		long radix = 1;
		long result = 0;
		char c;
		for (int i = 0; i < str.length(); i++) {
			c = str.charAt(str.length() - 1 - i);
			result += CHARS64.indexOf(c) * radix;
			radix = radix * 64;
		}
		return result;
	}

	public static String md5id(String str) {
		// DigestUtils.md5DigestAsHex(str.getBytes());
		return DigestUtils.md5Hex(str);
	}

}
