package cn.ps1.aolai.utils;

import java.io.Closeable;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PushbackInputStream;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @ClassName: 操作Excel的工具类
 * @Date: 2022/9/25 15:23
 * @author Aolai
 */
public class XlsxUtil {

	private static Logger LOG = LoggerFactory.getLogger(XlsxUtil.class);

	private String xFile; // Excel文件名
	private Workbook workbook;
	private Sheet sheet;

	/** 构造函数 */
	public XlsxUtil() {
	}

	/** 构造函数 */
	public XlsxUtil(String xFile) {
		this.xFile = xFile; // Excel文件名
		initWorkbook();
	}

	/** 初始化 */
	public void initWorkbook() {
		InputStream inp = null;
		try {
			// 读取Excel文件
			inp = new FileInputStream(xFile);
			initWorkbook(inp);
		} catch (Exception e) {
			LOG.error("-> initWorkbook... " + e.getMessage());
			newWorkbook();
		} finally {
			close(inp);
		}
	}

	/** 初始化 */
	public void initWorkbook(InputStream inp) {
		try {
			// 获取输入流是否支持mark和reset操作
			if (!inp.markSupported())
				inp = new PushbackInputStream(inp, 8);
			initSheet(WorkbookFactory.create(inp), 0);
		} catch (Exception e) {
			LOG.error("-> initWorkbook... " + e.getMessage());
			newWorkbook();
		}
	}

	/** 初始化 */
	public void newWorkbook() {
		workbook = new XSSFWorkbook(); // 仅支持xlsx文件
		sheet = workbook.createSheet(); // 创建工作表
	}

	/**
	 * 设置工作薄
	 */
	private void initSheet(Workbook workbook, int index) {
		this.workbook = workbook;
		this.sheet = workbook.getSheetAt(index);
	}

	/**
	 * 读取Excel文件中的数据
	 */
	synchronized public List<Map<String, String>> readExcel(String file,
			int startRow, Map<String, Integer> header) {
		List<Map<String, String>> dataList = new ArrayList<>();
		XlsxUtil excel = new XlsxUtil(file);
		dataList = excel.getDataByRow(startRow, header);
		return dataList;
	}

	/**
	 * 把Workbook缓存数据写入Excel文件
	 */
	synchronized public void writeExcel(String file) {
		OutputStream out = null;
		try {
			out = new FileOutputStream(file);
			writeExcel(out);
			out.flush();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			close(out);
		}
	}

	/**
	 * 把Workbook缓存数据写入Excel文件
	 */
	synchronized public void writeExcel(OutputStream out) throws IOException {
		workbook.write(out);
	}

	/**
	 * 关闭对象
	 */
	public static void close(Closeable... closeables) {
		if (closeables != null && closeables.length > 0) {
			for (Closeable closeable : closeables) {
				if (closeable == null)
					continue;
				try {
					closeable.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * 复制行信息
	 */
	public void shiftRows(int startRow, int endRow, int rows) {
		sheet.shiftRows(startRow, endRow, rows, true, false);
	}

	/**
	 * 克隆空白行
	 */
	public void cloneBlankRow(int curRow, int oriRow) {
		Row insertRow = sheet.createRow(curRow); // 插入新行
		Row originRow = sheet.getRow(oriRow); // 模板参照行
		insertRow.setHeight(originRow.getHeight());

		// 获取当前行样式
		CellStyle rowStyle = originRow.getRowStyle();
		if (rowStyle != null) {
			insertRow.setRowStyle(rowStyle); // 继承样式
		}
		cloneMergedRegion(curRow, oriRow); // 设置合并单元格
	}

	/**
	 * 设置合并单元格
	 */
	public void cloneMergedRegion(int curRow, int oriRow) {
		// 原始的合并单元格的地址集合
//		List<CellRangeAddress> originMerged = sheet.getMergedRegions();
		for (int i = 0; i < sheet.getNumMergedRegions(); i++) {
			CellRangeAddress region = sheet.getMergedRegion(i);
			if (region.getFirstRow() == oriRow) {
				int rows = region.getLastRow() - oriRow;
				CellRangeAddress newRegion = new CellRangeAddress(curRow,
						curRow + rows, region.getFirstColumn(),
						region.getLastColumn());
				sheet.addMergedRegion(newRegion);
			}
		}
	}

	/**
	 * 增加一条Excel空白数据行
	 *
	 * @param curRow 当前空行
	 * @param srcRow 资源行
	 */
//	private void addExcelBlankRow(int curRow, int srcRow) {
//		// 从当前行全部下移一行，留空后插入rowScope行
//		int rows = sheet.getLastRowNum() + rowScope;
//		sheet.shiftRows(curRow, rows, rowScope, true, false);
//		for (int i = 0; i < rowScope; i++) {
//			// 获取当前行
//			Row rowSource = sheet.getRow(srcRow + i);
//			Row rowInsert = sheet.createRow(curRow + i); // 新增插入一行
//			rowInsert.setHeight(rowSource.getHeight());
//			// 获取当前行样式
//			CellStyle rowStyle = rowSource.getRowStyle();
//			if (rowStyle != null) {
//				rowInsert.setRowStyle(rowStyle); // 继承样式
//			}
//		}
//	}

	/**
	 * 从Excel当前行（curRow）开始删除多行（rows）
	 */
	public void removeExcelRow(int curRow, int rows) {
		int endRow = curRow + rows;
		// 倒序删除合并单元格
		for (int i = sheet.getNumMergedRegions() - 1; i >= 0; i--) {
			CellRangeAddress region = sheet.getMergedRegion(i);
			if (region.getFirstRow() >= curRow && region.getFirstRow() < endRow) {
				sheet.removeMergedRegion(i); // 删除合并单元格
			}
		}
		sheet.shiftRows(endRow, sheet.getLastRowNum() + 1, -rows, true, false);
	}

	/**
	 * 逐行读取数据
	 *
	 * @param startRow 开始行
	 * @param header 对应列序号
	 */
	public List<Map<String, String>> getDataByRow(int startRow,
			Map<String, Integer> header) {
		List<Map<String, String>> dataList = new ArrayList<>();
		if (header == null || header.isEmpty())
			return dataList;
		for (int i = startRow; i <= sheet.getLastRowNum(); i++) {
			Row row = sheet.getRow(i);
			if (row == null)
				break;
			// 逐行读取行数据
			Map<String, String> rowMap = new HashMap<>();
			for (Map.Entry<String, Integer> entry : header.entrySet()) {
				if ("".equals(entry.getKey()))
					continue;
				// 逐列读取数据
				Cell cell = row.getCell(entry.getValue()); // 列序号
				rowMap.put(entry.getKey(), getCellValue(cell));
			}
			dataList.add(rowMap);
		}
		return dataList;
	}

	/**
	 * 获取单元格的数据
	 *
	 * @param cell 单元格
	 * @return String 值
	 */
	public String getCellValue(Cell cell) {
		if (cell == null)
			return "";
		Object val = "";
		switch (cell.getCellType()) {
		case STRING: // 1-字符串
			val = cell.getStringCellValue();
			break;
		case NUMERIC: // 0-数值
			val = new BigDecimal(cell.getNumericCellValue()).toPlainString();
//			val = cell.getNumericCellValue();
			break;
		case FORMULA: // 2-公式
			// val = cell.getCellFormula(); // 返回：SUM(C4:E4)
			val = cell.getNumericCellValue(); // 返回预先计算的值
			break;
		case BOOLEAN: // 4-Boolean
			val = cell.getBooleanCellValue();
			break;
		// case BLANK: // 3-空格，返回 0或空串
		// case ERROR: // 5-错误
		default: // 字符串
			val = "";
		}
		return String.valueOf(val);
	}

	/**
	 * 根据行和列的索引获取单元格的数据
	 *
	 * @param rowNo 行号
	 * @param colNo 列号
	 * @return String 值
	 */
	public String getCellValue(int rowNo, int colNo) {
		Row row = sheet.getRow(rowNo);
		if (row == null)
			return "";
		Cell cell = row.getCell(colNo);
		return getCellValue(cell);
	}

	/**
	 * 根据行和列的索引设置单元格的数据
	 *
	 * @param rowNo 行
	 * @param colNo 列
	 * @param value 值
	 */
	public void setCellValue(int rowNo, int colNo, String value) {
		Row row = sheet.getRow(rowNo);
		if (row == null) {
			row = sheet.createRow(rowNo);
		}
		Cell cell = row.getCell(colNo);
		if (cell == null) {
			cell = row.createCell(colNo);
		}
		cell.setCellValue(value);
	}

	/**
	 * 获取指定列的列宽
	 */
	public float[] getColWidth(int maxColNum) {
		float[] widths = new float[maxColNum];
		for (int j = 0; j < maxColNum; j++) {
			widths[j] = sheet.getColumnWidthInPixels(j);
		}
		return widths;
	}

	/**
	 * 获取最大行号
	 */
	public int getMaxRowNum() {
		return sheet.getLastRowNum();
	}

	/**
	 * 获取Excel页的最大列数
	 */
	public int getMaxColNum(int rowNum) {
		int maxColNum = 0;
		for (int i = 0; i < rowNum; i++) {
			Row row = sheet.getRow(i);
			if (row == null)
				continue;
			int colNum = row.getLastCellNum();
			if (colNum > maxColNum)
				maxColNum = colNum;
		}
		return maxColNum;
	}

	public Workbook getWorkbook() {
		return workbook;
	}

	public Sheet getSheet() {
		return sheet;
	}

	public Row getRow(int i) {
		return sheet.getRow(i);
	}

	public String getFile() {
		return xFile;
	}

}
