package cn.ps1.aolai.utils;

import java.io.Closeable;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Base64;
import java.util.Random;
import java.util.UUID;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;

/**
 * 根据自定义字符串对任意字节数据进行类似Base64编码的简单算法
 * 
 * @author Aolai
 * @since 1.8
 *
 */
public class Digest {

	static final String CHARS64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
	public static String KEY = "DFJKMQSVWXZ"; // dfjkmqsvwxz

	static ScriptEngineManager sem = new ScriptEngineManager();
	/** 为防止脚本注入，慎用！ */
	static ScriptEngine se = sem.getEngineByExtension("js");

	/**
	 * 任意随机数
	 */
	public static Random rand = new Random();

	/**
	 * 任意随机生成 N位数字
	 * 
	 * @param len 长度
	 * @return String
	 */
	public static String randInt(int len) {
		String str = "";
		for (int i = 0; i < len; i++) {
			str += String.valueOf(rand.nextInt(10));
		}
		return str;
	}

	/**
	 * 根据11位字符串随机排序生成KeyId
	 * 
	 * @return String 11位字符串（如：‘dfjkmqsvwxz’）
	 */
	public static String randKey() {
		return randStr(KEY, 11);
	}

	/**
	 * 任意生成的包含26位字母的随机字符串
	 * 
	 * @return String 随机26位字母
	 */
	public static String randStr() {
		return randStr(26);
	}
	
	public static String randStr(int len) {
		//return randStr(CHARS64, len);
		return randStr(CHARS64.substring(0, len), len);
	}

	/**
	 * 任意字符串随机打乱顺序
	 * 
	 * @return String 打乱顺序后的字符串
	 */
	public static String randStr(String str) {
		return randStr(str, str.length());
	}

	/**
	 * 把一个N位字符串随机排序
	 * @param str 原始字符串
	 * @param len 保留的长度
	 * @return String 随机排序的N位字符串
	 */
	public static String randStr(String str, int len) {
		char[] charArr = str.toCharArray();
		for (int i = 0; i < len; i++) {
			//int idx = rand.nextInt(len);
			int idx = rand.nextInt(str.length());
			if (idx == i) // 跳过自己
				continue;
			
			// 序列变换：第i个与第n个位置互换
			char c = charArr[idx];
//			if (idx % 3 == 0 && c < 91)
//				c = (char) (c + 32); // 字母随机变个小写
			
			charArr[idx] = charArr[i];
			charArr[i] = c;
		}
		return new String(Arrays.copyOf(charArr, len));
	}

	/**
	 * 根据certId和certKey拼接解码用的通行证字符串
	 * 
	 * @return String
	 */
	public static String certStr(String str, String key) {
		String cert = str + key;
		return cert.toUpperCase() + cert.toLowerCase()
				+ CHARS64.substring(52);
	}

	/**
	 * 根据 key(certId)拼接完整的通行证
	 * 
	 * @param key 默认为11位或26位通行证(certId)
	 */
	public static String certStr(String key) {
		if (key == null) {
			return CHARS64;
		} else if (key.length() == 26) {
			return certStr(key, "");
		} else if (key.length() > 64) {
			return key;
		}
		return certStr("uncopyrightable", key); // 默认通行证
	}

	/**
	 * 对字符串进行编码
	 */
	public static String encrypt(String str) {
		return encrypt(str, CHARS64);
	}

	/**
	 * 对字符数组编码
	 */
	public static String encrypt(byte[] bytes) {
		return encrypt(bytes, CHARS64);
	}

	/**
	 * 用指定通行证，对字符串进行编码
	 * 
	 * @param str
	 * @param cert 默认11位或 26位通行证
	 */
	public static String encrypt(String str, String cert) {
		if (str == null)
			return str;
		return encrypt(str.getBytes(Const.CHARSET), cert);
	}

	/**
	 * 用指定通行证，对字符数组进行编码
	 * 
	 * @param bytes
	 * @param cert 默认11位或 26位通行证
	 */
	public static String encrypt(byte[] bytes, String cert) {
		cert = certStr(cert);
		// 汉字三个字节，英文一个字符，如： 字符a的ASCII码是97，字符o的ASCII码是111
		byte[] keys = cert.getBytes(Const.CHARSET);
		byte[] out = new byte[((bytes.length + 2) / 3) * 4];
		for (int i = 0, idx = 0; i < bytes.length; i += 3, idx += 4) {
			boolean[] trip = { true, false, false };
			int bit = (0xFF & bytes[i]);
			for (int b = 1; b < 3; b++) {
				bit <<= 8; // 先向左移一个字节
				if (i < bytes.length - b) {
					bit |= (0xFF & bytes[i + b]); // 再拼接一个字节
					trip[b] = true;
				}
			}
			for (int b = 0; b < 3; b++) {
				out[idx + 3 - b] = keys[(trip[2 - b] ? (bit & 0x3F) : 64)];
				bit >>= 6;
			}
			out[idx] = keys[bit & 0x3F];
		}
		return new String(out);
	}

	/**
	 * 对编码过的字符串进行解码
	 */
	public static String decrypt(String str) {
		return decrypt(str, CHARS64);
	}

	/**
	 * 对编码过的字符数组进行解码
	 */
	public static byte[] decrypt(char[] arr) {
		return decrypt(arr, CHARS64);
	}

	/**
	 * 用指定通行证，对编码过的字符串进行解码
	 * 
	 * @param str
	 * @param cert 默认11位或 26位通行证
	 */
	public static String decrypt(String str, String cert) {
		if (str == null)
			return str;
		byte[] bytes = decrypt(str.toCharArray(), cert);
		return new String(bytes, Const.CHARSET);
	}

	/**
	 * 用指定通行证，对字符数组进行解码
	 * <p>
	 * @param cert 默认11位或 26位通行证
	 */
	public static byte[] decrypt(char[] arr, String cert) {
		cert = certStr(cert);
		char end = cert.charAt(cert.length() - 1);
		int len = arr.length * 6 / 8;
		for (int i = arr.length - 1; arr[i] == end; i--)
			len--;
		byte[] raw = new byte[len];
		int[] quad = new int[4];
		int rawIdx = 0;
		for (int i = 0; i < arr.length; i += 4) {
			int block = 0;
			for (int idx = 0; idx < 4; idx++) {
				char c = arr[i + idx];
				quad[idx] = c == end ? 0 : cert.indexOf(c);
				block += (quad[idx] << 6 * (3 - idx));
			}
			for (int b = 0; b < 3 && rawIdx + b < raw.length; b++)
				raw[rawIdx + b] = (byte) ((block >> (8 * (2 - b))) & 0xFF);
			rawIdx += 3;
		}
		return raw;
	}

	/**
	 * 用于浏览器cookie编码，为防止脚本注入，慎用！
	 * 
	 * @param src
	 * @return String
	 */
	public static String escape(String src) {
		try {
			Object obj = se.eval("escape('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}

	/**
	 * 用于浏览器cookie解码，为防止脚本注入，慎用！
	 * 
	 * @param src
	 * @return String
	 */
	public static String unescape(String src) {
		try {
			Object obj = se.eval("unescape('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}
	
	/**
	 * 用于浏览器cookie编码，为防止脚本注入，慎用！
	 * 
	 * @param src
	 * @return String
	 */
	public static String encodeURIComponent(String src) {
		try {
			Object obj = se.eval("encodeURIComponent('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}

	/**
	 * 用于浏览器cookie解码，为防止脚本注入，慎用！
	 * 
	 * @param src
	 * @return String
	 */
	public static String decodeURIComponent(String src) {
		try {
			Object obj = se.eval("decodeURIComponent('" + src + "')");
			return String.valueOf(obj);
		} catch (Exception e) {
			return src;
		}
	}

	/**
	 * 读取图片字节数组，把图片image转换为base64格式的字符数据集
	 * 
	 * @param imgFile 文件路径
	 */
//	@SuppressWarnings("restriction")
	public static String imgEncode(String imgFile) {
		byte[] dataBuf = null;
		InputStream inp = null;
		try {
			inp = new FileInputStream(imgFile);
			dataBuf = new byte[inp.available()];
			inp.read(dataBuf);
//			inp.close();
		} catch (IOException e) {
			e.printStackTrace();
			return "";
		} finally {
			close(inp);
		}
//		return new sun.misc.BASE64Encoder().encode(dataBuf);
		// 获取编码器
		Base64.Encoder encoder = Base64.getEncoder();
		return encoder.encodeToString(dataBuf);
	}

	/**
	 * 还原图片，把base64格式的字符数据集转换为图片，存到指定文件中
	 * 
	 * @param imgStr 字符数据集
	 * @param imgFile 文件路径
	 */
//	@SuppressWarnings("restriction")
	public static boolean imgDecode(String imgCode, String imgName) {
		OutputStream out = null;
		boolean result = false;
		try {
			Base64.Decoder decoder = Base64.getDecoder();
			String[] baseStrs = imgCode.split(",");
			byte[] dataBuf = decoder.decode(baseStrs[1]); // 解码为字节数组			

			// 字节流转存文件
			out = new FileOutputStream(imgName);
			out.write(dataBuf);
			out.flush();
//			out.close();
			result = true;
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			close(out);
		}
		return result;
	}

	public static String urlEncode(String url) {
		try {
			return URLEncoder.encode(url, Const.UTF8);
		} catch (Exception e) {
			e.printStackTrace();
			return url;
		}
	}

	public static String urlDecode(String url) {
		try {
			return URLDecoder.decode(url, Const.UTF8);
		} catch (Exception e) {
			e.printStackTrace();
			return url;
		}
	}

	/**
	 * 生成通用唯一识别码
	 */
	public static String uuid() {
		return UUID.randomUUID().toString();
	}

	/**
	 * 生成通用唯一识别码
	 */
	/*private static String uuid8(String uuid) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < 8; i++) {
			String str = uuid.substring(i * 4, i * 4 + 4);
			int x = Integer.parseInt(str, 16);
			// 如果是 chars62，则是x%62
			buf.append(CHARS64.charAt(x % 36));
		}
		return buf.toString();
	}*/

	/**
	 * 根据时间戳生成8位唯一识别码，可支持6.2万并发
	 */
	public static String uuid8() {
		// return uuid8(uuid().replace("-", ""));
		//return enBase64(System.nanoTime() / 100);
		char c = CHARS64.charAt(rand.nextInt(62)); // 为避免使用加号“+”
		// 截至2081年08月05日前有效
		return encode(System.currentTimeMillis(), 62) + c;
	}

	/**
	 * 把十进制的数字转换为64进制编码
	 */
	public static String enBase64(long num) {
		return encode(num, 64);
	}

	public static String enBase62(long num) {
		return encode(num, 62);
	}

	/**
	 * 把十进制的数字转换为N进制编码
	 */
	private static String encode(long num, int n) {
		if (num == 0)
			return "A";
		StringBuffer buf = new StringBuffer();
		char c;
		while (num > 0) {
			c = CHARS64.charAt(new Long(num % n).intValue());
			buf.append(c);
			num = num / n;
		}
		return buf.reverse().toString();
	}

	/**
	 * 把64进制字符解码为十进制数字
	 */
	public static long deBase64(String str) {
		return decode(str, 64);
	}

	/**
	 * 把62进制字符解码为十进制数字
	 */
	public static long deBase62(String str) {
		return decode(str, 62);
	}

	/**
	 * 把N进制编码转换为十进制数字
	 */
	private static long decode(String str, int n) {
		// 基数
		long result = 0, radix = 1;
		char c;
		for (int i = 0; i < str.length(); i++) {
			c = str.charAt(str.length() - 1 - i);
			result += CHARS64.indexOf(c) * radix;
			radix = radix * n;
		}
		return result;
	}

	/**
	 * 数字串进行编码，保留最后n位，如：身份证最后1位X，n=0时全部编码
	 */
	public static String enBase62(String str, int n) {
		try {
			long num = Long.valueOf(str.substring(0, str.length() - n));
			return enBase62(num) + str.substring(str.length() - n);
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * 解码编码的数字串，保留最后n位，如：身份证最后1位X，n=0时全部解码
	 */
	public static String deBase62(String str, int n) {
		String num = str.substring(0, str.length() - n);
		return deBase62(num) + str.substring(str.length()- n);
	}

	/**
	 * 关闭对象
	 */
	private static void close(Closeable... closeables) {
		if (closeables != null && closeables.length > 0) {
			for (Closeable closeable : closeables) {
				if (closeable == null)
					continue;
				try {
					closeable.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}

}
