package cn.ps1.aolai.service;

import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 第三方对接（微信相关的登录认证）安全服务
 * 
 * @author Aolai
 * @since  1.7 $Date: 2020.6.17
 * @version 1.0
 * 
 */

@Service
public class ThirdService {

	private static Logger LOG = LoggerFactory.getLogger(ThirdService.class);
	private static String AppKey = null;
	private static String Md5Key = null;

	@Autowired
	private AolaiService aolaiSvc;
	@Autowired
	private HttpService httpSvc;
	@Autowired
	private RedisService redisSvc;
	@Autowired
	private UtilsService utilsSvc;

	private boolean checkAppKey(Object k) {
		if (Md5Key == null) {
			AppKey = utilsSvc.getConf("app.key");
			Md5Key = DigestUtils.md5Hex(AppKey);
		}
		return AppKey.equals(k) || Md5Key.equals(k);
	}

	/**
	 * 支持第三方（如微信）快速授权登录
	 * 
	 * @return boolean 成功与否
	 */
	public boolean thirdLogin(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> cookies) {
		// 微信自动登录
		if (cookies.containsKey("userId") && cookies.containsKey("bindId")) {
			/*Map<String, String> map = wxLogin(req, rsp, cookies);
			if (map != null) {
				// 生成Token，返回前端并写到Cookie中
				map = utilsSvc.newToken(req, rsp, map.get("binduid"));
				redisSvc.setToken(map);
				return true;
			}*/
		}
		return false;
	}

	/**
	 * 校验11位（或10位、8位）通行证 k码（相当于appId）
	 * <p>
	 * 接口调用：app/s/getCertKey，参数：{ticket="LCEvNBmyjO", k="www.ps1.cn"}
	 * <p>
	 * 或加密参数：{jsonstr:{..., ticket="LCEvNBmyjO"}, k="FDIUwGXHRpSZqATK"}
	 */
	public boolean isSecretKey(HttpServletRequest req) {

		Object k = req.getParameter("k");
		if (k == null) // 是否携带了 k参数
			return false;

		// 来自前台请求的加密字符串
		String jsonStr = req.getParameter(Const.JSON_STR); // json对象、或加密串
		// 获取 ticket时：k="www.ps1.cn" 或 MD5("www.ps1.cn")
		if (checkAppKey(k)) { // 本平台的接口调用时：k="www.ps1.cn"

			if (jsonStr == null)
				return true; // 可直接处理非 jsonStr格式参数

			// 未加密的 jsonStr参数（www.ps1.cn）
			return setAttr(req, utilsSvc.json2Map(jsonStr), k);
		}

		// 第三方应用跨平台的接口调用：
		// 根据动态的 k="FDIUwGXHRpSZqATK"（即：certKey）

		// 获取缓存的 ticket="LCEvNBmyjO"
		String ticket = redisSvc.get(Const.RDS_CERT + k);
		if (ticket == null)
			return false; // 失败

		// 根据获取的 ticket="LCEvNBmyjO"、携带的动态 k="FDIUwGXHRpSZqATK"
		// 还原通行证、并解密 jsonStr数据
		jsonStr = Digest.decrypt(jsonStr, ticket + k);
		Map<String, Object> map = utilsSvc.json2Map(jsonStr);

		// 注意：解密的对象也必须携带了 ticket="LCEvNBmyjO"

		// 双检第一次，缓存 ticket 与携带的 ticket校验
		if (!ticket.equals(map.get("ticket")))
			return false; // 失败

		// 双检第二次，根据 ticket 获取缓存的 certKey、与携带的 certKey校验
		String certId = redisSvc.get(Const.RDS_APPID + ticket);
		if (certId != null && k.equals(certId)) {
			return setAttr(req, map, k); // 成功
		}

		return false; // 失败
	}
	
	private boolean setAttr(HttpServletRequest req, Object obj,
			Object certId) {
		req.setAttribute("json", obj);
		//req.setAttribute("certId", certId);
		//req.setAttribute("userId", userId);
		return true;
	}

	/**
	 * 对用户授权的功能进行鉴权，如果是其他非法登录，拒绝访问
	 */
	public boolean authAccess(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> cookies) throws Exception {
		// 如果前端对userId编码处理过，这里需要解码
		String cert = cookies.get("certId"); // 默认11位通行证
		String userId = Digest.decrypt(cookies.get("userId"), cert);
		// 根据用户 duty权限鉴权

		Map<String, String> user = redisSvc.getUserInfo(userId);
		if (user == null) {
			redisSvc.clearToken(cookies);
			return invlidToken(rsp); // redis缓存失效
		}
		
		// 根据APP编码、岗位，验证（服务请求的地址）接口的访问权限：{M:"D01"}
		Map<String, Object> tmp = utilsSvc.json2Map(user.get("userDuty"));
		// 每个应用、每个账套、访问权限功能列表
		final String appKey = utilsSvc.getConf("app.code");
//		if (!tmp.containsKey(appKey)) // 应用编码：A\M\D
//			return invlidParams(rsp); // 鉴权失败
		
		// 增加两种处理方法：1.如果app.code为空则用户不受权限控制，2.app.code可以是多个应用
		// 修改后的鉴权方式：只要有一个应用权限符合即可
		if (appKey.length() > 0) {
			String[] keys = appKey.split(";");
			boolean isValid = false;
			for (String key : keys) {
				if (tmp.containsKey(key)) { // 应用编码：A\M\D
					isValid = true; // 验证通过
					break;
				}
			}
			if (!isValid) // 未通过
				return invlidParams(rsp); // 鉴权失败
		}
		/** 获取请求参数，并解码处理 */
		//tmp = utilsSvc.decryptParams(req, userId, cert);
		tmp = decryptParams(req, user, cert);

		String uri = utilsSvc.getRequestURI(req); // 如：wsTest1
		LOG.info("-> authAccess..." + uri + tmp.toString());
		String args = utilsSvc.getValid(uri);
		if (args.length() > 0 && !utilsSvc.availParams(tmp, args.split(";")))
			return invlidParams(rsp);

		// TODO: 这里可以根据需求进一步优化处理
//		Map<String, String> map = redisSvc.hmget(Const.RDS_ROLE
//				+ tmp.get(appKey));
//		if (map == null || map.containsKey(uri)) {
//			return false;
//		}

		return true;
	}

	/**
	 * 获取HTTP请求从前端传递来的参数并解密
	 * 
	 * @param req 请求参数
	 * @param user 当前登录用户的userInfo
	 * @param certId 默认为11位或10位通行证
	 */
	private Map<String, Object> decryptParams(HttpServletRequest req,
			Map<String, String> user, String cert) {
		req.setAttribute("user", user);
		req.setAttribute("userId", user.get("userId"));
		req.setAttribute("certId", cert);
		// 从前端传递来的参数
		String jsonStr = req.getParameter(Const.JSON_STR);
		Map<String, Object> params = new HashMap<>();
		if (jsonStr != null) {
			jsonStr = Digest.decrypt(jsonStr, cert);
			params = utilsSvc.json2Map(jsonStr);
		}
		LOG.info("-> jsonStr..." + params.toString());
		req.setAttribute("json", params);
		return params;
	}

	/**
	 * 验证无效的token
	 */
	private boolean invalidResult(HttpServletResponse rsp, String status)
			throws Exception {
		rsp.setContentType("application/json;charset=UTF-8");
		rsp.setCharacterEncoding("UTF-8");
		PrintWriter out = rsp.getWriter();
		out.write(utilsSvc.obj2Str(utilsSvc.result(status))); // "3"
		out.flush();
		out.close();
		return false;
	}

	/**
	 * 验证无效的token
	 */
	public boolean invlidToken(HttpServletResponse rsp) throws Exception {
		return invalidResult(rsp, "3");
	}

	/**
	 * 验证无效的参数
	 */
	public boolean invlidParams(HttpServletResponse rsp) throws Exception {
		return invalidResult(rsp, "2");
	}

	/**
	 * 根据每个终端用户唯一的 ticket获取各自的 certKey通行证
	 */
	public Object getCertKey(HttpServletRequest req) {
		String ticket = req.getParameter("ticket");
		if (!utilsSvc.isEmpty(ticket)
				&& "true".equals(utilsSvc.getConf("api.open"))) {
			Map<String, Object> where = new HashMap<>();
			Map<String, String> result = new HashMap<>();
			where.put("certState", "1");
			where.put("certId", ticket);
			// 获取通行证
			result = aolaiSvc.findOne("CERT", where, "getCertInfo");
			if (result.containsKey("status")) {
				int ms = getMsec(req.getParameter("msec"));
				// 变换后的16位通行证
				String certKey = Digest.randStr(result.get("certKey"));
				redisSvc.set(Const.RDS_APPID + ticket, certKey, ms); // 缓存
				redisSvc.set(Const.RDS_CERT + certKey, ticket, ms); // 缓存
				result.put("certKey", certKey);
				// 返回参数
				return utilsSvc.success(result);
			}
		}
		return utilsSvc.result("2"); // 无效参数，或暂不开放API
	}

	/**
	 * 前端传递的一个缓存时效，默认缓存10小时，测试可改 msec值
	 * 
	 * @param msec
	 */
	private int getMsec(String msec) {
		try {
			if (utilsSvc.isInteger(msec))
				return Integer.parseInt(msec); // 前端传递的一个缓存时效
		} catch (Exception e) {
			LOG.error("-> getMsec..." + e.getMessage());
		}
		return Const.TEN_HH; // 默认缓存10小时
	}

}
