package cn.ps1.aolai.service;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.utils.Const;
import redis.clients.jedis.ShardedJedis;
import redis.clients.jedis.ShardedJedisPool;

/**
 * Redi的增删改查及Token基本业务操作
 * 
 * @author Aolai
 * @since 2017.6.17
 * @version 1.0
 * 
 */

@Service
public class RedisService {

	private static Logger LOG = LoggerFactory.getLogger(RedisService.class);
	private static final boolean oneUser = true; // 仅限单一用户登录

	@Autowired
	private ShardedJedisPool jedisPool;

	/**
	 * 清除token信息：{userId,token,certId}
	 */
	public boolean clearToken(Map<String, String> params) {
		if (!checkToken(params))
			return false;
		LOG.debug("-> clearToken..." + params);

		String key = oneUser ? params.get("userId") : params.get("token");
		del(Const.RDS_TOKEN + key);

		/** 兼容旧版格式：清除旧版的userid，否则新版退出后再登录旧版时无法登录 */
		del(Const.RDS_USER + params.get("userId"));
		return true;
	}

	/**
	 * 设置新token或保持已有token持续有效
	 */
	public void setToken(Map<String, String> params) {
		// 仅限单一用户登录，或支持多用户同时登录
		// 为了兼容，cookie的主键全为小写
		String key = oneUser ? params.get("userId") : params.get("token");
		hmset(Const.RDS_TOKEN + key, params, Const.TWO_HH);
	}

	/**
	 * 验证token是否有效：单用户不能同时缓存多个token，多用户可以有多个token缓存同一个userid
	 */
	public boolean checkToken(Map<String, String> cookie) {
		// 为了兼容，cookie的主键全为小写
		String token = cookie.get("token");
		String userid = cookie.get("userId");
		String certid = cookie.get("certId");
		if (token != null && userid != null && certid != null) {
			String uid = Const.RDS_TOKEN + (oneUser ? userid : token);
			String str = oneUser ? token : userid;
			String key = oneUser ? "token" : "userId";
			
			// 根据登录用户userid从缓存中读取并验证缓存信息并进行校验
			Map<String, String> map = hmget(uid);
			if (map != null && str.equals(map.get(key))
					&& certid.equals(map.get("certId"))) {
				
				return expire(uid); // 验证成功后自动延长有效期
				//return true;
			}
		}
		return false;
	}
	
	/**
	 * 缓存当前用户信息
	 * @param user
	 */
	public void setUserInfo(Map<String, String> user) {
		String uid = Const.RDS_USER + user.get("userId");
		hmset(uid, user, Const.TWO_HH);
	}
	
	/**
	 * 获取当前用户信息
	 * @param userId
	 * @return Map 当前用户信息
	 */
	public Map<String, String> getUserInfo(Object userId) {
		String uid = Const.RDS_USER + userId;
		if (expire(uid))
			return hmget(uid);
		return null;
	}
	
	/**
	 * 设置单个键值，永久有效
	 * @param key 主键
	 * @param val 值
	 */
	public String set(String key, String val) {
		return set(key, val, 0); // 返回OK：成功
	}

	/**
	 * 设置单个键值，并设置有效时间
	 * @param key 主键
	 * @param val 值
	 * @param sec 有效期（秒）
	 */
	public String set(String key, String val, int sec) {
		LOG.debug("-> set..." + key + "=" + val);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
//			result = jedis.set(key, val);
//			if (sec > 0)
//				expire(key, sec);
			result = jedis.setex(key, sec, val); // 返回OK：成功
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result; // 返回OK：成功
	}

	/**
	 * 设置map对象，永久有效
	 * @param key 主键
	 * @param map 对象
	 */
	public String hmset(String key, Map<String, String> map) {
		return hmset(key, map, 0);
	}

	/**
	 * 设置map对象
	 * @param key 主键
	 * @param map 对象
	 * @param sec 有效期（秒）
	 * @return
	 */
	public String hmset(String key, Map<String, String> map, int sec) {
		LOG.debug("-> hmset..." + key + "=" + map);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			// jedis.del(key);
			result = jedis.hmset(key, map);
			if (sec > 0)
				expire(key, sec);
		} catch (Exception e) {
			LOG.error("->hmset..." + e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 获取map对象
	 */
	public Map<String, String> hmget(String key) {
		LOG.debug("-> hmget..." + key);
		Map<String, String> result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			result = jedis.hgetAll(key);
			if (result != null && result.isEmpty())
				result = null;
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 获取map对象单个域的键值
	 */
	public String hget(String key, String field) {
		LOG.debug("-> hget..." + key);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			result = jedis.hget(key, field);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 设置map对象单个域的键值，永久有效
	 */
	public boolean hset(String key, String field, String val) {
		return hset(key, field, val, 0);
	}

	/**
	 * 设置map对象单个域的键值
	 */
	public boolean hset(String key, String field, String val, int sec) {
		LOG.debug("-> hset..." + key);
		boolean result = false;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			jedis.hset(key, field, val); // 新建返回1，覆盖返回0
			if (sec > 0)
				expire(key, sec);
			result = true;
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 获取单个键值
	 */
	public String get(String key) {
		LOG.debug("-> get..." + key);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return null;
		try {
			result = jedis.get(key);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 根据key删除键值对数据
	 */
	public boolean del(String key) {
		boolean result = false;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			jedis.del(key);
			result = true;
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 判断key是否存在
	 */
	public boolean exists(String key) {
		Boolean result = false;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			result = jedis.exists(key);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 设置key默认有效期，默认1小时有效
	 */
	public boolean expire(String key) {
		return expire(key, Const.TWO_HH);
	}

	/**
	 * 设置key的有效期，或延续时间
	 */
	private boolean expire(String key, int sec) {
		Long result = 0L;
		ShardedJedis jedis = jedisPool.getResource();
		if (sec == 0 || jedis == null)
			return false;
		try {
			// 返回1-成功，0-失败
			result = jedis.expire(key, sec);
		} catch (Exception e) {
			LOG.error(e.getMessage());
			e.printStackTrace();
		} finally {
			jedis.close();
		}
		return result > 0;
	}

	/**
	 * 对短时间内尝试登陆的ip次数进行限制
	 * @param ip 客户ip地址
	 * @param num 一定时间内及次数限制
	 * @return boolean
	 */
	public boolean isBlocked(String ip, int[] num) {
		String rds_ip = Const.RDS_DENY + ip;
		Map<String, String> addr = hmget(rds_ip);
		String time = String.valueOf(new Date().getTime());
		int loop = num.length/2;
		if (addr == null) {
			addr = new HashMap<>();
			addr.put("locked", "0"); // 初次访问，未锁
		} else if ("1".equals(addr.get("locked"))) { // 已锁定
			return true;
		} else {
			String[] arr = addr.get("time").split(",");
			int len = arr.length, i = 0;
			while (i < len) {
				long diff = new Date().getTime() - Long.parseLong(arr[i]);
				for (int n = 0; n < loop; n++) {
					if (diff < num[0]) { // 指定时间之内（1分钟、30分钟）
						if (len - i < num[n + loop]) // 小于N次
							break;
						// N次及以上（10次、50次）
						addr.put("locked", "1"); // 锁定
						hmset(rds_ip, addr, loop - 1);
						return true;
					}
				}
				i ++;
			}
			while (i < len) {
				time = arr[i] + "," + time;
				i++;
			}
		}
		addr.put("time", time);
		hmset(rds_ip, addr, loop - 1);
		return false;
	}
	
	/**
	 * 阻止指定ip访问
	 * @param ip
	 * @return
	 */
	public boolean isBlocked(String ip) {
		int[] num = {Const.ONE_MM, 3*Const.TEN_MM, 10, 50};
		return isBlocked(ip, num);
	}

}
