package cn.ps1.aolai.service;

import java.io.Closeable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 全局通用的公共方法Utils类
 * 
 * @author Aolai
 * @since 2017年6月17日
 * @verdion 1.0
 */

@Service
public class UtilsService {

	private static Logger LOG = LoggerFactory.getLogger(UtilsService.class);
	private static ObjectMapper jsonMapper = new ObjectMapper();

	/**
	 * 构造函数（为了测试用）
	 */
	public UtilsService() {
	}

	/**
	 * 系统配置参数
	 */
	public String getConf(String key) {
		return ConfUtil.getConf(key); // config.properties
	}

	public String getArgs(String key) {
		return ConfUtil.getParam(key); // params.properties
	}

	public String getValid(String key) {
		return ConfUtil.getValid(key); // valid.properties
	}
	
	/**
	 * 获取HTTP请求从前端传递来的参数并解密
	 */
	public Map<String, Object> decryptParams(HttpServletRequest req,
			String userId, String certId) {
		// 从前端传递来的参数
		String jsonStr = req.getParameter(Const.JSON_STR);
		Map<String, Object> params = new HashMap<>();
		if (jsonStr != null)
			params = json2Map(Digest.decrypt(jsonStr, certId));
		
		LOG.info("-> jsonStr..." + params.toString());
		req.setAttribute("json", params);
		req.setAttribute("certId", certId);
		req.setAttribute("userId", userId);
		return params;
	}

	/**
	 * 获取HTTP请求从前端传递来的参数
	 */
	public Map<String, Object> jsonParams(HttpServletRequest req) {
		// 从前端传递来的参数
		Map<String, Object> params = obj2Map(req.getAttribute("json"));
		if (params.isEmpty()) {
			Enumeration<String> map = req.getParameterNames();
			while (map.hasMoreElements()) {
				String key = map.nextElement();
				String val = req.getParameter(key);
				if (val.length() != 0)
					params.put(key, val);
			}
		}
		// 先以传递的参数为主，再获取当前客户端参数
		if (!params.containsKey("i18n"))
			params.put("i18n", getLocale(req));
		if (params.containsKey("baseid"))
			params.put("base", getConf("base.name") + params.get("baseid"));	
		return params;
	}
	
	/**
	 * 获取HTTP请求的传递参数，私钥解码json参数后的对象
	 */
	/*public Map<String, Object> certParams(HttpServletRequest req) {
		// 从前端传递来的参数
		Map<String, Object> params = getParams(req);
		// 先以传递的参数为主，再获取当前客户端参数
		if (!params.containsKey("i18n"))
			params.put("i18n", getLocale(req));
		if (params.containsKey("baseid"))
			params.put("base", getConf("base.name") + params.get("baseid"));
		return params;
	}*/

	/**
	 * 检查请求（输入）参数无效
	 * @param map
	 * @param keys
	 * @return
	 */
	public <T> boolean availParams(Map<String, T> map, String[] keys) {
		if (keys.length == 0)
			return true;
		if (map.isEmpty())
			return false;
		for (int i = 0; i < keys.length; i++) {
			if (!map.containsKey(keys[i]))
				return false;
		}
		return true;
	}

	/**
	 * 把数串逐级累计（科目层级转换）
	 * 
	 * @param numStr 数字的字符串
	 * @return Array 累计数组
	 */
	public int[] num2Arr(String numStr) {
		String[] oldArr = numStr.split("");
		int[] newArr = new int[oldArr.length];
		int n = 0;
		for (int i = 0; i < oldArr.length; i++) {
			n += Integer.parseInt(oldArr[i]);
			newArr[i] = n;
		}
		return newArr;
	}

	public String arr2Str(String[] arr) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < arr.length; i++) {
			sb.append(arr[i]);
		}
		return sb.toString();
	}

	/**
	 * 对象转换为Map对象
	 */
	@SuppressWarnings("unchecked")
	public <T> Map<String, T> obj2Map(Object data) {
		if (data == null)
			return new HashMap<>();
		return (Map<String, T>) data;
	}

	/**
	 * 对象转换为List对象
	 */
	@SuppressWarnings("unchecked")
	public <T> List<Map<String, T>> obj2List(Object data) {
		if (data == null)
			return new ArrayList<>();
		return (List<Map<String, T>>) data;
	}
	
	/**
	 * Map对象数据互转
	 */
	public Map<String, String> obj2Map(Map<String, Object> obj) {
		Map<String, String> map = new HashMap<>();
		for (Map.Entry<String, Object> e : obj.entrySet()) {
			map.put(e.getKey(), String.valueOf(e.getValue()));
		}
		return map;
	}
	
	/**
	 * Map对象数据互转
	 */
	public Map<String, Object> map2Obj(Map<String, String> obj) {
		Map<String, Object> map = new HashMap<>();
		for (Map.Entry<String, String> e : obj.entrySet()) {
			map.put(e.getKey(), e.getValue());
		}
		return map;
	}

	/**
	 * 把对象（Map、List）转为jsonStr字符串
	 * @param obj Map或List对象
	 * @return jsonStr 字符串
	 */
	public String obj2Str(Object obj) {
		String jsonStr = null;
		try {
			jsonStr = jsonMapper.writeValueAsString(obj);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		}
		return jsonStr;
	}

	/**
	 * 从Json字符串中找出Map对象节点
	 */
	public Map<String, Object> node2Obj(String jsonStr, String path) {
		Map<String, Object> map = new HashMap<>();
		try {
			if (!isEmpty(jsonStr)) {
				JsonNode node = jsonMapper.readTree(jsonStr);
				map = node2Obj(path == null ? node : node.path(path));
			}
		} catch (Exception e) {
			LOG.error("-> node2Obj...");
			e.printStackTrace();
		}
		return map;
	}
	
	/**
	 * Json对象字符串转换为Map对象
	 * 
	 * @param jsonStr
	 * @return
	 */
	public Map<String, Object> node2Obj(String jsonStr) {
		return node2Obj(jsonStr, null);
	}

	/**
	 * Json对象转换为Map对象
	 * 
	 * @param jsonNode
	 * @return
	 */
	public Map<String, Object> node2Obj(JsonNode jsonNode) {
		Map<String, Object> map = new HashMap<>();
		if (jsonNode.isValueNode()) {
			map.put("", jsonNode.asText());
		} else if (jsonNode.isArray()) {
			List<Object> list = new ArrayList<>();
			Iterator<JsonNode> it = jsonNode.iterator();
			while (it.hasNext()) {
				Map<String, Object> child = node2Obj(it.next());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					list.add(child.get(""));
				} else {
					list.add(child);
				}
			}
			map.put("", list);
		} else {
			Iterator<Map.Entry<String, JsonNode>> it = jsonNode.fields();
			while (it.hasNext()) {
				Map.Entry<String, JsonNode> entity = it.next();
				Map<String, Object> child = node2Obj(entity.getValue());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					map.put(entity.getKey(), child.get(""));
				} else {
					map.put(entity.getKey(), child);
				}
			}
		}
		return map;
	}

	/**
	 * 把jsonStr转换成为Map对象
	 * @param jsonStr
	 * @return Map
	 */
	public <T> Map<String, T> json2Map(String str) {
		Map<String, T> map = new HashMap<>();
		if (isEmpty(str))
			return map;
		try {
			map = jsonMapper.readValue(str,
					new TypeReference<Map<String, T>>() {
					});
		} catch (Exception e) {
			// 例如：HTTP/1.1 500 Internal Server Error
			LOG.error("-> json2Map...");
			e.printStackTrace();
		}
		if (str.length() > 200)
			LOG.debug("-> json2Map..." + str.substring(0, 200) + "...");
		else
			LOG.debug("-> json2Map..." + str);
		return map;
	}

	/**
	 * 把jsonStr字符串转换成为list对象
	 * @param jsonStr
	 * @return List
	 */
	public <T> List<Map<String, T>> json2List(String str) {
		List<Map<String, T>> list = new ArrayList<>();
		if (isEmpty(str))
			return list;
		try {
			list = jsonMapper.readValue(str,
					new TypeReference<List<Map<String, T>>>() {
					});
		} catch (Exception e) {
			LOG.error("-> json2List..." + e.getMessage());
		}
		if (str.length() > 200)
			LOG.debug("-> json2List..." + str.substring(0, 200) + "...");
		else
			LOG.debug("-> json2List..." + str);
		return list;
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public Map<String, Map<String, String>> list2Map(
			List<Map<String, String>> list, String key) {
		Map<String, Map<String, String>> map = new HashMap<>();
		if (list != null && key != null) {
			for (int i = 0; i < list.size(); i++) {
				Map<String, String> item = list.get(i);
				map.put(item.get(key), item);
			}
		}
		return map;
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public Map<String, String> list2Map(List<Map<String, String>> list,
			String key, String val) {
		Map<String, String> map = new HashMap<>();
		if (list != null && key != null && val != null) {
			for (int i = 0; i < list.size(); i++) {
				Map<String, String> item = list.get(i);
				map.put(item.get(key), item.get(val));
			}
		}
		return map;
	}

	/**
	 * 返回状态结果
	 * @param status 1-成功，0-失败、2-参数错误、3-Token失效
	 * @return
	 */
	public Map<String, String> result(String status) {
		Map<String, String> result = new HashMap<>();
		result.put("status", status);
		return result;
	}

	/**
	 * 返回状态结果：false-失败、true-成功
	 */
	public Map<String, String> result(boolean status) {
		return result(status ? "1" : "0");
	}

	/**
	 * 返回状态结果
	 * @param status 1-成功，0-失败、2-参数错误、3-Token失效
	 * @param info 返回信息或数据
	 */
	public Map<String, String> result(String status, String info) {
		Map<String, String> result = result(status);
		if (info != null)
			result.put("info", info);
		return result;
	}

	/**
	 * 返回失败消息
	 */
	public Map<String, String> failed(String info) {
		Map<String, String> result = result("0");
		result.put("info", info);
		return result;
	}

	/**
	 * 返回成功
	 */
	public Map<String, String> success() {
		return result("1");
	}
	
	public Map<String, String> success(String info) {
		return result("1", info);
	}

	/**
	 * 返回数据对象
	 */
	public Map<String, Object> success(Object obj) {
		Map<String, Object> result = new HashMap<>();
		result.put("status", "1");
		result.put("info", obj);
		return result;
	}

	public Map<String, String> success(Object obj, HttpServletRequest req) {
		// 返回加密信息
		String cert = String.valueOf(req.getAttribute("certId"));
		String info = Digest.decrypt(obj2Str(obj), cert);
		return result("1", info);
	}

	/**
	 * 获取request参数属性值
	 * 
	 * @param req HTTP请求
	 * @param key 参数名
	 * @return String
	 */
	public Object getAttr(HttpServletRequest req, String key) {
		return req.getAttribute(key);
	}

	/**
	 * // 新建用户Token并写入Cookies，并返回前端
	 * @param req 来自用户的请求
	 * @param rsp 响应
	 * @param userId 当前用户
	 * @return Map 返回Token对象
	 */
	public Map<String, String> newToken(HttpServletRequest req,
			HttpServletResponse rsp, String userId, String ip) {
		String certId = Digest.randKey(); // 11位证书Id
		// 为了兼容，token的主键全为小写
		Map<String, String> map = new HashMap<>();
		map.put("userId", Digest.encrypt(userId, certId)); // 编码处理
		map.put("token", Digest.uuid8());
		map.put("certId", certId);
		map.put("ipAddr", ip); // 缓存登录ip
		
		map.put("uuid", map.get("token")); // 为了兼容旧版三资管理
		
		LOG.info("-> newToken..." + map);		
		// 设置Cookie（如果前端处理此处可以忽略！！）		
		setCookies(req, rsp, map);
		
		return map;
	}

	/**
	 * 找出List列表中匹配的指定字符串
	 */
	public Map<String, String> findIn(List<Map<String, String>> list,
			String key, String val) {
		for (int i = 0; i < list.size(); i++) {
			if (val.equals(list.get(i).get(key)))
				return list.get(i);
		}
		return null;
	}

	/**
	 * 找出List列表中是否存在指定字符串
	 */
	public boolean findIn(List<String> list, String str) {
		for (int i = 0; i < list.size(); i++) {
			if (list.get(i).equals(str))
				return true;
		}
		return false;
	}

	/**
	 * 是否存在
	 */
	public boolean findIn(String[] arr, String str) {
		for (int i = 0; i < arr.length; i++) {
			if (arr[i].equals(str))
				return true;
		}
		return false;
	}

	/**
	 * 找出字符串中匹配的字符或数字
	 */
	public int matcher(String str, String regex) {
		Pattern pat = Pattern.compile(regex); // "\\d+"
		Matcher mat = pat.matcher(str);
		if (mat.find())
			return Integer.valueOf(mat.group());
		return 0;
	}
	
	/**
	 * 字符串中是否有匹配的字符或数字
	 * @return
	 */
	public boolean isMatch(String str, String regex) {
		if (str == null || regex == null) return false;
		Pattern pat = Pattern.compile(regex);
		return pat.matcher(str).matches();
	}

	/**
	 * 判断是否为整数、负数
	 */
	public boolean isInteger(String str) { // 整数
		// ^[-\\+]?[\\d]+$匹配前面的子表达式一次或多次
		return isMatch(str, "^-?\\d+$");
	}

	/**
	 * 判断是否为整数、小数、负数
	 */
	public boolean isNumeric(String str) { // 浮点数
		return isMatch(str, "^(-?\\d+)(\\.\\d+)?$");
	}

	/**
	 * 判断是否为时间格式
	 */
	public boolean isDatetime(String str) {
		try {
			new SimpleDateFormat(Const.DTF).parse(str);
			return true;
		} catch (ParseException e) {
			return false;
		}
	}

	/**
	 * 计算两个日期之间的差数多少秒：sTime开始时间，eTime结束时间
	 */
	public long timeDiff(Date begin, Date end) {
		long sec = 0;
		Calendar cal = Calendar.getInstance();
		cal.setTime(end);
		sec = cal.getTimeInMillis();
		cal.setTime(begin);
		sec -= cal.getTimeInMillis();
		return sec;
	}

	/**
	 * 计算两个日期之间的差数多少秒：sDate开始时间，eDate结束时间
	 */
	public long timeDiff(String begin, String end) {
		long sec = 0;
		try {
			// yyyy-MM-dd HH:mm:ss
			SimpleDateFormat sdf = new SimpleDateFormat(Const.DTF);
			if (end == null)
				end = today(Const.DTF);
			if (begin != null)
				return timeDiff(sdf.parse(begin), sdf.parse(end));
		} catch (ParseException e) {
			LOG.error("-> timeDiff...");
			e.printStackTrace();
		}
		return sec;
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public String today(String pattern) {
		return today(pattern, Calendar.getInstance());
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	private String today(String pattern, Calendar cal) {
		SimpleDateFormat sdf = new SimpleDateFormat(pattern);
		return sdf.format(cal.getTime());
	}

	public String nextYear(String pattern, int y) {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.YEAR, y);
		return today(pattern, cal);
	}

	/**
	 * 获得下个月期间
	 */
	public String nextMonth(String month, int m) {
		SimpleDateFormat sdf = new SimpleDateFormat(Const.MON);
		Calendar cal = Calendar.getInstance();
		try {
			cal.setTime(sdf.parse(month)); // 设置当前月份
		} catch (ParseException e) {
			LOG.error("-> nextMonth...");
			e.printStackTrace();
		}
		cal.add(Calendar.MONTH, m);
		return sdf.format(cal.getTime());
	}

	/**
	 * 获得下个月期间
	 */
	public String nextMonth(String month) {
		return nextMonth(month, 1);
	}

	/**
	 * 获得上个月期间
	 */
	public String prevMonth(String month, int m) {
		return nextMonth(month, -m);
	}

	/**
	 * 获得上个月期间
	 */
	public String prevMonth(String month) {
		return prevMonth(month, 1);
	}

	/**
	 * 获取host、Port
	 * 
	 * @param uri
	 * @return
	 */
	public String[] getHost(String uri) {
		int idx = uri.indexOf("://");
		uri = idx > 0 ? uri.substring(idx + 3) : uri;
		idx = uri.indexOf('/');
		uri = idx > 0 ? uri.substring(0, idx) : uri;
		return uri.split(":");
	}

	/**
	 * 获取请求参数
	 */
	public String getRequestURI(HttpServletRequest req) {
		LOG.info("-> HttpRequest..." + req.getHeader("Referer"));

		// String path = req.getContextPath(); // 如：path=“/webapp”
		String uri = req.getRequestURI(); // uri=“/webapp/ws/wsTest1”
		LOG.info("-> HttpRequest..." + uri);

		// uri = uri.substring(path.length()); // uri=“/ws/wsTest1”
		return uri.substring(uri.lastIndexOf('/') + 1); // 如：uri=“wsTest1”
	}

	/**
	 * 获取域名（放置cookie使用）
	 */
	private String getDomain(HttpServletRequest req) {
		// final String core = getConf("app.core"); //
		// http://xxx.ps1.cn:8080/xxx
		final String ref = req.getHeader("Referer"); // 泉州同时包含域名、IP登录，需要动态获取。
		LOG.info("-> host..." + ref);
		String host = getHost(ref)[0];
		LOG.debug("-> getDomain..." + host);
		String[] arr = host.split("\\.");
		return arr.length == 3 ? arr[1] + "." + arr[2] : host;
	}

	/**
	 * 设置cookies
	 */
	public void setCookies(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> map) {
		if (map == null)
			return;
		String domain = getDomain(req);
		for (Map.Entry<String, String> e : map.entrySet()) {
			// Cookie c =new Cookie(key, e.getValue());
			String s = Digest.encodeURIComponent(e.getValue());
			Cookie c = new Cookie(e.getKey(), s);
			c.setDomain(domain);
			c.setPath("/");
			// c.setHttpOnly(true); // 安全保护
			rsp.addCookie(c);
		}
	}

	/**
	 * 获取所有Cookie键值对
	 */
	public Map<String, String> getCookies(HttpServletRequest req) {
		Map<String, String> map = new HashMap<>();
		Cookie[] cookies = req.getCookies();
		if (cookies == null)
			return map;
		for (Cookie c : cookies) {
			// map.put(c.getName(), c.getValue());
			String s = Digest.decodeURIComponent(c.getValue());
			map.put(c.getName(), s);
		}
		return map;
	}

	/**
	 * 获取指定的Cookie（暂未用）
	 */
	public String getCookie(HttpServletRequest req, String key) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null)
			for (Cookie c : cookies) {
				if (c.getName().equals(key)) {
					// return c.getValue();
					return Digest.decodeURIComponent(c.getValue());
				}
			}
		return null;
	}

	/**
	 * 刪除指定的Cookie（暂未用）
	 */
	public void delCookie(HttpServletRequest req, HttpServletResponse rsp,
			String key) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null) {
			for (Cookie c : cookies) {
				if (c.getName().equals(key)) {
					c.setDomain(getDomain(req));
					c.setPath("/");
					c.setMaxAge(0); // 立即销毁cookie
					rsp.addCookie(c);
					break;
				}
			}
		}
	}

	/**
	 * 清除全部Cookie的键值
	 */
	public void clearCookies(HttpServletRequest req, HttpServletResponse rsp) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null) {
			String domain = getDomain(req);
			for (Cookie c : cookies) {
				c.setDomain(domain);
				c.setPath("/");
				c.setMaxAge(0); // 立即销毁cookie
				rsp.addCookie(c);
			}
		}
	}

	/**
	 * 获取用户请求者的IP地址
	 */
	public String getClientIp(HttpServletRequest req) {
		String ip = req.getHeader("X-Forwarded-For");
		LOG.info("-> X-Forwarded-For..." + ip);
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = req.getRemoteAddr();
		}
		// 多个代理多个IP以逗号分割，第一个IP为客户端真实IP
		if (ip != null && ip.indexOf(',') > 0) {
			ip = ip.substring(0, ip.indexOf(','));
		}
		return ip;
	}

	/**
	 * 本地语言（国家CN）
	 */
	public String getLocale(HttpServletRequest req) {
		// 这里不添加“_”符号，i18n方法中再添加
		return req.getLocale().getLanguage().toUpperCase();
	}

	/**
	 * 支持国际化的语言类型
	 */
	final public String[] getLocales() {
		return getConf("i18n.locales").split(",");
	}

	/**
	 * 构建query
	 */
	public String buildQuery(Map<String, String> params) {
		if (params == null || params.isEmpty())
			return "";
		StringBuilder sb = new StringBuilder();
		boolean isJoint = false;
		for (Map.Entry<String, String> e : params.entrySet()) {
			if (isJoint) {
				sb.append("&");
			} else {
				isJoint = true;
			}
			String key = e.getKey();
			String val = e.getValue();
			if (notEmpty(key, val)) {
				sb.append(key).append("=").append(Digest.urlEncode(val));
			}
		}
		return sb.toString();
	}

	/**
	 * 判断字符数组，不为空
	 *
	 * @param values
	 *            字符数组
	 * @return true or false
	 */
	public boolean notEmpty(String... values) {
		if (values == null || values.length == 0)
			return false;
		for (int i = 0; i < values.length; i++) {
			if (isEmpty(values[i]))
				return false;
		}
		return true;
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isEmpty(String str) {
		return (str == null || str.length() == 0);
	}

	/**
	 * 判断对象是否为空
	 */
	public <T> boolean isEmpty(Map<String, T> map) {
		return (map == null || map.isEmpty());
	}

	/**
	 * 步进
	 */
	public int toStep(int min, int max, int step) {
		int num = min + step;
		if (num > max)
			num = max;
		return num;
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable... closeables) {
		if (closeables != null && closeables.length > 0) {
			for (Closeable closeable : closeables) {
				if (closeable == null)
					continue;
				try {
					closeable.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}

}
