package cn.ps1.aolai.service;

import java.io.Closeable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 全局通用的公共方法Utils类
 * 
 * @author Aolai
 * @since 2017年6月17日
 * @verdion 1.0
 */

@Service
public class UtilsService {

	static Logger LOG = LoggerFactory.getLogger(UtilsService.class);
	static ObjectMapper jsonMapper = new ObjectMapper();

	@Autowired
	private ConfService confSvc;

	/**
	 * 构造函数（为了测试用）
	 */
	public UtilsService() {
		confSvc = new ConfService();
	}

	/**
	 * 支持国际化的语言类型
	 */
	final public String[] locales() {
		return getConf("i18n.locales").split(",");
	}

	public int[] intArray(String str) {
		String[] arr = str.split("");
		int[] array = new int[arr.length];
		int n = 0;
		for (int i = 0; i < arr.length; i++) {
			n += Integer.parseInt(arr[i]);
			array[i] = n;
		}
		return array;
	}

	/**
	 * 系统配置参数
	 */
	public String getConf(String key) {
		return confSvc.getConf(key); // config.properties
	}

	public String getArgs(String key) {
		return confSvc.getParam(key); // params.properties
	}

	public String getBase(Map<String, String> map) {
		return getBase(map.get("baseid"));
	}

	public String getBase(String id) {
		return getConf("base.name") + (id == null ? "" : id);
	}

	public String toString(String[] arr) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < arr.length; i++) {
			sb.append(arr[i]);
		}
		return sb.toString();
	}

	/**
	 * 获取请求参数
	 */
	public Map<String, String> jsonParams(HttpServletRequest req) {
		Map<String, String> params = new HashMap<>();
		// 从前端传递来的参数
		String jsonstr = req.getParameter(Const.JSON_STR);
		if (jsonstr == null) {
			Enumeration<String> map = req.getParameterNames();
			while (map.hasMoreElements()) {
				String key = map.nextElement();
				String val = req.getParameter(key);
				if (val.length() != 0)
					params.put(key, val);
			}
		} else {
			params = json2Map(jsonstr);
		}
		// 先以传递的参数为主，再获取当前客户端参数
		if (!params.containsKey("i18n"))
			params.put("i18n", getLocale(req));
		params.put("base", getBase(params));
		return params;
	}

	/**
	 * 获取请求参数，用私钥解码后的json参数
	 */
	public Map<String, String> certParams(HttpServletRequest req) {
		// 从前端传递来的参数
		@SuppressWarnings("unchecked")
		Map<String, String> params = (Map<String, String>) req.getAttribute("json");

		// 先以传递的参数为主，再获取当前客户端参数
		if (!params.containsKey("i18n"))
			params.put("i18n", getLocale(req));
		return params;
	}

	/**
	 * 检查请求（输入）参数无效
	 * 
	 * @param <T>
	 */
	public <T> boolean availParams(Map<String, T> map, String[] keys) {
		if (keys.length == 0)
			return true;
		if (map.isEmpty())
			return false;
		for (int i = 0; i < keys.length; i++) {
			if (!map.containsKey(keys[i]))
				return false;
		}
		return true;
	}

	/**
	 * 对象转换
	 */
	@SuppressWarnings("unchecked")
	public Map<String, Object> obj2Map(Object data) {
		Map<String, Object> temp = (Map<String, Object>) data;
		return (Map<String, Object>) temp.get("info");
	}

	@SuppressWarnings("unchecked")
	public List<Map<String, Object>> obj2List(Object data) {
		Map<String, Object> temp = (Map<String, Object>) data;
		return (List<Map<String, Object>>) temp.get("info");
	}

	/**
	 * 把对象（Map、List）转为json字符串
	 */
	public String obj2Json(Object obj) {
		String jsonStr = null;
		try {
			jsonStr = jsonMapper.writeValueAsString(obj);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		}
		return jsonStr;
	}

	/**
	 * 解码json字符串为map
	 * 
	 * @param <T>
	 */
	public <T> Map<String, T> json2Map(String jsonStr) {
		Map<String, T> map = new HashMap<>();
		if (isEmpty(jsonStr))
			return map;
		try {
			map = jsonMapper.readValue(jsonStr,
					new TypeReference<Map<String, T>>() {
					});
		} catch (Exception e) {
			// 例如：HTTP/1.1 500 Internal Server Error
			LOG.error("-> json2Map..." + jsonStr);
		}
		if (jsonStr.length() > 200)
			LOG.debug("-> json2Map..." + jsonStr.substring(0, 200) + "...");
		else
			LOG.debug("-> json2Map..." + jsonStr);
		return map;
	}

	/**
	 * 解码json字符串为list
	 * 
	 * @param <T>
	 */
	public <T> List<Map<String, T>> json2List(String jsonStr) {
		List<Map<String, T>> list = new ArrayList<>();
		if (isEmpty(jsonStr))
			return list;
		try {
			list = jsonMapper.readValue(jsonStr,
					new TypeReference<List<Map<String, T>>>() {
					});
		} catch (Exception e) {
			LOG.error("-> json2List..." + e.getMessage());
		}
		if (jsonStr.length() > 200)
			LOG.debug("-> json2List..." + jsonStr.substring(0, 200) + "...");
		else
			LOG.debug("-> json2List..." + jsonStr);
		return list;
	}

	/**
	 * list转换为map对象
	 */
	public Map<String, Map<String, String>> list2Map(
			List<Map<String, String>> list, String key) {
		Map<String, Map<String, String>> map = new HashMap<>();
		if (list != null && key != null) {
			for (int i = 0; i < list.size(); i++) {
				Map<String, String> item = list.get(i);
				map.put(item.get(key), item);
			}
		}
		return map;
	}

	public Map<String, String> list2Map(List<Map<String, String>> list,
			String key, String val) {
		Map<String, String> map = new HashMap<>();
		if (list != null && key != null && val != null) {
			for (int i = 0; i < list.size(); i++) {
				Map<String, String> item = list.get(i);
				map.put(item.get(key), item.get(val));
			}
		}
		return map;
	}

	/**
	 * Json对象转换为Map对象
	 * 
	 * @param jsonNode
	 * @return
	 */
	public Map<String, Object> node2Obj(JsonNode jsonNode) {
		Map<String, Object> map = new HashMap<>();
		if (jsonNode.isValueNode()) {
			map.put("", jsonNode.asText());
		} else if (jsonNode.isArray()) {
			List<Object> list = new ArrayList<>();
			Iterator<JsonNode> it = jsonNode.iterator();
			while (it.hasNext()) {
				Map<String, Object> child = node2Obj(it.next());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					list.add(child.get(""));
				} else {
					list.add(child);
				}
			}
			map.put("", list);
		} else {
			Iterator<Map.Entry<String, JsonNode>> it = jsonNode.fields();
			while (it.hasNext()) {
				Map.Entry<String, JsonNode> entity = it.next();
				Map<String, Object> child = node2Obj(entity.getValue());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					map.put(entity.getKey(), child.get(""));
				} else {
					map.put(entity.getKey(), child);
				}
			}
		}
		return map;
	}

	/**
	 * Json字符串转换为Map对象
	 */
	public Map<String, Object> json2Obj(String jsonStr, String path) {
		Map<String, Object> map = new HashMap<>();
		try {
			if (!isEmpty(jsonStr)) {
				JsonNode node = jsonMapper.readTree(jsonStr);
				map = node2Obj(path == null ? node : node.path(path));
			}
		} catch (Exception e) {
			LOG.error("-> json2Obj...");
			e.printStackTrace();
		}
		return map;
	}

	public <T> Map<String, T> json2Obj(String jsonStr) {
		Map<String, T> map = new HashMap<>();
		try {
			if (!isEmpty(jsonStr)) {
				map = jsonMapper.readValue(jsonStr,
						new TypeReference<Map<String, T>>() {
						});
			}
		} catch (Exception e) {
			LOG.error("-> json2Obj...");
			e.printStackTrace();
		}
		return map;
	}

	/**
	 * 发送手机验证码
	 */
	/*public boolean sendVcode(String mobile, String vcode) {
		SendSms sms = new SendSms();
		return sms.sendSms(mobile, vcode);
	}*/

	/**
	 * 返回状态結果：2-参数错误、3-Token失效
	 */
	public Map<String, String> result(String status) {
		Map<String, String> result = new HashMap<>();
		result.put("status", status);
		return result;
	}

	/**
	 * 返回状态結果：false-失败、true-成功
	 */
	public Map<String, String> result(boolean status) {
		return result(status ? "1" : "0");
	}

	/**
	 * 返回状态結果：2-参数错误、3-Token失效
	 */
	public Map<String, String> result(String status, String info) {
		Map<String, String> result = result(status);
		if (info != null)
			result.put("info", info);
		return result;
	}

	/**
	 * 返回失败消息
	 */
	public Map<String, String> failed(String msg) {
		Map<String, String> result = result("0");
		result.put("info", msg);
		return result;
	}

	/**
	 * 返回成功
	 */
	public Map<String, String> success() {
		return result("1");
	}

	/**
	 * 返回数据对象
	 */
	public Map<String, String> success(Object obj) {
		return result("1", obj2Json(obj));
	}

	public Map<String, String> success(Object obj, HttpServletRequest req) {
		// 返回加密信息
		String cert = getAttr(req, "cert");
		String info = Digest.decrypt(obj2Json(obj), cert);
		return result("1", info);
	}

	/**
	 * 获取request属性值
	 * 
	 * @param req
	 * @param key
	 * @return
	 */
	public String getAttr(HttpServletRequest req, String key) {
		return String.valueOf(req.getAttribute(key));
	}

	public Map<String, String> newToken(HttpServletRequest req,
			HttpServletResponse rsp, String userid) {
		Map<String, String> token = new HashMap<>();

		// 设置encode，对userid编码后的结果
		// userid = base64.encode(userid);

		token.put("userid", userid);
		token.put("token", Digest.uuid8());
		token.put("certid", Digest.randKey());
		token.put("uuid", token.get("token")); // 为了兼容旧版三资管理

		// 设置Cookie（前端处理此处可以忽略！！）
		setCookies(req, rsp, token);
		// 获取IP地址，同时把Token写到Redis中
		token.put("ip", getClientIp(req));
		LOG.info("-> newToken..." + token);
		return token;
	}

	/**
	 * 找出列表中匹配的目标
	 */
	public Map<String, String> findIn(List<Map<String, String>> list,
			String key, String val) {
		for (int i = 0; i < list.size(); i++) {
			if (val.equals(list.get(i).get(key)))
				return list.get(i);
		}
		return null;
	}

	/**
	 * 是否存在
	 */
	public boolean findIn(List<String> list, String str) {
		for (int i = 0; i < list.size(); i++) {
			if (list.get(i).equals(str))
				return true;
		}
		return false;
	}

	/**
	 * 是否存在
	 */
	public boolean findIn(String[] arr, String str) {
		for (int i = 0; i < arr.length; i++) {
			if (arr[i].equals(str))
				return true;
		}
		return false;
	}

	/**
	 * 找出字符串中匹配的数字
	 */
	public int matcher(String str, String regex) {
		Pattern p = Pattern.compile(regex); // "\\d+"
		Matcher m = p.matcher(str);
		if (m.find())
			return Integer.valueOf(m.group());
		return 0;
	}

	/**
	 * 判断是否为整数、负数
	 */
	public boolean isInteger(String str) { // 整数
		Pattern p = Pattern.compile("^-?\\d+$"); // ^[-\\+]?[\\d]+$匹配前面的子表达式一次或多次
		return (str == null) ? false : p.matcher(str).matches();
	}

	/**
	 * 判断是否为整数、小数、负数
	 */
	public boolean isNumeric(String str) { // 浮点数
		Pattern p = Pattern.compile("^(-?\\d+)(\\.\\d+)?$"); // ^[-\\+]?\\d+(\\.\\d+)?$
		return (str == null) ? false : p.matcher(str).matches();
	}

	/**
	 * 判断是否为时间格式
	 */
	public boolean isDatetime(String str) {
		try {
			new SimpleDateFormat(Const.DTF).parse(str);
			return true;
		} catch (ParseException e) {
			return false;
		}
	}

	/**
	 * 计算两个日期之间的差数多少秒：sTime开始时间，eTime结束时间
	 */
	public long timeDiff(Date begin, Date end) {
		long sec = 0;
		Calendar cal = Calendar.getInstance();
		cal.setTime(end);
		sec = cal.getTimeInMillis();
		cal.setTime(begin);
		sec -= cal.getTimeInMillis();
		return sec;
	}

	/**
	 * 计算两个日期之间的差数多少秒：sDate开始时间，eDate结束时间
	 */
	public long timeDiff(String begin, String end) {
		long sec = 0;
		try {
			SimpleDateFormat sdf = new SimpleDateFormat(Const.DTF);
			if (begin == null && end == null)
				return timeDiff(sdf.parse(begin), sdf.parse(end));
		} catch (ParseException e) {
			LOG.error("-> timeDiff...");
			e.printStackTrace();
		}
		return sec;
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public String today(String pattern) {
		return today(pattern, Calendar.getInstance());
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	private String today(String pattern, Calendar cal) {
		SimpleDateFormat sdf = new SimpleDateFormat(pattern);
		return sdf.format(cal.getTime());
	}

	public String nextYear(String pattern, int y) {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.YEAR, y);
		return today(pattern, cal);
	}

	/**
	 * 获得下个月期间
	 */
	public String nextMonth(String month, int m) {
		SimpleDateFormat sdf = new SimpleDateFormat(Const.MON);
		Calendar cal = Calendar.getInstance();
		try {
			cal.setTime(sdf.parse(month)); // 设置当前月份
		} catch (ParseException e) {
			LOG.error("-> nextMonth...");
			e.printStackTrace();
		}
		cal.add(Calendar.MONTH, m);
		return sdf.format(cal.getTime());
	}

	/**
	 * 获得下个月期间
	 */
	public String nextMonth(String month) {
		return nextMonth(month, 1);
	}

	/**
	 * 获得上个月期间
	 */
	public String prevMonth(String month, int m) {
		return nextMonth(month, -m);
	}

	/**
	 * 获得上个月期间
	 */
	public String prevMonth(String month) {
		return prevMonth(month, 1);
	}

	/**
	 * 获取host、Port
	 * 
	 * @param uri
	 * @return
	 */
	public String[] getHost(String uri) {
		int idx = uri.indexOf("://");
		uri = idx > 0 ? uri.substring(idx + 3) : uri;
		idx = uri.indexOf('/');
		uri = idx > 0 ? uri.substring(0, idx) : uri;
		return uri.split(":");
	}

	/**
	 * 获取请求参数
	 */
	public String getRequestURI(HttpServletRequest req) {
		LOG.debug("-> HttpRequest..." + req.getHeader("Referer"));

		// String path = req.getContextPath(); // path=“/xxxx”
		String uri = req.getRequestURI(); // uri=“/xxxx/ws/test1”
		LOG.debug("-> HttpRequest..." + uri);

		// uri = uri.substring(path.length()); // uri=“/ws/test1”
		uri = uri.substring(uri.lastIndexOf('/') + 1); // uri=“test1”

		return uri;
	}

	/**
	 * 获取域名（放置cookie使用）
	 */
	private String getDomain(HttpServletRequest req) {
		// final String core = getConf("app.core"); //
		// http://xxx.ps1.cn:8080/xxx
		final String ref = req.getHeader("Referer"); // 泉州同时包含域名、IP登录，需要动态获取。
		LOG.info("-> host..." + ref);
		String host = getHost(ref)[0];
		LOG.debug("-> getDomain..." + host);
		String[] arr = host.split("\\.");
		return arr.length == 3 ? arr[1] + "." + arr[2] : host;
	}

	/**
	 * 设置cookies
	 */
	public void setCookies(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> map) {
		if (map == null)
			return;
		String domain = getDomain(req);
		for (Map.Entry<String, String> e : map.entrySet()) {
			// Cookie c =new Cookie(key, e.getValue());
			Cookie c = new Cookie(e.getKey(), Digest.escape(e.getValue()));
			c.setDomain(domain);
			c.setPath("/");
			// c.setHttpOnly(true); // 安全保护
			rsp.addCookie(c);
		}
	}

	/**
	 * 获取所有Cookie键值对
	 */
	public Map<String, String> getCookies(HttpServletRequest req) {
		Map<String, String> map = new HashMap<>();
		Cookie[] cookies = req.getCookies();
		if (cookies == null)
			return map;
		for (Cookie c : cookies) {
			// map.put(c.getName(), c.getValue());
			map.put(c.getName(), Digest.unescape(c.getValue()));
		}
		return map;
	}

	/**
	 * 获取指定的Cookie（暂未用）
	 */
	public String getCookie(HttpServletRequest req, String key) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null)
			for (Cookie c : cookies) {
				if (c.getName().equals(key)) {
					// return c.getValue();
					return Digest.unescape(c.getValue());
				}
			}
		return null;
	}

	/**
	 * 刪除指定的Cookie（暂未用）
	 */
	public void delCookie(HttpServletRequest req, HttpServletResponse rsp,
			String key) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null) {
			for (Cookie c : cookies) {
				if (c.getName().equals(key)) {
					c.setDomain(getDomain(req));
					c.setPath("/");
					c.setMaxAge(0); // 立即销毁cookie
					rsp.addCookie(c);
					break;
				}
			}
		}
	}

	/**
	 * 清除全部Cookie的键值
	 */
	public void clearCookies(HttpServletRequest req, HttpServletResponse rsp) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null) {
			String domain = getDomain(req);
			for (Cookie c : cookies) {
				c.setDomain(domain);
				c.setPath("/");
				c.setMaxAge(0); // 立即销毁cookie
				rsp.addCookie(c);
			}
		}
	}

	/**
	 * 阻止非法访客频繁调用
	 */
	public boolean denyIp(HttpServletRequest req, RedisService rds) {
		String ip = Const.RDS_DENY + getClientIp(req);
		String val = rds.get(ip); // 获取非法访问次数
		if (val == null) {
			rds.set(ip, "1", Const.ONE_HH); // 限制非法ip访问1小时
		} else {
			int count = Integer.parseInt(val) + 1;
			if (count > 1000) // 因为企业网关是一个地址，暂时调整为1000上限
				return true; // 若1小时内非法请求1000次，则成功拒绝

			if (count < 1000000) // 大于百万次后不再更新
				rds.set(ip, String.valueOf(count)); // , rds.ONE_HH);
		}
		return false; // 1000次以内，暂时不用拒绝
	}

	/**
	 * 获取用户请求者的IP地址
	 */
	public String getClientIp(HttpServletRequest req) {
		String ip = req.getHeader("X-Forwarded-For");
		LOG.info("-> X-Forwarded-For..." + ip);
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = req.getRemoteAddr();
		}
		// 多个代理多个IP以逗号分割，第一个IP为客户端真实IP
		if (ip != null && ip.indexOf(',') > 0) {
			ip = ip.substring(0, ip.indexOf(','));
		}
		return ip;
	}

	/**
	 * 本地语言（国家CN）
	 */
	public String getLocale(HttpServletRequest req) {
		// 这里不添加“_”符号，i18n方法中再添加
		return req.getLocale().getLanguage().toUpperCase();
	}

	/**
	 * 构建query
	 */
	public String buildQuery(Map<String, String> params) {
		if (params == null || params.isEmpty())
			return "";
		StringBuilder sb = new StringBuilder();
		boolean isJoint = false;
		for (Map.Entry<String, String> e : params.entrySet()) {
			if (isJoint) {
				sb.append("&");
			} else {
				isJoint = true;
			}
			String key = e.getKey();
			String val = e.getValue();
			if (isNotEmpty(key, val)) {
				sb.append(key).append("=").append(Digest.urlEncode(val));
			}
		}
		return sb.toString();
	}

	/**
	 * 判断字符数组，不为空
	 *
	 * @param values
	 *            字符数组
	 * @return true or false
	 */
	public boolean isNotEmpty(String... values) {
		if (values == null || values.length == 0)
			return false;
		for (int i = 0; i < values.length; i++) {
			if (isEmpty(values[i]))
				return false;
		}
		return true;
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isEmpty(String str) {
		return (str == null || str.length() == 0);
	}

	/**
	 * 判断对象是否为空
	 */
	public <T> boolean isEmpty(Map<String, T> map) {
		return (map == null || map.isEmpty());
	}

	/**
	 * 步进
	 */
	public int toStep(int min, int max, int step) {
		int num = min + step;
		if (num > max)
			num = max;
		return num;
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable... closeables) {
		if (closeables != null && closeables.length > 0) {
			for (Closeable closeable : closeables) {
				if (closeable == null)
					continue;
				try {
					closeable.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}

}
