package cn.ps1.aolai.service;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.utils.Const;
import redis.clients.jedis.ShardedJedis;
import redis.clients.jedis.ShardedJedisPool;

/**
 * Redi的增删改查及Token基本业务操作
 * 
 * @author Aolai
 * @since 2017.6.17
 * @version 1.0
 * 
 */

@Service
public class RedisService {

	private static Logger LOG = LoggerFactory.getLogger(RedisService.class);
	private static final boolean singleUser = true; // 仅限单一用户登录

	@Autowired
	private ShardedJedisPool jedisPool;

	/**
	 * 清除token信息：{userid,token,certid}
	 */
	public boolean clearToken(Map<String, String> params) {
		if (!checkToken(params))
			return false;
		LOG.debug("-> clearToken..." + params);

		String key = singleUser ? params.get("userid") : params.get("token");
		del(Const.RDS_TOKEN + key);

		/** 兼容旧版格式：清除旧版的userid，否则新版退出后再登录旧版时无法登录 */
		del("u" + params.get("userid"));

		return true;
	}

	/**
	 * 设置新token或保持已有token持续有效
	 */
	public void setToken(Map<String, String> params) {
		// 仅限单一用户登录，或支持多用户同时登录
		String key = singleUser ? params.get("userid") : params.get("token");
		hmset(Const.RDS_TOKEN + key, params, Const.TWO_HH);
	}

	/**
	 * 验证token是否有效：单用户不能同时缓存多个token，多用户可以有多个token缓存同一个userid
	 */
	public boolean checkToken(Map<String, String> cookie) {
		String token = cookie.get("token");
		String userid = cookie.get("userid");
		String certid = cookie.get("certid");
		if (token != null && userid != null && certid != null) {
			String key = Const.RDS_TOKEN + (singleUser ? userid : token);
			String result = singleUser ? token : userid;
			String val = singleUser ? "token" : "userid";
			
			// 根据登录用户userid从缓存中读取并验证缓存信息并进行校验
			Map<String, String> map = hmget(key);
			if (map != null && result.equals(map.get(val))
					&& certid.equals(map.get("certid"))) {
				// 验证成功后自动延长有效期
				expire(key);
				return true; //result;
			}
		}
		return false; //null
	}

	/**
	 * 设置单个键值，永久有效
	 */
	public String set(String key, String val) {
		return set(key, val, 0); // 返回OK：成功
	}

	/**
	 * 设置单个键值，并设置有效时间
	 */
	public String set(String key, String val, int sec) {
		LOG.debug("-> set..." + key + "=" + val);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			result = jedis.set(key, val);
			if (sec > 0)
				expire(key, sec);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result; // 返回OK：成功
	}

	/**
	 * 设置map对象，永久有效
	 */
	public String hmset(String key, Map<String, String> map) {
		return hmset(key, map, 0);
	}

	/**
	 * 设置map对象
	 */
	public String hmset(String key, Map<String, String> map, int sec) {
		LOG.debug("-> hmset..." + key + "=" + map);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			// jedis.del(key);
			result = jedis.hmset(key, map);
			if (sec > 0)
				expire(key, sec);
		} catch (Exception e) {
			LOG.error("->hmset..." + e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 获取map对象
	 */
	public Map<String, String> hmget(String key) {
		LOG.debug("-> hmget..." + key);
		Map<String, String> result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			result = jedis.hgetAll(key);
			if (result != null && result.isEmpty())
				result = null;
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 获取map对象单个域的键值
	 */
	public String hget(String key, String field) {
		LOG.debug("-> hget..." + key);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			result = jedis.hget(key, field);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 设置map对象单个域的键值，永久有效
	 */
	public boolean hset(String key, String field, String val) {
		return hset(key, field, val, 0);
	}

	/**
	 * 设置map对象单个域的键值
	 */
	public boolean hset(String key, String field, String val, int sec) {
		LOG.debug("-> hset..." + key);
		boolean result = false;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return result;
		try {
			jedis.hset(key, field, val);
			if (sec > 0)
				expire(key, sec);
			result = true;
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 获取单个键值
	 */
	public String get(String key) {
		LOG.debug("-> get..." + key);
		String result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return null;
		try {
			result = jedis.get(key);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 根据key删除键值对数据
	 */
	public boolean del(String key) {
		boolean result = false;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return false;
		try {
			jedis.del(key);
			result = true;
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 判断key是否存在
	 */
	public boolean exists(String key) {
		Boolean result = false;
		ShardedJedis jedis = jedisPool.getResource();
		if (jedis == null)
			return false;
		try {
			result = jedis.exists(key);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

	/**
	 * 设置key默认有效期，默认1小时有效
	 */
	public Long expire(String key) {
		return expire(key, Const.ONE_HH);
	}

	/**
	 * 设置key的有效期，或延续时间
	 */
	private Long expire(String key, int sec) {
		Long result = null;
		ShardedJedis jedis = jedisPool.getResource();
		if (sec == 0 || jedis == null)
			return null;
		try {
			result = jedis.expire(key, sec);
		} catch (Exception e) {
			LOG.error(e.getMessage());
		} finally {
			jedis.close();
		}
		return result;
	}

}
