package cn.ps1.aolai.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.dao.AolaiDao;
import cn.ps1.aolai.utils.Const;

/**
 * Aolai基础框架相关服务层，数据库基础操作类(CRUD)
 * 
 * @author Aolai
 * @version 1.0 $Date: 2019/7/6
 * @since 1.7
 */

@Service
public class AolaiService {

	private static Logger LOG = LoggerFactory.getLogger(AolaiService.class);
	private static final String META = "GMETA";

	@Autowired
	private AolaiDao aolaiDao;
	@Autowired
	private RedisService redisSvc;
	@Autowired
	private UtilsService utilsSvc;

	private static Map<String, String> gmeta = new HashMap<>();
	static {
		gmeta.put("field", "META_FIELD");
		gmeta.put("alias", "META_ALIAS");
		gmeta.put("name", "META_NAME");
		gmeta.put("type", "META_TYPE");
		gmeta.put("null", "META_NULL");
		gmeta.put("i18n", "META_I18N");
		gmeta.put("pkey", "META_PKEY");
		gmeta.put("width", "META_WIDTH");
		gmeta.put("default", "META_DEFAULT");
		gmeta.put("sort", "META_SORT");
	}

	/*********************************************************
	 * 往“table”表中增加一条记录
	 */

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 * 
	 * @param table 表名
	 * @param dto 字段映射
	 * @param data 数据
	 * @param dupl 更新标记
	 * @return result 返回状态
	 */
	private Map<String, String> addRecord(String table, Map<String, String> dto,
			Map<String, Object> data, boolean dupl) {
		// 根据table表名获取“别名”“字段”的映射关系
		data = setFieldData(dto, data, true);
		if (utilsSvc.isEmpty(data))
			utilsSvc.result("2");

		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		if (dupl)
			map.put("dupl", "1");

		// 返回执行结果
		LOG.info("-> addRecord..." + data);
		return utilsSvc.result(aolaiDao.addOne(map, data) > 0);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(table, getDto(table), data, dupl);
	}

	/**
	 * 往基础表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table, Map<String, Object> data) {
		return addRecord(table, data, false);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String base, String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(getTable(base, table), getDto(table), data, dupl);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String base, String table,
			Map<String, Object> data) {
		return addRecord(base, table, data, false);
	}

	/*********************************************************
	 * 批量往业务表中插入多条数据
	 */
	private Map<String, String> batchAdd(String table, Map<String, String> dto,
			List<Map<String, Object>> items, String lang, boolean dupl) {
		if (items.size() == 0)
			return utilsSvc.result("2");
		Map<String, Object> data = items.get(0);
		data.put("i18n", lang);
		data = setFieldData(dto, data, false);
		if (utilsSvc.isEmpty(data))
			utilsSvc.result("2");

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		if (dupl)
			map.put("dupl", "1");

		// 返回执行结果
		LOG.info("->batchAdd..." + map);
		return utilsSvc.result(aolaiDao.batchAdd(map, data, items) > 0);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String base, String table,
			List<Map<String, Object>> items, String lang, boolean dupl) {
		// 根据table表名，获取表的列名与map的映射关系
		// {uid:USER_ID,uid!t:INTEGER,uid!i}
		Map<String, String> dto = getDto(table);
		return batchAdd(getTable(base, table), dto, items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, String lang, boolean dupl) {
		return batchAdd(table, getDto(table), items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, String lang) {
		return batchAdd(table, items, lang, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items) {
		return batchAdd(table, items, null);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(Map<String, String> map) {
		// 整理“数据转换对象”，并检查参数：fields,duplte
		List<Map<String, Object>> list = utilsSvc.json2List(map.get("fields"));
		return batchAdd(map.get("base"), map.get("table"), list,
				map.get("i18n"), map.containsKey("duplte"));
	}

	/*********************************************************
	 * 删除表中的数据
	 */
	private Map<String, String> delete(String table, Map<String, String> dto,
			Map<String, Object> where, String joint) {
		// 梳理删除条件
		Map<String, Object> cond = setCondition(dto, where, joint);
		if (utilsSvc.isEmpty(cond))
			return utilsSvc.result("2");
		LOG.info("-> delete..." + cond);
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		return utilsSvc.result(aolaiDao.delete(map, cond) > 0);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> where,
			String joint) {
		return delete(table, getDto(table), where, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> where) {
		return delete(table, getDto(table), where, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String base, String table,
			Map<String, Object> where, String joint) {
		return delete(getTable(base, table), getDto(table), where, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String base, String table,
			Map<String, Object> where) {
		return delete(base, table, where, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Map<String, String> map) {
		// 整理“数据转换对象”，并检查参数
		Map<String, Object> where = utilsSvc.json2Map(map.get("where"));
		return delete(map.get("base"), map.get("table"), where);
	}

	/*********************************************************
	 * 批量删除
	 */
	private Map<String, String> batchDelete(String table,
			Map<String, String> dto, List<Map<String, Object>> list) {
		int count = 0;
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		for (int i = 0; i < list.size(); i++) {
			cond = setCondition(dto, list.get(i), null);
			if (utilsSvc.isEmpty(cond)) {
				continue;
				// return utilsSvc.result("2");
			}
			map.put("table", table);
			// 计数
			count += aolaiDao.delete(map, cond);
		}
		// if (count == list.size())
		return utilsSvc.result(count > 0);
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(String base, String table,
			List<Map<String, Object>> list) {
		return batchDelete(getTable(base, table), getDto(table), list);
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(String base, String table,
			String where) {
		return batchDelete(base, table, utilsSvc.json2List(where));
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(String table,
			List<Map<String, Object>> list) {
		return batchDelete(table, getDto(table), list);
	}

	/**
	 * 批量删除表中的数据
	 */
	public Map<String, String> batchDelete(String table, String where) {
		return batchDelete(table, utilsSvc.json2List(where));
	}

	/*********************************************************
	 * 更新表中的数据
	 */
	private Map<String, String> update(String table, Map<String, String> dto,
			Map<String, Object> fields, Map<String, Object> where, String joint) {
		Map<String, Object> cond = setCondition(dto, where, joint);
		Map<String, Object> values = setCondition(dto, fields);
		if (utilsSvc.isEmpty(cond) || utilsSvc.isEmpty(values))
			return utilsSvc.result("2");
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		// 返回执行结果
		return utilsSvc.result(aolaiDao.update(map, values, cond) > 0);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String base, String table,
			Map<String, Object> fields, Map<String, Object> where, String joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// {uid:USER_ID,uid!t:INTEGER,uid!i}
		Map<String, String> dto = getDto(table);
		return update(getTable(base, table), dto, fields, where, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String base, String table,
			Map<String, Object> fields, Map<String, Object> where) {
		return update(base, table, fields, where, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> fields,
			Map<String, Object> where, String joint) {
		return update(null, table, fields, where, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> fields,
			Map<String, Object> where) {
		return update(null, table, fields, where, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Map<String, String> map, String lang) {
		// 整理“数据转换对象”，并检查参数
		Map<String, Object> fields = utilsSvc.json2Map(map.get("fields"));
		Map<String, Object> where = utilsSvc.json2Map(map.get("where"));
		fields.put("i18n", lang);
		where.put("i18n", lang);
		return update(map.get("base"), map.get("table"), fields, where);
	}

	/*********************************************************
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public List<Map<String, String>> findList(Map<String, String> args,
			Map<String, Object> where, Map<String, String> order) {
		String table = args.get("table");
		Map<String, String> dto = getDto(table);
		String base = args.get("base");
		String alias = args.get("alias"); // 传回前端的参数数组
		String limit = args.get("limit");
		String joint = args.get("joint");
		String i18n = null;
		if (where != null)
			i18n = String.valueOf(where.get("i18n"));
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		map.put("alias", setAlias(dto, i18n, alias)); // 无SQL注入风险

		Map<String, Object> cond = setCondition(dto, where, joint);

		// 整理“数据转换对象”，并检查参数
		String sort = getOrder(dto, order, i18n);
		if (sort.length() > 0)
			map.put("order", sort);

		// 整理“数据转换对象”，并检查参数
		if (utilsSvc.isEmpty(limit)) {
			final String rows = utilsSvc.getConf("limit.rows");
			map.put("limit", rows); // 仅limit=''
		} else {
			map.put("limit", limit);
		}
		// 返回执行结果
		return aolaiDao.findList(map, cond);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(Map<String, String> args,
			Map<String, Object> where) {
		return findList(args, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String base, String table,
			String alias, Map<String, Object> where) {
		Map<String, String> args = new HashMap<>();
		if (base != null)
			args.put("base", base);
		args.put("table", table);
		args.put("alias", alias); // 传回前端的参数数组
		return findList(args, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table, String alias,
			Map<String, Object> where) {
		Map<String, String> args = new HashMap<>();
		args.put("table", table);
		args.put("alias", alias); // 传回前端的参数数组
		return findList(args, where, null);
	}

	/*********************************************************
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	private Map<String, String> findOne(String table, Map<String, String> dto,
			Map<String, Object> where, String keys) {
		Map<String, Object> cond = setCondition(dto, where, null);
		if (utilsSvc.isEmpty(cond))
			return utilsSvc.result("2");
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		String i18n = String.valueOf(where.get("i18n"));
		map.put("table", table);
		map.put("alias", setAlias(dto, i18n, keys)); // 无SQL注入风险

		// 返回执行结果，限定了只返回一条记录
		map = aolaiDao.findOne(map, cond);
		if (map == null)
			return utilsSvc.result(false);
		return map;
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String base, String table,
			Map<String, Object> where, String keys) {
		return findOne(getTable(base, table), getDto(table), where, keys);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String base, String table,
			Map<String, Object> where) {
		return findOne(getTable(base, table), getDto(table), where, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> where,
			String keys) {
		return findOne(null, table, where, keys);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> where) {
		return findOne(null, table, where, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Map<String, String> map, String keys) {
		Map<String, Object> where = utilsSvc.json2Map(map.get("where"));
		return findOne(map.get("base"), map.get("table"), where, keys);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Map<String, String> map) {
		return findOne(map, null);
	}

	/**
	 * 自定义组合条件检索数据
	 */
	private Map<String, String> findOne(String table, Map<String, String> dto,
			String alias, Map<String, Object> where) {
		Map<String, Object> cond = setCondition(dto, where, null);
		Map<String, String> map = new HashMap<>();
		map.put("table", table);
		map.put("alias", alias);
		// 返回执行结果
		return aolaiDao.findOne(map, cond);
	}

	/**
	 * 自定义组合条件检索数据
	 */
	public Map<String, String> findOne(String base, String table, String alias,
			Map<String, Object> where) {
		return findOne(getTable(base, table), getDto(table), alias, where);
	}

	/**
	 * 自定义组合条件检索数据
	 */
	public Map<String, String> queryOne(String table, String alias,
			Map<String, Object> where) {
		return findOne(table, getDto(table), alias, where);
	}

	/*********************************************************
	 * 自定义组合条件检索数据
	 */
	public List<Map<String, String>> query(String sql) {
		// 返回执行结果
		return aolaiDao.query(sql);
	}

	/*********************************************************
	 * 判断表中是否有数据
	 */
	private boolean exists(String table, Map<String, String> dto,
			Map<String, Object> where, String joint) {
		// 数据库执行参数
		Map<String, Object> cond = setCondition(dto, where, joint);
		Map<String, String> map = new HashMap<>();
		map.put("table", table);

		// 返回执行结果
		return aolaiDao.exists(map, cond) > 0;
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String base, String table, Map<String, Object> where,
			String joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// {uid:USER_ID,uid!t:INTEGER,uid!i}
		Map<String, String> dto = getDto(table);
		return exists(getTable(base, table), dto, where, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String base, String table, Map<String, Object> where) {
		return exists(base, table, where, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> where, String joint) {
		return exists(table, getDto(table), where, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> where) {
		return exists(table, getDto(table), where, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Map<String, String> map) {
		// 整理“数据转换对象”，并检查参数
		Map<String, Object> where = utilsSvc.json2Map(map.get("where"));
		return exists(map.get("base"), map.get("table"), where);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/*********************************************************
	 * 获取某个应用的菜单权限信息一览
	 * 
	 * @param map
	 * @param args
	 * @return
	 */
	public List<Map<String, String>> getMenuRole(Map<String, String> map,
			String args) {
		map.put("alias", setAlias(getDto("PAGE"), map.get("i18n"), args)); // 无SQL注入风险
		// 关联岗位、权限、菜单表中信息
		return aolaiDao.getMenuRole(map);
	}

	/**
	 * 获取某个应用的菜单权限信息一览
	 */
	public List<Map<String, String>> getMenuRole(Map<String, String> map) {
		// 数据库执行参数："getMenuRole"
		return getMenuRole(map, "getMenuRole");
	}

	/**
	 * 获取某个页面上的按钮项的操作权限一览
	 */
	public List<Map<String, String>> getActionRole(Map<String, String> map,
			String args) {
		map.put("alias", setAlias(getDto("ACTION"), map.get("i18n"), args)); // 无SQL注入风险
		// 关联岗位、权限、菜单表中信息
		return aolaiDao.getActionRole(map);
	}

	/**
	 * 获取某个页面上的按钮项的操作权限一览
	 */
	public List<Map<String, String>> getActionRole(Map<String, String> map) {
		// 数据库执行参数actionbyte
		return getActionRole(map, "getActionRole");
	}

	/**
	 * 获取系统配置参数
	 * 
	 * @param base 帐套名
	 * @param confTag 参数标签
	 * @return String 返回键值对
	 */
	public Map<String, String> getConf(String base, String confTag) {
		String conf = getTable(base, "CONF");
		Map<String, Object> where = new HashMap<>();
		where.put("confTag", confTag);
		List<Map<String, String>> list = findList(conf, "getConfList", where);
		return utilsSvc.list2Map(list, "confKey", "confVal");
	}

	/**
	 * 获取系统配置参数
	 * 
	 * @param base 帐套名
	 * @param paramKey 参数主键
	 * @return String 返回值
	 */
	public String getParam(String base, String paramKey) {
		Map<String, Object> where = new HashMap<>();
		Map<String, String> result = new HashMap<>();
		// 获取系统配置参数
		where.put("paramKey", paramKey);
		result = findOne(base, "PARAM", where, "getParamList");
		return result.get("parmVal");
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 增删改查，字段自动支持国际化
	 */
	private String i18n(Map<String, String> dto, String key, String lang) {
		String field = dto.get(key);
		// 判断是否支持国际化
		if (lang != null && dto.containsKey(key + "!i"))
			field += "_" + lang; // 获取clientLang，添加“_”符号
		return field;
	}

	/**
	 * 查询排序条件：根据“数据转换对象”，拼接排序条件
	 */
	private String getOrder(Map<String, String> dto, Map<String, String> data,
			String lang) {
		String str = "", key, field;
		// 检查排序条件
		if (data == null || data.isEmpty()) { // 默认按主键（pkey）排序
			for (Map.Entry<String, String> e : dto.entrySet()) {
				key = e.getKey();
				if (key.indexOf("!p") > 0 && "1".equals(e.getValue())) {
					if (str.length() > 0)
						str += ",";
					key = key.substring(0, key.indexOf("!p"));
					str += dto.get(key);
				}
			}
			return str;
		}
		for (Map.Entry<String, String> e : data.entrySet()) {
			key = e.getKey();
			if (dto.containsKey(key)) {
				if (str.length() > 0)
					str += ",";
				field = i18n(dto, key, lang); // 支持国际化
				str += field + " " + e.getValue();
			}
		}
		return str;
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	private Map<String, Object> setCondition(Map<String, String> dto,
			Map<String, Object> params) {
		return setCondition(dto, params, null);
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	private Map<String, Object> setCondition(Map<String, String> dto,
			Map<String, Object> params, String joint) {
		if (utilsSvc.isEmpty(params))
			return params;
		Map<String, Object> cond = new HashMap<>();
		joint = joint == null ? "=" : " " + joint;
		for (Map.Entry<String, Object> e : params.entrySet()) {
			String key = e.getKey();
			if (!dto.containsKey(key))
				continue;
			key = i18n(dto, key, String.valueOf(params.get("i18n")));
			cond.put(key + joint, e.getValue());
		}
		return cond;
	}

	/**
	 * 添加数据时的键值对：字段名对应的数值
	 */
	private Map<String, Object> setFieldData(Map<String, String> dto,
			Map<String, Object> params, boolean isVal) {
		if (utilsSvc.isEmpty(params))
			return null;
		Map<String, Object> data = new HashMap<>();
		for (Map.Entry<String, Object> e : params.entrySet()) {
			String key = e.getKey();
			if (!dto.containsKey(key))
				continue;
			// if (checkDataType(dto, key, val))
			// return null;
			key = i18n(dto, key, String.valueOf(params.get("i18n")));
			data.put(key, isVal ? e.getValue() : e.getKey());
		}
		return data;
	}

	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String lang) {
		String alias = "", key;
		for (Map.Entry<String, String> e : dto.entrySet()) {
			key = e.getKey();
			if (key.contains("!")) // 为区分type、i18n，标注为!t,!i
				continue;
			if (alias.length() > 0)
				alias += ",";
			alias += setAlias(dto, key, e.getValue(), lang);
		}
		return alias;
	}

	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String lang, String args) {
		if (utilsSvc.isEmpty(args))
			return setAlias(dto, lang);
		String val = utilsSvc.getArgs(args); // 未定义时默认返回空（""）
		if (val.length() == 0) // && args.length() == 0
			return setAlias(dto, lang);
		// val = val.length() > 0 ? val : args;

		String[] arr = val.split("\\|");
		String alias = "";
		for (int i = 0; i < arr.length; i++) {
			if (i > 0)
				alias += ",";
			alias += setAlias(dto, arr[i], dto.get(arr[i]), lang);
		}
		return alias;
	}

	/**
	 * 拼接别名（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String key, String val,
			String lang) {
		String type = dto.get(key + "!t"); // 为区分type、i18n，标注为!t,!i
		if (type != null) { // META_FIELD || VARCHAR
			if (type.matches("T.+")) { // TIMESTAMP
				val = "DATE_FORMAT(" + val + ",'%Y-%m-%d %T')";
			} else if (type.matches("[IDBS].+")) {
				// INTEGER、DOUBLE、BIGINT、SMALLINT
				val = "CAST(" + val + " AS CHAR)";
			} else if (dto.containsKey(key + "!i")) { // VARCHAR
				val = val + "_" + lang;
			}
		}
		return val + " as '" + key + "'";
	}

	/**
	 * 数据库与前台交互传递的参数（暂时未用）
	 */
	public String getFieldAlias(List<Map<String, String>> meta, String lang) {
		return setAlias(getDto(meta), lang); // 无SQL注入风险
	}

	/**
	 * 元数据对象：获取表的“别名”与“字段”的映射关系，以及数据类型及是否国际化 {uid:USER_ID}
	 */
	private Map<String, String> getDto(String table) {
		String key = Const.RDS_META + table;
		Map<String, String> dto = redisSvc.hmget(key);
		if (dto == null) {
			dto = getDto(getMetaList(table));
			if (dto != null)
				redisSvc.hmset(key, dto, Const.TEN_HH); // 至少缓存10小时
		}
		return dto;
	}

	/**
	 * 元数据对象
	 */
	private Map<String, String> getDto(List<Map<String, String>> metaList) {
		if (metaList.size() == 0)
			return null;
		Map<String, String> dto = new HashMap<>();
		for (int i = 0; i < metaList.size(); i++) {
			Map<String, String> meta = metaList.get(i);
			String alias = meta.get("alias");
			dto.put(alias, meta.get("field"));
			dto.put(alias + "!t", meta.get("type"));
			dto.put(alias + "!p", meta.get("pkey"));
			if (meta.get("i18n").equals("1"))
				dto.put(alias + "!i", meta.get("i18n"));
		}
		return dto;
	}

	/**
	 * 根据表名table从GMETA表中获取Meta元数据一览
	 * 
	 * @param table 表名
	 * @return list 元数据一览
	 */
	private List<Map<String, String>> getMetaList(String table) {
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		map.put("table", META);
		map.put("alias", setAlias(gmeta, null)); // 无SQL注入风险
		map.put("order", "META_SORT");
		cond.put("META_TABLE=", table);
		return aolaiDao.findList(map, cond);
	}

	/**
	 * 检索元数据
	 */
	public List<Map<String, String>> getMetaData(String args) {
		args = utilsSvc.getArgs(args);
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		map.put("table", META);
		map.put("alias", setAlias(gmeta, null)); // 无SQL注入风险
		map.put("order", "META_SORT");
		cond.put("META_ALIAS in", "('" + args.replace("|", "','") + "')");
		return aolaiDao.findList(map, cond);
	}

	/**
	 * 账套（base）及表名组合
	 */
	public String getTable(Map<String, String> map, String table) {
		return getTable(utilsSvc.getBase(map), table);
	}

	/**
	 * 账套（base）及表名组合
	 */
	public String getTable(String base, String table) {
		return (base == null) ? table : (base + "." + table);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 仅操作GMETA表的元数据“键值对”
	 */
	private Map<String, String> getMetaDto() {
		Map<String, String> map = new HashMap<>();
		map.putAll(gmeta);
		map.put("table", "META_TABLE");
		return map;
	}

	/**
	 * 往GMETA表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addMetaData(Map<String, Object> data,
			boolean dupl) {
		return addRecord(META, getMetaDto(), data, dupl);
	}

	/**
	 * 从Meta表中批量删除数据
	 */
	public Map<String, String> delMetaData(List<Map<String, Object>> list) {
		return batchDelete(META, getMetaDto(), list);
	}

	/**
	 * 从Meta表中获取元数据表名信息（暂停使用）
	 */
	public List<Map<String, String>> getMetaTable() {
		return aolaiDao.getMetaTable();
	}

	/**
	 * 判断某库、某表名是否存在（暂未使用：暂无SQL注入风险）
	 */
	public boolean tableExists(String base, String table) {
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		cond.put("TABLE_SCHEMA", base);
		cond.put("TABLE_NAME", table);
		map.put("table", "information_schema.TABLES");
		return aolaiDao.exists(map, cond) > 0;
	}

	/**
	 * 判断某库、某表、某列名是否存在（暂未使用：暂无SQL注入风险）
	 */
	public boolean columnExists(String base, String table) {
		Map<String, String> map = new HashMap<>();
		Map<String, Object> cond = new HashMap<>();
		cond.put("TABLE_SCHEMA=", base);
		cond.put("COLUMN_NAME=", table);
		map.put("table", "information_schema.COLUMNS");
		return aolaiDao.exists(map, cond) > 0;
	}

	/*********************************************************
	 * 删除表
	 * 
	 * @param table
	 */
	public Map<String, String> dropTable(String base, String table) {
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		// 动态删表
		aolaiDao.drop(map);
		redisSvc.del(Const.RDS_META + table);
		// 返回成功
		return utilsSvc.result(true);
	}

	/*********************************************************
	 * 创建表
	 */
	public Map<String, String> createTable(String base, String table) {
		// 先从META表中获取元数据
		Map<String, String> map = new HashMap<>();
		map.put("table", getTable(base, table));
		map.put("fields", getMetaSql(getMetaList(table)));
		// 动态建表
		LOG.debug("-> createTable..." + map);
		aolaiDao.create(map);
		// 返回成功
		return utilsSvc.result(true);
	}

	/**
	 * 根据Meta元数据整理建表Sql
	 */
	private String getMetaSql(List<Map<String, String>> meta) {
		String sql = "", field, pkey = "";
		String[] lang = utilsSvc.locales(); // 国际化支持
		for (int i = 0; i < meta.size(); i++) {
			if (i > 0)
				sql += ",";
			Map<String, String> map = meta.get(i);
			// 支持国际化的字段
			if (map.get("i18n").equals("1")) {
				for (int j = 0; j < lang.length; j++) {
					if (j > 0)
						sql += ",";
					field = map.get("field") + "_" + lang[j];
					sql += field + " " + map.get("type");
					if (map.get("null").equals("1"))
						sql += " NOT NULL";
					if (map.get("pkey").equals("1")) {
						if (pkey.length() > 0)
							pkey += ",";
						pkey += field;
					}
					if (map.get("default").length() > 0)
						sql += " DEFAULT " + map.get("default");
				}
			} else {
				field = map.get("field");
				sql += field + " " + map.get("type");
				if (map.get("null").equals("1"))
					sql += " NOT NULL";
				if (map.get("pkey").equals("1")) {
					if (pkey.length() > 0)
						pkey += ",";
					pkey += field;
				}
				if (map.get("default").length() > 0)
					sql += " DEFAULT " + map.get("default");
			}

		}
		if (pkey.length() > 0)
			sql += ",PRIMARY KEY (" + pkey + ")";
		return sql;
	}

	/**
	 * 直接获取Meta对象的元数据的属性keys数组（暂未使用）
	 */
/*	public String[] getDtoKeys() {
		return getDtoKeys(gmeta);
	}*/

	/**
	 * 根据Meta先获取“数据转换对象”，再获取元数据的属性keys数组（暂未使用）
	 */
/*	public String[] getDtoKeys(Map<String, String> map) {
		int size = map.size();
		String[] keys = new String[size + 1];
		map.keySet().toArray(keys);
		keys[size] = "table"; // 扩展一个表名‘table’字段
		return keys;
	}*/

	/**
	 * 根据table先获取“数据转换对象”，再获取元数据的属性keys数组（暂未使用）
	 */
/*	public String[] getDtoKeys(String table) {
		List<Map<String, String>> metaList = getMetaList(table);
		Map<String, String> map = new HashMap<>();
		for (int i = 0; i < metaList.size(); i++) {
			Map<String, String> meta = metaList.get(i);
			if ("1".equals(meta.get("pkey")))
				map.put(meta.get("alias"), meta.get("field"));
		}
		return getDtoKeys(map);
	}*/

}
