package cn.ps1.soar.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.service.AolaiService;
import cn.ps1.aolai.service.UtilsService;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.soar.entity.Chose;
import cn.ps1.soar.entity.Emp;
import cn.ps1.soar.entity.Inform;
import cn.ps1.soar.entity.Mode;
import cn.ps1.soar.entity.Node;
import cn.ps1.soar.entity.Task;
import cn.ps1.soar.entity.Work;
import cn.ps1.soar.utils.FlowUtil;

/**
 * 审批任务处理
 * 
 * @author Aolai
 * @version 1.0 $Date: 2024.01.20
 * @since openjdk-1.8
 */
@Service
public class TaskService {

	private static Logger log = LoggerFactory.getLogger(TaskService.class);

	@Autowired
	private AolaiService aolai;
	@Autowired
	private UtilsService utils;

	@Autowired
	private HttpServletRequest req;
	
	private static final String PAGE_SIZE = "pageSize";
	private static final String PAGE_NO = "pageNo";
	private static final String TASK_IDS = "taskIds";
	private static final String NODE_TYPES = "nodeTypes";

	/**
	 * workComment,workESign,workEmpId,workEmpName,workAgent,workOpUid,
	 * workHandover,workNote<br>
	 * 在pickCandidates()选新“候选人”时flowEngine.setNodeApprover()<br>
	 * 赋了4个值：workEmpId(empId)、workEmpName、workAgent、workOpUid(empUid)
	 */
	private static final String[] WORK_FIELDS = { Work.COMMENT, Work.ESIGN, Work.EMPID,
			Work.EMPNAME, Work.AGENT, Work.UID, Work.NOTE };// , Work.TRACE值默认为空

	/**
	 * 非完成的各种任务状态：0、1、4、5、6
	 */
//	private static final String[] INCOMPLETE = { FlowUtil.STATE_READY, FlowUtil.STATE_WAITING,
//			FlowUtil.STATE_REJECT, FlowUtil.STATE_CANCEL, FlowUtil.STATE_DELETE };

	/** 可删除的状态 */
	private static final String[] CAN_DELETE = { FlowUtil.STATE_CANCEL, FlowUtil.STATE_READY,
			FlowUtil.STATE_REJECT };

	/**
	 * 已审核状态包含：完成、驳回、移交
	 */
	private static final String[] WORK_DONE = { FlowUtil.STATE_COMPLETE, FlowUtil.STATE_REJECT,
			FlowUtil.STATE_WAITING };

	/**
	 * workOpUid\workAgent
	 */
	private static final String[] WORK_OP = { Work.UID, Work.AGENT };
	/**
	 * taskOpUid\taskSecty
	 */
	private static final String[] TASK_OP = { Task.UID, Task.SECTY };

	/**
	 * 公共：初始工作流（任务）的参数条件：默认操作员为“当前用户”或“秘书”
	 */
	Map<String, Object> initTaskParams() {
		// 初始化参数，设置当前操作员信息
		Map<String, Object> params = newTaskParams();
		if (!params.containsKey(Task.ID)) {
			// 以下是增加了taskMode参数的校验
			Object obj = params.get(Task.MODE);
			Map<String, String> mode = utils.obj2Map(obj); // 转map对象
			// 检查必须携带的参数
			String[] keys = ConfUtil.getValid(Task.MODE).split(ConfUtil.COMMA);
			if (!utils.availParams(mode, keys))
				throw new FailedException();
			// 设置流程标题
			params.put(Task.TITLE, mode.get(Mode.NAME));
			params.put(Task.MODE, utils.obj2Str(obj)); // 转字符串
		} else {
			// 任务重复保存发起时，taskMode保持不变，所以无需再更新
			params.remove(Task.MODE);
		}
		return params;
	}

	/**
	 * 初始任务参数，设置当前操作员信息
	 */
	Map<String, Object> newTaskParams() {
		Map<String, Object> params = utils.jsonParams(req);
		// 当前用户或“秘书”
		Map<String, String> user = utils.userSelf(req);
		// 增加暂时参数empSecty，默认当前用户或“秘书”
		params.put(Emp.SECTY, user.get(User.ID));
		params.put(Task.COMP, user.get(User.COMP));
		return params;
	}

	/**
	 * 批量保存审批过程中的节点，并更新节点工作状态
	 */
	Map<String, String> saveWorkNodes(Map<String, Object> params,
			List<Map<String, String>> workNodes) {
		if (workNodes.isEmpty())
			throw new FailedException(FlowUtil.IS_LOST_WAY);

		// 当前审批节点（workNodes）的基本信息：
		// workComp,workTaskId,workNodeNo,workState,workComment
		// ,workESign,workIdx..[,workEmpId,workAgent,workESign,workUpdate]
		List<Map<String, Object>> workItems = new ArrayList<>();

		// 当前提交审批的节点信息
		Object workId = params.get(Task.ID);
		Object workNo = params.get(Work.NODENO);
		// 前台选的审批人
//		Object selectee = params.get(FlowUtil.EMPLOYEE);
		// 当前审批节点的审批人
		Object approver = params.get(Work.EMPID);

		// 原始node数据来源于getFlowNodes()
		for (Map<String, String> node : workNodes) {
			node.remove(Work.UPDATE); // 更新日期不克隆

			// 为了克隆全部节点的属性（排除workUpdate）
			Map<String, Object> theNode = utils.map2Obj(node);
			String nodeNo = node.get(Work.NODENO);
			// 待审批为空（新工作任务），需要补充设置一个节点的基本属性
			if (nodeNo == null) {
				nodeNo =  node.get(Node.NO);
				// 这里是新建节点（但taskId可能新建或携带）
				theNode.put(Work.TASKID, workId);
				theNode.put(Work.NODENO, nodeNo);
				theNode.put(Work.COMP, node.get(Node.COMP));
				// 克隆数据，新建默认改为“0”，历史已自增“1++”
				theNode.put(Work.IDX, Const.S_0);
				// 其中workState已赋值
			}

			// 加签、移交：新增节点处理，有个选中的人selectee
			// 追问（此处暂未处理）：新增节点处理，有个选中的节点编号
			/*if (nodeNo.equals(workNodeNo) && selectee != null && selectee.equals(workEmpId)) {
				theNode.remove(Work.CREATE); // 新增节点
				// 方便加签或移交的跟踪信息，WORK_FIELDS中要补充workNote\workHandover
				theNode.put(Work.NOTE, params.get(Work.COMMENT));
				theNode.put(Work.HANDOVER, theEmpId);
				workItems.add(theNode);
				continue;
			}*/

			// 非新增节点：一种情况workEmpId==null,一种情况workEmpId已设审批人
			// 需要补充完善节点信息：WORK_FIELDS，与getFlowNodes()返回值保持一致
			String workEmpId = node.get(Work.EMPID);
			for (String key : WORK_FIELDS) {
				// 提交审批时携带“审批人”“审批意见”等页面数据如：workComment,workESign
				// 刚发起任务时 theEmpId为空
				if (nodeNo.equals(workNo) && approver != null && approver.equals(workEmpId)) {
					// 保留原node数据：workEmpId,workEmpName,workAgent,workOpUid
					if (params.containsKey(key))
						theNode.put(key, params.get(key));
					else if (node.get(key) == null)
						theNode.put(key, "");
				} else if (node.get(key) == null) {
					// 新设候选人、及其他节点默认置为空
					theNode.put(key, "");
				}
				// 保留其他已有theNode属性
			}
			// 批量更新时，请勿携带的数据Work.TRACE默认为空
			String trace = node.get(Work.TRACE);
			if (trace == null) {
				// 为方便追踪，补充当前的审批信息
				Map<String, Object> map = new HashMap<>();
				map.put(Node.NO, params.get(Work.NODENO));
				map.put(Emp.ID, params.get(Work.EMPID));
				theNode.put(Work.TRACE, utils.obj2Str(map));
			}
			workItems.add(theNode);
		}
		log.debug("> saveWorkNodes...{}", workItems);

		if (utils.isFailed(aolai.batchAdd(null, Work.TABLE, workItems, null, true)))
			throw new FailedException(FlowUtil.DO_FAILED);
		return utils.success();
	}

	/**
	 * 添加一条新审批流程（任务）：如果携带taskState=0时，保存为草稿
	 * <p>
	 * 根据前台提交的数据params，过程中又补充了taskOpUid,taskSecty[,taskState]
	 */
	Map<String, String> addNewTask(Map<String, Object> params,
			Map<String, String> sponsor, String taskState) {
		// 参数：{taskFlowNo,taskEmpId,taskDept,taskTitle,taskFormData,..candidates}

		// 新发起一个流程，设置任务发起人的姓名
		params.put(Task.EMPNAME, sponsor.get(Emp.NAME));
		// 便于查询待办任务：
		// 这里必须把“taskOpUid”设置为发起人“empUid”
		params.put(Task.UID, sponsor.get(Emp.UID));
		params.put(Task.STATE, taskState);

		// 登录用户“empSecty”可能是“发起人”，也可能“秘书”发起
		Object userId = params.get(Emp.SECTY);
		// 如果由“秘书”发起，则“秘书”可以操作，否则“秘书”不可操作
		// 如果非“秘书”发起，则“秘书”不可操作
		if (userId.equals(sponsor.get(Emp.SECTY)))
			params.put(Task.SECTY, userId);

		log.debug("> addNewTask...{}", params);
		return aolai.addRecord(null, Task.TABLE, params, true);
	}

	/**
	 * 根据taskId、taskComp获取工作流程审批（任务）信息
	 */
	Map<String, String> getTaskInfo(Map<String, Object> where) {
		// taskComp,taskId
		Map<String, Object> cond = utils.sameId(where, Task.KEY);
		return aolai.findOne(null, Task.TABLE, cond, "getTaskInfo");
	}

	/**
	 * 获取当前用户（指定节点）工作流程审批（任务）信息taskId,workNodeNo
	 */
	Map<String, String> getReadyTask(Map<String, Object> params) {
		// 默认查询就绪“0”的待办任务
		return getReadyTask(params, FlowUtil.STATE_READY);
	}

	/**
	 * 获取当前用户（指定节点）工作流程审批（任务）信息taskId,workNodeNo
	 */
	Map<String, String> getReadyTask(Map<String, Object> params, String workState) {
		// taskId,workNodeNo
		Map<String, Object> where = utils.sameId(params, Task.KEY);
		where.put(Work.NODENO, params.get(Work.NODENO));
		where.put(Work.STATE, workState); // 默认查询就绪“0”

//		String uid = utils.sqlVal(params.get(Work.AGENT));
		// 注意：WORK_OPUID为审批人登录账号，或“委托代理人”
		Object uid = params.get(Work.AGENT);
		Object[] values = { uid, uid };
		// 权限校验用，有权限的用户才能查询到数据
		where.put(utils.sqlOr(WORK_OP), values);
		Map<String, Map<String, String>> tables = workTables(where);
		Map<String, String> sort = new HashMap<>();
		sort.put(Task.ID, "ASC");
		List<Map<String, String>> list = aolai.findAll(null, tables,
				"getTaskList", where, sort);
		if (list.isEmpty())
			throw new FailedException(ConfUtil.DENY_ACCESS); // 无权限操作

		// 待办任务
		Map<String, String> task = list.get(0);
		// 这里的workEmpId用作在findNextNode()中鉴权
		// 流程发起人taskEmpId 不等于审批人workEmpId
		params.put(Work.EMPID, task.get(Work.EMPID));
		// 当前用户提交的工作流
		params.put(Task.FLOWNO, task.get(Task.FLOWNO));
		// 返回当前任务信息
		return task;
	}

	/**
	 * 更新工作流程（任务）状态：“完成”或“驳回”
	 */
	Map<String, String> setTaskState(Map<String, Object> params) {
		/**
		 * 抄送知会某些人
		 */
		batchInformSb(params);

		// taskId\taskComp
		Map<String, Object> where = utils.sameId(params, Task.KEY);
		// 限定条件：状态=0\1草稿或等待中的，才可以更新，
		where.put(utils.pHolder(Task.STATE, "<"), FlowUtil.STATE_COMPLETE);
		// 更新任务状态
		return aolai.update(null, Task.TABLE, params, where);
	}

	/**
	 * 抄送知会某些人
	 */
	private void batchInformSb(Map<String, Object> params) {
		List<Map<String, String>> empList = utils.obj2List(params.get(FlowUtil.INFORM_SB));
		if (empList.isEmpty())
			return;
		List<Map<String, Object>> items = new ArrayList<>();
		for (Map<String, String> emp : empList) {
			// taskId,workNodeNo,taskComp
			Map<String, Object> item = new HashMap<>();
			item.put(Inform.COMP, params.get(Task.COMP));
			item.put(Inform.TASKID, params.get(Task.ID));
			item.put(Inform.NODENO, params.get(Task.NODENO));
			item.put(Inform.EMPID, emp.get(Emp.ID));
			item.put(Inform.EMPUID, emp.get(Emp.UID));
			items.add(item);
		}
		if (utils.isFailed(aolai.batchAdd(null, Inform.TABLE, items, params.get(Const.I18N), true)))
			throw new FailedException();
		params.put(Task.INFORM, Const.S_1);
	}

	/**
	 * 会签“父”节点jointlyNode!=null、未签节点却不存在（为“零”）时，则并发冲突
	 */
	boolean jointlyCompleted(Map<String, String> jointlyNode) {
		if (jointlyNode == null)
			return false;
		Map<String, Object> where = workIf(jointlyNode);
//		where.put(utils.pHolder(Work.STATE, "<>"), FlowUtil.STATE_COMPLETE);
		// 是否还有没完成的节点状态，不存在需要重新处理
		String[] values = { FlowUtil.STATE_READY, FlowUtil.STATE_WAITING };
		where.put(pHolderIn(Work.STATE), values);

		// 会签节点编号
		String nodeNo = jointlyNode.get(Node.NO);
		// 会签节点类型，则不含父节点自己
		if (FlowUtil.STYLE_JOINTLY.equals(jointlyNode.get(Node.STYLE))) {
			where.put(utils.pHolder(Work.NODENO, Const.LIKE), nodeNo + "%");
			where.put(utils.pHolder(Work.NODENO, Const.NEQ), nodeNo);
		} else {
			where.put(utils.pHolder(Work.NODENO, Const.EQU), nodeNo);
		}

		// 不存在即“会签”完成，需要重新处理
		return !aolai.exists(null, Work.TABLE, where);
	}

	/**
	 * 检索条件转换：workComp、workTaskId
	 */
	<T> Map<String, Object> workIf(Map<String, T> params) {
		Map<String, Object> where = new HashMap<>();
		where.put(Work.COMP, params.get(Task.COMP));
		where.put(Work.TASKID, params.get(Task.ID));
		return where;
	}

	/**
	 * 根据taskId取消一个审批中的任务，取消“审批中”的审批流程
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> cancelTask() {

		// 有审批权的用户才能执行：taskComp、taskOpUid、taskSecty
		Map<String, Object> where = taskParams();
		Object taskState = where.get(Task.STATE);
		where.put(Task.STATE, FlowUtil.STATE_WAITING); // 1.等待中的任务

		// 有审批权的用户才能执行
		Object userId = req.getAttribute(User.ID);
		Map<String, Object> cond = workIf(where);
//		String val = utils.sqlVal(userId);
		Object[] values = { userId, userId };
		cond.put(utils.sqlOr(WORK_OP), values);
		cond.put(Work.STATE, FlowUtil.STATE_READY);

		// 将要删除的数据必须存在才行
		if (aolai.exists(null, Task.TABLE, where) || aolai.exists(null, Work.TABLE, cond)) {
			// 先取消当前任务
			Map<String, Object> fields = new HashMap<>();
			// 删除作废状态为“6”、取消任务状态为：“5”
			taskState = FlowUtil.STATE_DELETE.equals(taskState) ? taskState : FlowUtil.STATE_CANCEL;
			fields.put(Task.STATE, taskState); // 5.取消 6.删除
			// 更新状态
			aolai.update(null, Task.TABLE, fields, utils.sameId(where, Task.KEY));

			// 再撤销审批中的节点，把所有审批中“就绪”节点改为：“5”
			fields.put(Work.STATE, FlowUtil.STATE_CANCEL); // 5.取消
			// 设置当前节点的操作人是当前用户本人
			fields.put(Work.UID, userId);
			// 获取当前操作员工编号

			// 再取消审批中的节点
			Map<String, Object> cond0 = workIf(where);
			cond0.put(Work.STATE, FlowUtil.STATE_READY); // 0.就绪节点
			// 执行取消
			aolai.update(null, Work.TABLE, fields, cond0);

			// 取消成功
			return utils.success();
		}
		// 取消失败
		return utils.invalidParams();
	}

	/**
	 * 撤销上次执行的审批任务：taskId,workNodeNo
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> undoTask() {
		// 先补充参数：taskComp\workAgent
		Map<String, Object> params = getWorkParams();
		// 审批人撤回的情况
		if (params.containsKey(Work.NODENO))
			return undoWork(params);

		// 发起人撤回的处理：
		// 删除工作中的“就绪”节点
		deleteWorking(params);
		// 撤销：更新任务的状态
		Map<String, Object> fields = new HashMap<>();
		fields.put(Task.STATE, FlowUtil.STATE_READY);

		if (utils.isFailed(aolai.update(null, Task.TABLE, fields, taskParams())))
			throw new FailedException();
		return utils.success();
	}

	/**
	 * 撤销上次执行的审批任务：taskId,workNodeNo
	 */
	public Map<String, String> undoWork(Map<String, Object> params) {
		// 再读取办结的任务节点信息, 并补充params参数：workEmpId、taskFlowNo
		Map<String, String> task = getReadyTask(params, FlowUtil.STATE_COMPLETE);

		// 修改当前的节点状态：从完成“2”->>改变为“0”就绪
		Map<String, Object> fields = new HashMap<>();
		fields.put(Work.STATE, FlowUtil.STATE_READY);

		// 刚刚审批通过的：workComp、workTaskId、workEmpId、workNodeNo
		Map<String, Object> cond = workIf(params);
//		String val = utils.sqlVal(params.get(Work.EMPID));
//		String[] values = { val, val };
//		cond.put(utils.sqlOr(WORK_OP), values);
		cond.put(Work.EMPID, params.get(Work.EMPID));
		cond.put(Work.NODENO, params.get(Work.NODENO));
		cond.put(Work.STATE, FlowUtil.STATE_COMPLETE);

		// 执行撤回操作
		if (utils.isFailed(aolai.update(null, Work.TABLE, fields, cond)))
			return utils.invalidParams();

		// 会签节点，全批过去后就不允许撤回了
		if (FlowUtil.ISSUE_FULL.equals(task.get(Node.ISSUE))
				&& getNodeApprovers(params, FlowUtil.STATE_READY).size() == 1) {
			// 理论上撤回一条数据后，至少有两条及以上数据，否则是不允许撤回的，需要事务回滚
			throw new FailedException(FlowUtil.LOST_CHANCE);
		}
		deleteWorking(params);
		return utils.success();
	}

	/**
	 * 删除所有未处理“就绪”的审批节点
	 */
	private void deleteWorking(Map<String, Object> params) {
		// 删除所有未处理“就绪”、审批人不能为空的数据，且经本节点流转的数据
		Map<String, Object> cond = workIf(params);
		cond.put(utils.pHolder(Work.EMPID, Const.NEQ), "");
		if (params.containsKey(Work.EMPID))
			cond.put(utils.jsonExt(Work.TRACE, Emp.ID), params.get(Work.EMPID));
		if (params.containsKey(Work.NODENO))
			cond.put(utils.jsonExt(Work.TRACE, Node.NO), params.get(Work.NODENO));
		cond.put(Work.STATE, FlowUtil.STATE_READY);

		// 如果无有效数据可删除，则也不允许撤回了
		if (utils.isFailed(aolai.delete(null, Work.TABLE, cond)))
			throw new FailedException(FlowUtil.LOST_CHANCE);

		// 如果还有非“就绪”数据，需要事务回滚
		cond.put(utils.pHolder(Work.STATE, Const.NEQ), FlowUtil.STATE_READY);
		cond.remove(Work.STATE);
		if (aolai.exists(null, Work.TABLE, cond))
			throw new FailedException(FlowUtil.LOST_CHANCE);
	}

	/**
	 * 公共：设置工作流任务的参数条件：taskComp\workAgent
	 */
	Map<String, Object> getWorkParams() {
		Map<String, Object> params = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		params.put(Task.COMP, user.get(User.COMP));
		// 根据taskId、workNodeNo获取审批节点及表单信息
		// 这里userId可能是“代理人”
		params.put(Work.AGENT, user.get(User.ID));
		return params;
	}

	/**
	 * 我的待办（两种视角）、已办（审批人视角）、办结事项（发起人视角）<br>
	 * <br>
	 * 审批人视角：查询workState=“0”待办的节点（含“代理人”视角）
	 * <p>
	 * 选项：taskUrgent根据紧急状态查询，taskInform根据抄送状态查询<br>
	 * 如果是我的办结（含驳回的和已完成的taskState=2、4）
	 */
	public Map<String, Object> getTaskList() {
		// workState,taskState[,taskSdate,taskEdate]
		Map<String, Object> params = utils.jsonParams(req);
		// 查询任务表的组合条件
		Map<String, Object> where = workTaskCond(params);
		// 查询任务表的组合条件，再补充条件，如：taskUrgent、taskInform
		Map<String, Map<String, String>> tables = workTaskCond(params, where);

		// 按“发起时间”倒排序，最新的任务在上面
		Map<String, String> order = new LinkedHashMap<>();
		order.put(Task.CREATE, "DESC"); // 按发起时间倒序

		// 根据状态（workState,taskState）查询的结果
		log.debug("> getTaskList...{}", tables);
		Map<String, Object> result = aolai.queryList(null, tables,
				"getTaskList", where, order, ConfUtil.limitRows());

		// 分页数据
		return utils.success(result);
	}

	/**
	 * 我的待办（两种视角）、已办（审批人视角）、办结事项（发起人视角）的查询条件
	 */
	private Map<String, Object> workTaskCond(Map<String, Object> params) {
		// 分页
		Map<String, Object> where = new HashMap<>();
		if (params.containsKey(PAGE_SIZE)) {
			where.put(PAGE_NO, params.get(PAGE_NO));
			where.put(PAGE_SIZE, params.get(PAGE_SIZE));
		}

		if (!utils.isEmpty(params.get(TASK_IDS))) {
			where.put(pHolderIn(Task.ID), params.get(TASK_IDS));
		}
		// 节点类型，支持传递数组格式的参数
		if (params.containsKey(Node.TYPE)) {
			String nodeType = String.valueOf(params.get(Node.TYPE));
			where.put(pHolderIn(Node.TYPE), nodeType.split(Const.COMMA));
		} else if (!utils.isEmpty(params.get(NODE_TYPES))) {
			where.put(pHolderIn(Node.TYPE), params.get(NODE_TYPES));
		}
		// 拓展支持json表达式，把参数传入where条件中
		putJsonExtParams(params, where);
		return where;
	}

	/**
	 * 获取抄送人员能查看的任务一览：
	 */
	public Map<String, Object> getInformTasks() {
		// 仅靠taskInform此参数区分按抄送人查询
		return getTaskList();
	}

	/**
	 * 获取审批流程中的抄送信息一览：informTaskId
	 */
	public Map<String, Object> getInformList() {
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Inform.COMP);
		Map<String, Map<String, String>> tables = new HashMap<>();
		tables.put(Inform.TABLE, null);
		Map<String, String> cond = new HashMap<>();
		cond.put(Emp.COMP, Inform.COMP);
		cond.put(Emp.ID, Inform.EMPID);
		tables.put(Emp.TABLE, cond);
		return utils.success(aolai.findAll(null, tables, "getInformList", where, null));
	}

	/**
	 * 拓展支持json表达式参数传递，注意此参数非前端传入，而是saor作为jar注入到父工程，父工程组装好传入，没有sql注入风险。
	 * 前端传入，拦截器会拦截到，没有sql注入风险
	 * @param params
	 * @param where
	 */
	private void putJsonExtParams(Map<String, Object> params, Map<String, Object> where) {
		// 支持两种JSON类型数据： taskFormData\taskMode
//		String[] fields = { Task.FORMDATA, Task.MODE };
//		String jsonExtKey = "json_extract";
		for (Map.Entry<String, Object> e : params.entrySet()) {
			if (e.getKey().contains("json_extract")) {
				where.put(e.getKey(), e.getValue());
			}
		}
	}

	/**
	 * 我的待办（两种视角）、已办（审批人视角）、办结事项（发起人视角）的查询条件
	 */
	private Map<String, Map<String, String>> workTaskCond(Map<String, Object> params,
			Map<String, Object> where) {
		// 当前登录用户、“代理人”或“秘书”岗位职责
		Map<String, String> user = utils.userSelf(req);
		where.put(Task.COMP, user.get(User.COMP));
//		if (params.containsKey(Task.URGENT))
			where.put(Task.URGENT, params.get(Task.URGENT));

		// 参数必须携带其中一个参数
		Object taskState = params.get(Task.STATE);
		Object workState = params.get(Work.STATE);
		if (utils.isEmpty(taskState) && utils.isEmpty(workState))
			throw new FailedException();

//		String uid = utils.sqlVal(user.get(User.ID));
		String uid = user.get(User.ID);
		String[] values = { uid, uid };

		// 审批节点上的审批人视角：查询workState=“0”待办的节点（含“代理人”视角）
		if (utils.isEmpty(taskState)) {
			// 待办任务：等待中
			if (FlowUtil.STATE_READY.equals(workState)) {
				// 0.就绪
				where.put(Work.STATE, FlowUtil.STATE_READY);
				// 同时满足条件：taskState=“1”等待中
				where.put(Task.STATE, FlowUtil.STATE_WAITING); // 1.等待中

			} else if (FlowUtil.STATE_COMPLETE.equals(workState)) {
				// 已完成的需要有个开始时间（什么时间处理的）
				setTaskPeriod(params, where, Work.UPDATE);

				// 已办状态：2.完成 4.驳回 1.移交（或追问、前加签中的）
//				String[] vs = { FlowUtil.STATE_COMPLETE, FlowUtil.STATE_REJECT };
				where.put(pHolderIn(Work.STATE), WORK_DONE);
//				where.remove(Work.STATE);

			} else if (!utils.isEmpty(workState)) {
				// workState=4.驳回，需要有个开始时间（什么时间驳回的）
				setTaskPeriod(params, where, Work.UPDATE);

			} else {
				// 其他情况参数无效
				throw new FailedException();
			}
			// 必须是普通节点
			where.put(Node.STYLE, FlowUtil.STYLE_NORMAL); // 0.普通

			// 审批人岗位职责、或“代理”岗位职责
			where.put(utils.sqlOr(WORK_OP), values);

			// 以WORK为主表
			return workTables(where);

		} else {
			// 发起流程的发起人视角：查询taskState=“0”
			if (FlowUtil.STATE_READY.equals(taskState)) {
				// 未完成的各种状态： 0 1 4 5 6
//				where.put(pHolderIn(Work.STATE), INCOMPLETE);
				where.put(utils.pHolder(Work.STATE, Const.NEQ), FlowUtil.STATE_COMPLETE);

			} else if (Const.ALL.equals(taskState)) {
				// 已完成的需要有个开始时间
				setTaskPeriod(params, where, Task.CREATE);

			} else if (!utils.isEmpty(taskState)) {
				// 单条件状态查询：传入什么状态，查询什么状态。
				where.put(Task.STATE, taskState);
				setTaskPeriod(params, where, Task.CREATE);
			} else {
				throw new FailedException();
			}

			// 仅靠taskInform此参数区分按抄送人查询
			// 只能查询抄送的，抄送的不关心发起人
			if (Const.S_1.equals(params.get(Task.INFORM))) {
				where.put(Inform.EMPUID, user.get(User.ID));
				return informTables(where);
			} else {
				// “发起人”或“秘书”
				where.put(utils.sqlOr(TASK_OP), values);
				// 以TASK为主表
				return taskTables(where);
			}
		}
	}

	private String pHolderIn(String key) {
		return utils.pHolder(key, Const.IN);
	}

	/**
	 * 检索任务的日期条件
	 */
	private void setTaskPeriod(Map<String, Object> params, Map<String, Object> where, String time) {
		String startDate = (String) params.get("startDate");
		String endDate = (String) params.get("endDate");
		// 默认三个月之内的数据
		if (!utils.isDatetime(startDate, Const.SDF))
			startDate = utils.prevMonth("", 3) + "01";
		else
			startDate += " 00:00:00";
		where.put(utils.pHolder(time, ">="), startDate);
		if (utils.isDatetime(endDate, Const.SDF)) {
			where.put(utils.pHolder(time, "<="), endDate + " 23:59:59");
		} else if (endDate != null) {
			throw new FailedException();
		}
		log.debug("setTaskPeriod...{}", where);
	}

	/**
	 * 重新梳理的审批节点查询条件
	 */
	private Map<String, Map<String, String>> workTables(Map<String, Object> where) {
		// 根据请求参数查询数据并返回查询结果
		// 注意这里用LinkedHashMap确保三个关联表的顺序
		Map<String, Map<String, String>> tables = new LinkedHashMap<>();
		String taskComp = (String) where.get(Task.COMP);
		// “审批节点”表
		tables.put(Work.TABLE, null);
		// 关联“工作任务”表
		Map<String, String> cond1 = new HashMap<>();
		cond1.put(Task.COMP, taskComp);
		cond1.put(Task.ID, Work.TASKID); // 对应一条对应任务
		tables.put(Task.TABLE, cond1);

		// 关联“流程节点”名称
		// 有参数workNodeNo时，nodeStyle参数传递null
		return workJoinNode(taskComp, tables, (String) where.get(Node.STYLE));
	}

	/**
	 * 重新梳理的抄送相关流程的查询条件
	 */
	private Map<String, Map<String, String>> informTables(Map<String, Object> where) {
		// 根据请求参数查询数据并返回查询结果
		// 注意这里用LinkedHashMap确保三个关联表的顺序
		Map<String, Map<String, String>> tables = new LinkedHashMap<>();
		// “抄送知会人”表
		tables.put(Inform.TABLE, null);
		Map<String, String> cond = new HashMap<>();
		cond.put(Task.COMP, Inform.COMP);
		cond.put(Task.ID, Inform.TASKID);
		// “抄送知会人”关联“工作任务”表
		tables.put(Task.TABLE, cond);

		// “工作任务”关联“审批节点”表
		String taskComp = (String) where.get(Task.COMP);
		taskJoinWork(taskComp, tables,Inform.EMPID);
		// “审批节点”关联“流程节点”表
		return workJoinNode(taskComp, tables, (String) where.get(Node.STYLE));
	}

	/**
	 * 重新梳理的审批节点查询条件
	 */
	private Map<String, Map<String, String>> taskTables(Map<String, Object> where) {
		// 根据请求参数查询数据并返回查询结果
		// 注意这里用LinkedHashMap确保三个关联表的顺序
		Map<String, Map<String, String>> tables = new LinkedHashMap<>();
		// “工作任务”表
		tables.put(Task.TABLE, null);

		// 关联“审批节点”表
		String taskComp = (String) where.get(Task.COMP);
		taskJoinWork(taskComp, tables);

		// 关联“流程节点”表
		return workJoinNode(taskComp, tables, (String) where.get(Node.STYLE));
	}
	/**
	 * 流程任务表关联工作节点
	 */
	private void taskJoinWork(String workComp, Map<String, Map<String, String>> tables) {
		taskJoinWork(workComp,tables,null);
	}
	/**
	 * 流程任务表关联工作节点,增加workNode 关联员工EmpId
	 */
	private void taskJoinWork(String workComp, Map<String, Map<String, String>> tables,String empIdKey) {
		Map<String, String> cond1 = new HashMap<>();
		cond1.put(Work.COMP, workComp);
		// 只有一个对应节点
		cond1.put(Work.NODENO, Task.NODENO);
		cond1.put(Work.TASKID, Task.ID);
		// 默认选择审批次序必须为“0”的待审批节点
		cond1.put(Work.IDX, Const.S_0);
		if(empIdKey != null) {
			cond1.put(Work.EMPID, empIdKey);
		}
		// “审批节点”表（workIdx=‘0’）
		tables.put(Work.TABLE, cond1);
	}

	/**
	 * 工作节点表关联节点定义
	 */
	private Map<String, Map<String, String>> workJoinNode(String nodeComp,
			Map<String, Map<String, String>> tables, String nodeStyle) {
		// 关联“流程节点”表
		Map<String, String> cond2 = new HashMap<>();
		cond2.put(Node.COMP, nodeComp);
		cond2.put(Node.STYLE, nodeStyle);
		cond2.put(Node.NO, Work.NODENO); // 当前任务节点
		cond2.put(Node.FLOWNO, Task.FLOWNO); // 只有一个对应流程
		// “流程节点”表
		tables.put(Node.TABLE, cond2);
		return tables;
	}

	/**
	 * 单独获取当前选中任务的表单信息
	 */
	public Map<String, Object> getTaskInfo() {
		Map<String, Object> where = taskParams();
		if (aolai.exists(null, Task.TABLE, where)) {
			// 因为where条件中含有{taskOpUid}={V} or {taskSecty}={V}参数
			// 所以这里使用findOne时，去除一个参数
			Map<String, Object> cond = utils.sameId(where, Task.KEY);
			return utils.success(aolai.findOne(null, Task.TABLE, cond));
		}
		throw new FailedException();
	}

	/**
	 * 删除一个草稿（已撤回）工作任务（流程）
	 * <p>
	 * 限定条件：状态=5.已撤回的 0.草稿中的，才可以删除
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> delTask() {
		Map<String, Object> params = taskParams();
		Map<String, Object> where = utils.sameId(params, Task.KEY);
		// 可删除任务：主动撤回任务、草稿任务、被驳回任务
		where.put(pHolderIn(Task.STATE), CAN_DELETE);
		// 将要删除的数据必须存在
		if (!aolai.exists(null, Task.TABLE, where))
			return utils.invalidParams(); // 无效参数

		// 先删除任务
		aolai.delete(null, Task.TABLE, utils.sameId(where, Task.KEY));
		// 再删除过程中的审批节点
		aolai.delete(null, Work.TABLE, workIf(where));
		// 返回成功
		return utils.success();

	}

	/**
	 * 公共：工作流任务的参数条件
	 */
	private Map<String, Object> taskParams() {
		Map<String, Object> where = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		where.put(Task.COMP, user.get(User.COMP));
		// 根据taskId获取审批节点，这里userId可能是“代理人”
//		String uid = utils.sqlVal(user.get(User.ID));
		String uid = user.get(User.ID);
		String[] values = { uid, uid };
		// 仅允许当前用户或“秘书”可以撤销：taskOpUid、taskSecty
		where.put(utils.sqlOr(TASK_OP), values);
		return where;
	}

	/**
	 * 获取审批流中的普通审批节点一览，查询条件：workTaskId
	 */
	public Map<String, Object> getWorkNodes() {
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Work.COMP);
		where.put(utils.pHolder(Work.EMPID, Const.NEQ), "");
		// 优先以更新时间排序
		Map<String, String> order = new LinkedHashMap<>();
		order.put(Work.UPDATE, "DESC");
		order.put(Work.CREATE, "DESC");
		// 根据条件查询数据并返回结果
		String[] args = { null, Work.TABLE, null, Const.S_0 };
		return utils.success(aolai.findAll(where, order, args));
	}

	/**
	 * 检索任务节点的完成审批的审批人：taskComp、taskId、choseNodeNo
	 */
/*	public Map<String, Object> getNodeApprovers() {
		Map<String, Object> params = utils.jsonParams(req);
		utils.setUserComp(req, params, Task.COMP);
		return utils.success(getNodeApprovers(params));
	}*/

	/**
	 * 根据条件检索任务节点信息，条件：workComp、workTaskId、workNodeNo
	 */
	List<Map<String, String>> getNodeApprovers(Map<String, Object> params) {
		return getNodeApprovers(params, FlowUtil.STATE_COMPLETE);
	}

	/**
	 * 根据条件检索任务节点信息，条件：taskComp、taskId、choseNodeNo
	 */
	List<Map<String, String>> getNodeApprovers(Map<String, Object> params, String workState) {
		Map<String, Object> where = workIf(params);
		where.put(Work.NODENO, params.get(Chose.NODENO));
		where.put(Work.STATE, workState); // 默认已完成的
		where.put(Work.IDX, 0); // 当前审批过的所有人
		return aolai.findList(Work.TABLE, where);
	}

	/**
	 * 保存预选好全部节点审批人：items: [{choseTaskId,choseNodeNo,choseEmpId }, ..]
	 */
	public Object saveApprovers() {
		Map<String, Object> params = utils.jsonParams(req);
		List<Map<String, Object>> items = utils.obj2List(params.get(Const.ITEMS));
		if (items.isEmpty())
			utils.invalidParams();
		String[] keys = { Chose.TASKID, Chose.NODENO, Chose.EMPID };
		String compId = utils.setUserComp(req, params, Chose.COMP);
		for (Map<String, Object> item : items) {
			if (!utils.availParams(item, keys))
				utils.invalidParams();
			item.put(Chose.COMP, compId);
			item.put(Chose.STATE, Const.S_1);
		}
		return aolai.batchAdd(null, Chose.TABLE, items, params.get(Const.I18N), true);
	}

	/**
	 * 读取预选好某节点上的审批人信息<br>
	 * choseComp,choseTaskId,choseNodeNo,choseEmpId,choseState,<br>
	 * empComp,empId,empName,empUid,empAgent,empMobi,empMail,empWechat
	 */
	List<Map<String, String>> choseApprovers(Map<String, String> node) {
		Map<String, Map<String, String>> tables = new HashMap<>();
		tables.put(Chose.TABLE, null); // 主表
		Map<String, String> cond0 = new HashMap<>();
		cond0.put(Emp.COMP, Chose.COMP);
		cond0.put(Emp.ID, Chose.EMPID);
		cond0.put(Emp.UID, node.get(Work.UID));
		tables.put(Emp.TABLE, cond0); // 关联表
		// 主表检索条件
		Map<String, Object> cond = new HashMap<>();
		cond.put(Chose.COMP, node.get(Node.COMP));
		cond.put(Chose.TASKID, node.get(Task.ID));
		cond.put(Chose.NODENO, node.get(Node.NO));
		cond.put(Chose.STATE, Const.S_1);
		Map<String, String> sort = new HashMap<>();
		sort.put(Emp.ID, "ASC");
		return aolai.findAll(null, tables, "choseApprovers", cond, sort);
	}

}
