package cn.ps1.soar.service;

import java.util.*;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.ps1.aolai.entity.Dept;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.service.AolaiService;
import cn.ps1.aolai.service.HttpsService;
import cn.ps1.aolai.service.UtilsService;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.soar.dao.FlowDao;
import cn.ps1.soar.entity.Biz;
import cn.ps1.soar.entity.Emp;
import cn.ps1.soar.entity.Orgn;
import cn.ps1.soar.entity.Rank;
import cn.ps1.soar.entity.Job;
import cn.ps1.soar.entity.Node;
import cn.ps1.soar.entity.Work;
import cn.ps1.soar.utils.FlowUtil;

/**
 * 公司内部组织架构维护管理
 * <p>
 * 组织（Orgn）|部门（Division）|员工（Employee）|岗位职责（Job）|职级（Rank）
 * 
 * @author Aolai
 * @version 1.0 $Date: 2023.12.30
 * @since openjdk-1.8
 */
@Service
public class OrgnService {

	private static Logger log = LoggerFactory.getLogger(OrgnService.class);

	private static final String GET_EMPS = "getEmpList";

	@Autowired
	private AolaiService aolai;
	@Autowired
	private HttpsService https;
	@Autowired
	private UtilsService utils;
	
	@Autowired
	private HttpServletRequest req;
	@Autowired
	private FlowDao flowDao;
	
	/**
	 * 同一个组织架构下，根据部门（orgnDept）查询员工\岗位职责
	 * <p>
	 * 注意区别“getEmpList”接口，其按“主要任职部门”查询
	 */
	public Map<String, Object> getEmpOfDept() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		// 这里通过orgnDept锁定多个员工
		utils.setUserComp(req, where, Orgn.COMP);

		// 先处理tables，再把where传入getEmpList()
		Map<String, Map<String, String>> tables = getOrgnEmpCond(where);
		// 查询条件：orgnComp、orgnDept，返回员工信息分页展示数据
		// 按姓名排序
		return getEmpList(tables, where, empByName());
	}

	/**
	 * 默认按姓名排序
	 */
	private Map<String, String> empByName() {
		// 默认按部门倒序，再按姓名排序
		Map<String, String> order = new LinkedHashMap<>();
		order.put(Emp.NAME, "ASC");
		return order;
	}

	/**
	 * 同一个组织架构下，根据用户（orgnEmpId或empUid等）查询员工及所在部门数据
	 */
	public Map<String, Object> getEmpInDept() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);

		// 无orgnEmpId参数时，默认是当前用户或“秘书”
		if (!where.containsKey(Orgn.EMPID) && !where.containsKey(Emp.ID)) {
			// 默认查询当前用户或“秘书”
			// 无empUid参数时，或有 empUid参数且等于当前“用户”时
			Object empUid = where.get(Emp.UID);
			String userId = user.get(User.ID);
			if (empUid == null || userId.equals(empUid))
				where.put(Emp.SECTY, userId);
		}

		// 这里通过用户编号锁定某个员工，返回员工信息及部门、岗位职责等信息
		where.put(Orgn.COMP, user.get(User.COMP));

		// 先处理tables，再把where传入getEmpList()
		Map<String, Map<String, String>> tables = getOrgnEmpCond(where);
		// 返回发起人信息
		return getEmpList(tables, where, sortDept());
	}

	/**
	 * 根据岗位（orgnJob）查询用户及所在部门数据
	 */
	public Map<String, Object> getEmpByJob() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Orgn.COMP);

		// 先处理tables，再把where传入getEmpList()
		Map<String, Map<String, String>> tables = getOrgnEmpCond(where);
		// 返回发起人信息
		return getEmpList(tables, where, sortDept());
	}

	/**
	 * 默认按部门倒序，再按姓名排序
	 */
	private Map<String, String> sortDept() {
		// 默认按部门倒序，再按姓名排序
		Map<String, String> order = new LinkedHashMap<>();
		order.put(Dept.ID, "DESC");
		order.put(Emp.NAME, "ASC");
		return order;
	}

	/**
	 * 同一个组织架构下，根据用户（orgnEmpId或empUid）或岗位职责（orgnJob）查询员工及所在部门
	 * <p>
	 * empComp,empUid,empId,empName,empDept,empMobi,empCity,empGender,empWorkId,
	 * empIdcard,empLeader,empRank,empMail,empAgent,empSecty,empAccNo,empState,
	 * empDepart,orgnComp,orgnEmpId,orgnDept,orgnJobs,deptComp,deptId,deptName,
	 * deptType,deptTier,deptCost
	 */
	List<Map<String, String>> getEmpBy(Map<String, Object> where) {
		// 查询条件：根据orgnJobs->>'$.“orgnJob”'查询
		// 注意有些员工有重复分布在多个部门的情况
		Map<String, Map<String, String>> tables = getOrgnEmpCond(where);
		Map<String, String> order = sortDept();
		// 员工数据一览
		return aolai.findAll(null, tables, GET_EMPS, where, order, null);
	}

	/**
	 * 增加一条员工的组织数据
	 */
	public Map<String, String> addOrgn() {
		// 请求参数、当前用户信息
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户所属公司
		data.put(Orgn.COMP, user.get(User.COMP));
		// 当前操作用户
		data.put(Orgn.OPUID, user.get(User.ID));
		// 覆盖旧数据的状态
		data.put(Orgn.STATE, Const.S_1);
		// 注意前台参数为：empJobs
		String jobs = null;
		if (data.containsKey(Orgn.JOBS))
			jobs = utils.obj2Str(data.get(Orgn.JOBS));
		data.put(Orgn.JOBS, jobs == null ? "{}" : jobs);

		// 理论上可以重复添加
		return aolai.addRecord(Orgn.TABLE, data, true);
	}

	/**
	 * 删除一条员工的组织数据，条件：orgnComp,orgnEmpId,orgnDept
	 */
	public Map<String, String> delOrgn() {
		Map<String, Object> where = utils.jsonParams(req);
		// 设置当前操作用户所属公司
		utils.setUserComp(req, where, Orgn.COMP);
		return delOrgn(where);
	}

	/**
	 * 删除一条员工的组织数据，条件：orgnComp,orgnEmpId
	 */
	private Map<String, String> delOrgn(Map<String, Object> where) {
//		return aolai.delete(null, Orgn.TABLE, where);
		Map<String, Object> fields = new HashMap<>();
		fields.put(Orgn.STATE, Const.S_0);
		return aolai.update(null, Orgn.TABLE, fields, where);
	}

	/** 部门 */

	/**
	 * 根据条件查询所有部门数据
	 */
	public Map<String, Object> getDeptList() {

		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		// 获取用户信息，公司编号在用户信息中取
		Object userComp = utils.setUserComp(req, where, Dept.COMP);
		List<Map<String, String>> items = getDeptList(where);

		// 返回查询结果
		return utils.success(deptGroupCount(items, userComp));
	}

	/**
	 * 根据条件获取部门列表
	 */
	List<Map<String, String>> getDeptList(Map<String, Object> where) {
		// 父级ID
		Object pid = where.get(Dept.PID);
		if (!utils.isEmpty(pid)) {
			where.put(utils.pHolder(Dept.ID, "LIKE"), pid + "%");
			where.remove(Dept.PID);
		}
		// 正则匹配名称
		FlowUtil.likeCond(where, Dept.NAME);
		// 设置一个默认值状态值
		utils.resetState(where, Dept.STATE);
		// 按升序排序
		Map<String, String> order = new HashMap<>();
		order.put(Dept.ID, "ASC");
		// 过滤已删除的部门 0 代表已删除
		if(utils.isEmpty(where.get(Dept.STATE)))  {
			where.put(utils.pHolder(Dept.STATE, Const.NEQ), Const.S_0);
		}

		// 根据参数查询，并返回数据
		String[] args = { null, FlowUtil.deptTable(), "getDivnList", Const.S_0 };
		return aolai.findAll(where, order, args);
	}

	/**
	 * 归集每个部门包含的员工数量
	 */
	private List<Map<String, String>> deptGroupCount(
			List<Map<String, String>> items, Object deptComp) {
		List<Map<String, String>> list;
		if (!items.isEmpty()) {
			Map<String, Object> cond = new HashMap<>();
			cond.put(Emp.COMP, deptComp);
			cond.put(Emp.STATE, Const.S_1); // 在职员工

			String[] keys = { Emp.DEPT };
			list = aolai.groupCount(null, Emp.TABLE, keys, cond);
			Map<String, String> tmp = utils.list2Map(list, Emp.DEPT, "CNT");
			for (Map<String, String> item : items) {
				String cnt = tmp.get(item.get(Dept.ID));
				item.put("deptCount", cnt != null ? Const.S_0 : cnt);
			}
		}
		return items;
	}

	/**
	 * 增加一条部门数据
	 */
	public Map<String, String> addDept() {

        Map<String, Object> params = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户所属公司，公司编号在用户信息中取
		params.put(Dept.COMP, user.get(User.COMP));
		params.put(Dept.OPUID, user.get(User.ID));

		// 检查同一公司内同一层级内部门名称不能重复
		// 注意：必须把主键Dept.ID放在第一个参数中
		String[] keys = { Dept.ID, Dept.COMP, Dept.PID, Dept.NAME };
		if (aolai.exists(FlowUtil.deptTable(), utils.sameIf(params, keys)))
			throw new FailedException(ConfUtil.DUPL_NAME);

		// 注意这里确保解析器各字段的顺序一致
		String[] parser = { FlowUtil.deptTable(), Dept.COMP, Dept.ID, Dept.PID,
				Dept.LEAF, Dept.LEVEL, Dept.OPUID };

		// 根据层级宽度，新插入字符串
		params.put(Dept.STATE, Const.S_1);
		return aolai.addTreeNode(params, parser, FlowUtil.leafW());
	}

	/**
	 * 修改一条部门数据
	 */
	public Map<String, String> setDept() {

		// 请求参数、当前操作用户
		Map<String, Object> data = utils.jsonParams(req);
		// 允许修改状态使数据无效
		if (data.containsKey(Dept.STATE))
			utils.resetState(data, Dept.STATE);

		// 获取用户信息，公司编号在用户信息中取
		Map<String, String> user = utils.userSelf(req);
		data.put(Dept.COMP, user.get(User.COMP));
		data.put(Dept.OPUID, user.get(User.ID));

		// 检查同一级内是否有重名
		if (data.containsKey(Dept.NAME)) {
			// 注意：必须把主键Dept.ID放在第一个参数中
			String[] keys = { Dept.ID, Dept.COMP, Dept.PID, Dept.NAME };
			if (aolai.exists(FlowUtil.deptTable(), utils.sameIf(data, keys)))
				throw new FailedException(ConfUtil.DUPL_NAME);
		}

		// 修改一条部门数据
		Map<String, Object> where = utils.sameId(data, Dept.KEY);
		return aolai.update(FlowUtil.deptTable(), data, where);
	}

	/**
	 * 删除一条部门数据
	 */
	public Map<String, String> delDept() {

		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Dept.COMP);

		// 当前部门是否在组织架构中已经使用，则不能删除
		Map<String, Object> cond = new HashMap<>();
		cond.put(Orgn.COMP, where.get(Dept.COMP));
		cond.put(Orgn.DEPT, where.get(Dept.ID));
		// 已经有员工绑定该组织，则不能删除
		if (aolai.exists(null, Orgn.TABLE, cond))
			throw new FailedException(ConfUtil.CANT_REMOVE);
		
		// 部门下存在员工，即绑定的人员数量大于>0时，不能删除
		Map<String, Object> cond1 = new HashMap<>();
		cond1.put(Emp.COMP, where.get(Dept.COMP));
		cond1.put(Emp.DEPT, where.get(Dept.ID));
		if (aolai.exists(null, Emp.TABLE, cond1))
			throw new FailedException(ConfUtil.CANT_REMOVE);

		// 当前部门是否有下级部门，有则不能删除
		Map<String, Object> cond0 = new HashMap<>();
		cond0.put(Dept.COMP, where.get(Dept.COMP));
		cond0.put(Dept.PID, where.get(Dept.ID));
		if (aolai.exists(null, FlowUtil.deptTable(), cond0))
			throw new FailedException(ConfUtil.CANT_REMOVE);

		// 删除一条部门数据
		return aolai.delete(null, FlowUtil.deptTable(), where);
	}

	/** 员工 */

	/**
	 * 根据条件查询所有员工数据，返回员工信息分页展示数据
	 * <p>
	 * 注意：默认是按“主要任职部门”查询
	 */
	public Map<String, Object> getEmpList() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Emp.COMP);
		// 返回员工一览，有的员工可能还没设置部门
		// 关联“主要任职部门”：empComp
		// 返回员工信息分页展示数据
		// 按姓名排序
		Map<String, Map<String, String>> tables = getEmpCond(where);
		return getEmpList(tables, where, empByName());
	}

	/**
	 * 根据条件查询所有员工数据，返回员工信息分页展示数据
	 * <p>
	 * 注意“getEmpList”“getOrgnEmp”区别关联“主要任职部门”和非“主要任职部门”
	 */
	private Map<String, Object> getEmpList(
			Map<String, Map<String, String>> tables, Map<String, Object> where,
			Map<String, String> order) {

		// 查询部门内员工信息一览，这里是：inner join
		Map<String, Object> result = aolai.queryList(null, tables,
				GET_EMPS, where, order, null, ConfUtil.limitRows());
		List<Map<String, String>> list = utils.obj2List(result.get(Const.ITEMS));
		// 设置代理人、秘书的姓名
		list = setAgentName(list);

		// 设置岗位名称
		result.put(Const.ITEMS, setJobsName(list));

		// 返回员工信息分页展示数据
		return utils.success(result);
	}

	/**
	 * 关联多表的查询员工条件，关联“主要任职部门”
	 */
	Map<String, Map<String, String>> getEmpCond(Map<String, Object> where) {

		// 根据姓名正则匹配
		FlowUtil.likeCond(where, Emp.NAME);
		// 设置一个默认值状态值：在职员工
		utils.resetState(where, Emp.STATE);

		// 根据请求参数查询数据并返回查询结果
		Map<String, Map<String, String>> tables = new HashMap<>();
		tables.put(Emp.TABLE, null); // 员工表
		// 关联岗位职责关系表
		Map<String, String> cond1 = new HashMap<>();
		String empComp = (String) where.get(Emp.COMP);
		cond1.put(Orgn.COMP, empComp);
		cond1.put(Orgn.EMPID, Emp.ID);
		cond1.put(Orgn.DEPT, Emp.DEPT); // 这里是“主要任职部门”
		tables.put(Orgn.TABLE, cond1);
		// 关联部门表
		Map<String, String> cond2 = new HashMap<>();
		cond2.put(Dept.COMP, empComp);
		cond2.put(Dept.ID, Emp.DEPT); // 这里是“主要任职部门”
		tables.put(FlowUtil.deptTable(), cond2);
		return tables;
	}

	/** 员工 */
	
	/**
	 * 联合多表的组合查询条件：orgnComp、orgnEmpId、orgnDept、orgnJob
	 * <p>
	 * 可通过员工ID锁定员工：empUid、empId、empAgent、empSecty
	 */
	Map<String, Map<String, String>> getOrgnEmpCond(Map<String, Object> where) {
		
		// 联合多表的查询条件
		Map<String, Map<String, String>> tables = new HashMap<>();
		// 部门和员工关系表 ：orgnEmpId、orgnDept
		tables.put(Orgn.TABLE, null);
		// 关联员工表
		Map<String, String> cond1 = new HashMap<>();

		// 通过ID锁定员工：empSecty、empId(orgnEmpId)二选一
		if (where.containsKey(Emp.SECTY)) {
			// 优先使用当前用户或“秘书”
//			String uid = utils.sqlVal(where.get(Emp.SECTY));
			String[] keys = { Emp.UID, Emp.SECTY };
			Object uid = where.get(Emp.SECTY);
			Object[] values = { uid, uid };
			// “当前用户”或“秘书”
			where.put(utils.sqlOr(keys), values);
			where.remove(Emp.SECTY);
		} else if (where.containsKey(Emp.ID)) {
			where.put(Orgn.EMPID, where.get(Emp.ID));
		}
		// 设置一个默认值状态值：1.在职员工 0.离职
		boolean isDepart = Const.S_0.equals(where.get(Emp.STATE));
		String empState = isDepart ? Const.S_0 : Const.S_1;

//		if (where.containsKey(Orgn.EMPID))
//			cond1.put(Emp.ID, (String) where.get(Orgn.EMPID));
//		if (where.containsKey(Emp.UID))
//			cond1.put(Emp.UID, (String) where.get(Emp.UID));

		String compId = (String) where.get(Orgn.COMP);
		cond1.put(Emp.COMP, compId);
		cond1.put(Emp.ID, Orgn.EMPID);
		cond1.put(Emp.STATE, empState);
		tables.put(Emp.TABLE, cond1);
	
		// 关联组织表
		Map<String, String> cond2 = new HashMap<>();
		cond2.put(Dept.COMP, compId);
		cond2.put(Dept.ID, Orgn.DEPT); // 不一定是“主要任职部门”
		tables.put(FlowUtil.deptTable(), cond2);

		// 根据姓名正则匹配
		FlowUtil.likeCond(where, Emp.NAME);
		where.put(Emp.STATE, empState);

		// 选择包含子部门的情况，必须携带orgnDept
		String key;
		if (where.containsKey("include")) {
			key = utils.pHolder(Orgn.DEPT, Const.LIKE);
			where.put(key, where.get(Orgn.DEPT) + "%");
			where.remove(Orgn.DEPT);
		}
		// 按岗位职责（是JOB，非JOBS）匹配查询
		if (where.containsKey(Orgn.JOB)) {
			// 这里的查询条件为：Orgn.JOBS
			key = utils.jsonExt(Orgn.JOBS, where.get(Orgn.JOB));
			where.put(key, Const.S_1);
			where.remove(Orgn.JOB);
		}
		return tables;
	}

	/**
	 * 显示对应代理人（或秘书）的姓名
	 */
	List<Map<String, String>> setAgentName(List<Map<String, String>> empList) {
		if (!empList.isEmpty()) {
			// 检出有代理人empAgent、秘书empSecty的用户
			String[] keys = { Emp.AGENT, Emp.SECTY };
			Set<String> set = new HashSet<>();
			for (Map<String, String> emp : empList) {
				// 检出有代理人（秘书）的用户
				for (String key : keys) {
					if (!utils.isEmpty(emp.get(key))) {
						set.add(emp.get(key));
					}
				}
			}
			log.debug("setAgentName...{}", set);
			// 若代理人（秘书）非空时，获取代理人的姓名
			if (!set.isEmpty()) {
				// 公司编号
				String compId = empList.get(0).get(Emp.COMP), uid, name;
				// 获取代理人（秘书）的姓名
				Map<String, String> names = getEmpsName(compId, set);
				log.debug("getEmpsName...{}", names);
				// 对应"empAgentName"、"empSectyName"
				String[] vals = { Emp.AGENTNAME, Emp.SECTYNAME };
				for (Map<String, String> emp : empList) {
					for (int i = 0; i < 2; i++) {
						uid = emp.get(keys[i]);
						// 根据Emp.UID，获取代理人的姓名
//						if (names.containsKey(uid)) {
//							emp.put(keys[i], names.get(uid));
							// 仅用作前台显示或导出数据
						name = names.containsKey(uid) ? names.get(uid) : "";
						emp.put(vals[i], name);
//						}
					}
				}
			}
		}
		return empList;
	}

	/**
	 * 展示对应岗位职责的名称
	 */
	List<Map<String, String>> setJobsName(List<Map<String, String>> empList) {
		if (empList.isEmpty())
			return empList;
		// 获取岗位职责对应关系一览表
		Object empComp = empList.get(0).get(Emp.COMP);
		Map<String, String> jobNames = getJobsName(empComp);
		// 显示岗位职责名称
		// 一个部门多个岗位职责，匹配“候选人”后，只留了一个匹配上的岗位职责
		for (Map<String, String> emp : empList) {
			String jobs = emp.get(Orgn.JOBS);
			if (jobs == null) {
				emp.put(Orgn.JOBNAMES, "");
				continue;
			}
			// 这里有可能是多个岗位职责，格式：{"job0000":Const.S_1,"job0003":Const.S_1}
			Map<String, String> map = utils.json2Map(jobs);
			for (String key : map.keySet()) {
				map.put(key, jobNames.get(key));
			}
			// 显示岗位职责名称后：{"job0000":"财务经理","job0003":"部门经理"}
			emp.put(Orgn.JOBS, utils.obj2Str(map));
			// 仅用作前台显示或导出数据
			jobs = String.join(Const.COMMA, map.values());
			emp.put(Orgn.JOBNAMES, jobs);
		}
		return empList;
	}

	/**
	 * 所有岗位职责名称映射一览
	 */
	private Map<String, String> getJobsName(Object jobComp) {
		// 岗位职责一览
		Map<String, Object> cond = new HashMap<>();
		cond.put(Job.COMP, jobComp);
		String[] args = { null, Job.TABLE, "getJobList", Const.S_0 };
		List<Map<String, String>> list = aolai.findAll(cond, null, args);
		return utils.list2Map(list, Job.NO, Job.NAME);
	}

	/**
	 * 查询员工姓名映射一览，离职后还可以正常显示
	 */
	private Map<String, String> getEmpsName(Object empComp, Set<String> set) {
		// 岗位职责一览
		Map<String, Object> cond = new HashMap<>();
		cond.put(Emp.COMP, empComp);
		if (set != null && !set.isEmpty()) {
			// 此处处理HashSet无法转化为list问题
			List<String> empSetList = new ArrayList<>(set);
			cond.put(utils.pHolder(Emp.UID, Const.IN), empSetList);
		}
		String[] args = { null, Emp.TABLE, "getEmpName", Const.S_0 };
		List<Map<String, String>> list = aolai.findAll(cond, null, args);
		return utils.list2Map(list, Emp.UID, Emp.NAME);
	}

	/**
	 * 根据empId查询一个员工及关联“主要任职部门”的信息
	 */
	public Map<String, Object> getEmpInfo() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Emp.COMP);
		List<Map<String, String>> empList = getEmployee(where);
		// 没有查询到数据
		if (empList.isEmpty())
			throw new FailedException();
		// 返回员工信息
		return utils.success(empList.get(0));
	}

	/**
	 * 根据empId查询一个员工及关联“主要任职部门”的信息
	 */
	List<Map<String, String>> getEmployee(Map<String, Object> where) {
		// 关联“主要任职部门”：empComp\empId
		Map<String, Map<String, String>> tables = getEmpCond(where);
		// 查询部门内指定员工信息一览
		return aolai.findAll(null, tables, GET_EMPS, where, null);
	}

	/**
	 * 增加一条员工数据
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> addEmployee() {
		// 请求参数、当前用户信息
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户所属公司
		data.put(Emp.COMP, user.get(User.COMP));
		// 当前操作用户
		data.put(Emp.OPUID, user.get(User.ID));
		// 如未设置默认职级，设置一个默认的职级
		if(!data.containsKey(Emp.RANK))
			data.put(Emp.RANK, getDefaultRank(data.get(Emp.COMP)));
		// 没有UID时，用手机号
		if (!data.containsKey(Emp.UID)) {
			data.put(Emp.UID, data.get(Emp.MOBI));
		}
		// 检查手机号码规则
		String mobi = (String) data.get(Emp.MOBI);
		if (!utils.isMobile(mobi))
			return utils.result("mobileRules");

		// 检查员工数据是否重复的唯一属性
		sameEmpIf(data, Emp.UID);
		sameEmpIf(data, Emp.IDCARD);
		sameEmpIf(data, Emp.MOBI);
		// 检查员工工号
		if(!utils.isEmpty(data.get(Emp.WORKID)))
			sameEmpIf(data, Emp.WORKID);

		// 用手机号编码作为员工号
		data.put(Emp.ID, Digest.enBase62(Long.valueOf(mobi)));

		// 再新增一条员工数据，并返回新增结果
		return addEmployee(data);
	}

	/**
	 * 检查员工数据是否重复的唯一属性：empUid
	 */
	private void sameEmpIf(Map<String, Object> data, String field) {
		if (data.containsKey(field)) {
			String[] keys = { Emp.ID, field, Emp.COMP };
			// 检查是否有重复数据
			Map<String, Object> cond = utils.sameIf(data, keys);
			if (aolai.exists(null, Emp.TABLE, cond)) {
				log.debug("sameEmpIf...{}", cond);
				field = String.join("_", ConfUtil.DUPL_DATA, field);
				throw new FailedException(field);
			}
		}
	}

	/**
	 * 新增一条员工数据，增加数据时容错处理
	 */
	private Map<String, String> addEmployee(Map<String, Object> data) {
		// 首先增加一条组织架构数据
		Map<String, Object> orgn = new HashMap<>();
		orgn.put(Orgn.COMP, data.get(Emp.COMP));
		orgn.put(Orgn.DEPT, data.get(Emp.DEPT));
		orgn.put(Orgn.OPUID, data.get(Emp.OPUID));
		// 覆盖旧数据的状态
		orgn.put(Orgn.STATE, Const.S_1);
		// 注意前台参数为：empJobs
		String jobs = null;
		if (data.containsKey(Emp.JOBS))
			jobs = utils.obj2Str(data.get(Emp.JOBS));
		orgn.put(Orgn.JOBS, jobs == null ? "{}" : jobs);

		// 用手机号编码作为员工号
		orgn.put(Orgn.EMPID, data.get(Emp.ID));
		try {
			// 注意保持幂等性：
			// 先新增一条主要任职部门数据，再新增一条员工数据
			if (utils.isSuccess(aolai.addRecord(null, Orgn.TABLE, orgn, true))
					&& utils.isSuccess(aolai.addRecord(null, Emp.TABLE, data)))
				return utils.success();
			// 无效参数
			throw new FailedException();
		} catch (FailedException e) {
			throw new FailedException(e.getMessage());
		} catch (Exception e) {
			log.error("addEmployee...{}", e.getMessage());
		}

		// 失败，系统可能有并发冲突
		throw new FailedException(FlowUtil.DO_FAILED);
	}

	/**
	 * 修改一条员工数据
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> setEmployee() {
		// 请求参数、当前操作用户
		Map<String, Object> data = utils.jsonParams(req);
		// 不准许变更状态
		if (data.containsKey(Emp.STATE))
			data.remove(Emp.STATE);

		// 当前操作用户
		Map<String, String> user = utils.userSelf(req);
		data.put(Emp.COMP, user.get(User.COMP));
		data.put(Emp.OPUID, user.get(User.ID));

		// 检查员工数据是否重复的唯一属性
		sameEmpIf(data, Emp.UID);
		if (!utils.isEmpty(data.get(Emp.IDCARD)))
			sameEmpIf(data, Emp.IDCARD);
		if (!utils.isEmpty(data.get(Emp.MOBI)))
			sameEmpIf(data, Emp.MOBI);
		// 工号重复性校验
		if (!utils.isEmpty(data.get(Emp.WORKID)))
			sameEmpIf(data, Emp.WORKID);

		// 变更所在部门的岗位职责，如：empJobs={eDc1SFbs=1}
		if (data.containsKey(Emp.JOBS) && data.containsKey(Emp.DEPT)) {
			Map<String, Object> where = new HashMap<>();
			where.put(Orgn.COMP, data.get(Emp.COMP));
			where.put(Orgn.EMPID, data.get(Emp.ID));
			where.put(Orgn.DEPT, data.get(Emp.DEPT));
			// 变更所在部门的岗位
			Map<String, Object> fields = new HashMap<>();
			fields.put(Orgn.JOBS, utils.obj2Str(data.get(Emp.JOBS)));
			if (utils.isFailed(aolai.update(Orgn.TABLE, fields, where)))
				throw new FailedException();
		}
		if (data.containsKey(Emp.AGENT)) {
			// 仅更新未完成节点的WORK_AGENT
			Map<String, Object> fields = new HashMap<>();
			fields.put(Work.AGENT, data.get(Emp.AGENT));
			Map<String, Object> cond = new HashMap<>();
			cond.put(Work.COMP, data.get(Emp.COMP));
			cond.put(Work.EMPID, data.get(Emp.ID));
			cond.put(Work.STATE, FlowUtil.STATE_READY);
			if (utils.isFailed(aolai.update(Work.TABLE, fields, cond)))
				throw new FailedException();
		}

		// 修改一条员工数据
		return aolai.update(Emp.TABLE, data, utils.sameId(data, Emp.KEY));
	}

	/**
	 * 删除一条员工数据
	 */
	public Map<String, String> delEmployee() {
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		data.put(Emp.COMP, user.get(User.COMP));
		// 当前操作用户
		data.put(Emp.OPUID, user.get(User.ID));
		if (!data.containsKey(Emp.STATE)) {
			data.put(Emp.STATE, Const.S_0); // 删除（离职）标志
			// 增加离职时间
			data.put(Emp.DELETE, utils.today(Const.DTF));
		}

		Map<String, Object> cond = new HashMap<>();
		cond.put(Orgn.COMP, data.get(Emp.COMP));
		cond.put(Orgn.EMPID, data.get(Emp.ID));
		// 删除员工关联的组织信息
		delOrgn(cond);

		// 如果删除员工，也要把代理、秘书身份删除
		// data.get(Emp.UID)不存在，所有暂时无法删除
		// 离职后还可以正常显示
//		removeAgent(data, Emp.AGENT);

		// 删除一条员工数据，设为“离职”状态
		return aolai.update(Emp.TABLE, data, utils.sameId(data, Emp.KEY));
	}

	/**
	 * 删除代理人、秘书
	 */
//	void removeAgent(Map<String, Object> data, String field) {
//		Map<String, Object> where = new HashMap<>();
//		where.put(Emp.COMP, data.get(Emp.COMP));
//		where.put(field, data.get(Emp.UID));
//		Map<String, Object> fields = new HashMap<>();
//		fields.put(field, "");
//		aolai.update(Emp.TABLE, fields, where);
//	}

	/** 岗位职责职责 */

	/**
	 * 根据条件查询所有岗位职责职责数据
	 */
	public Map<String, Object> getJobList() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Job.COMP);
		// 正则匹配名称
		FlowUtil.likeCond(where, Job.NAME);
		// 根据请求参数查询数据并返回查询结果
		String[] args = { null, Job.TABLE, "getJobList", Const.S_0 };
		List<Map<String, String>> list = aolai.findAll(where, null, args);
		return utils.success(jobGroupCount(list, where.get(Job.COMP)));
	}

	/**
	 * 归集每个岗位职责包含的员工数
	 */
	private List<Map<String, String>> jobGroupCount(
			List<Map<String, String>> list, Object orgnComp) {
		if (!list.isEmpty()) {
			// 归类条件
			Map<String, Object> cond = new HashMap<>();
			cond.put(Orgn.COMP, orgnComp);
			for (Map<String, String> job : list) {
				// 住址这里的传递参数为：Orgn.JOB
				cond.put(Orgn.JOB, job.get(Job.NO));
				int cnt = flowDao.groupEmpByJob(cond);
				job.put(Job.COUNT, cnt > 0 ? String.valueOf(cnt) : "0");
			}
			log.debug("jobGroupCount...{}", list);
		}
		return list;
	}

	/**
	 * 增加一条岗位职责数据
	 */
	public Map<String, String> addJob() {
		// 请求参数、当前用户信息
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户所属公司
		data.put(Job.COMP, user.get(User.COMP));
		// 检查是否有重复数据（重名数据）
		String[] keys = { Job.NO, Job.COMP, Job.NAME };
		if (aolai.exists(Job.TABLE, utils.sameIf(data, keys)))
			throw new FailedException(ConfUtil.DUPL_NAME);
		// 当前操作用户
		data.put(Job.OPUID, user.get(User.ID));

		// 新增一条岗位职责数据
		return addJob(data);
	}

	/**
	 * 新增岗位职责数据，允许尝试2次
	 */
	private Map<String, String> addJob(Map<String, Object> data) {
		// 连续尝试2次
		int times = FlowUtil.TRY_TIMES;
		while (times-- > 0) {
			// 没有编码时，自动生成一个编码
			data.put(Job.NO, Digest.uuid8());
			try {
				// 新增一条岗位职责数据
				return aolai.addRecord(Job.TABLE, data);
			} catch (FailedException e) {
				throw new FailedException(e.getMessage());
			} catch (Exception e) {
				// 跳过、再尝试一次
				log.error("addJob...{}", e.getMessage());
			}
		}
		// 失败，系统可能有并发冲突
		throw new FailedException(FlowUtil.DO_FAILED);
	}

	/**
	 * 修改一条岗位职责数据
	 */
	public Map<String, String> setJob() {
		// 请求参数、当前操作用户
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		data.put(Job.COMP, user.get(User.COMP));

		// 检查是否有重复数据（重名数据）
		if (data.containsKey(Job.NAME)) {
			String[] keys = { Job.NO, Job.COMP, Job.NAME };
			if (aolai.exists(Job.TABLE, utils.sameIf(data, keys)))
				throw new FailedException(ConfUtil.DUPL_NAME);
		}

		// 当前操作用户
		data.put(Job.OPUID, user.get(User.ID));

		// 更新岗位职责职责数据
		return aolai.update(Job.TABLE, data, utils.sameNo(data, Job.KEY));
	}

	/**
	 * 删除一条岗位职责职责数据
	 */
	public Map<String, String> delJob() {
		Map<String, Object> where = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		where.put(Job.COMP, user.get(User.COMP));

		/** 检查“组织架构表”中是否使用的岗位 */
		Map<String, Object> cond0 = new HashMap<>();
		cond0.put(Orgn.COMP, where.get(Job.COMP));
		cond0.put(utils.jsonExt(Orgn.JOBS, where.get(Job.NO)), Const.S_1);

		/** 检查“流程表”中是否使用的岗位 */
		Map<String, Object> cond1 = new HashMap<>();
		cond1.put(Biz.COMP, where.get(Job.COMP));
		cond1.put(utils.jsonExt(Biz.JOBS, where.get(Job.NO)), Const.S_1);
		
		/**
		 * 检查在审批人查找条件中使用的岗位,岗位如被使用不允许删除，<br>
		 * 此处由于mysql 5.7 限制，没办法使用JSON_TABLE, 此处属于折衷处理方案
		 */
		// NODE_RULES like '%"nodeJob": "eDc1SFbs"%' 
		Map<String, Object> cond2 = new HashMap<>();
		cond2.put(Node.COMP, where.get(Job.COMP));
		// 此处冒号后需要有一个空格，不确定是不是所有的都是这种情况，暂定这样处理，如果后续测试中出现问题，直接用岗位值作为查找值
		String val = "%\"" + Node.JOB + "\": \"" + where.get(Job.NO) + "\"%";
		cond2.put(utils.pHolder(Node.RULES, Const.LIKE), val);
		// 数据存在时，不能删除
		if (aolai.exists(Orgn.TABLE, cond0) || aolai.exists(Biz.TABLE, cond1)
				|| aolai.exists(Node.TABLE, cond2))
			throw new FailedException(ConfUtil.CANT_REMOVE);

		// 删除一条岗位职责数据
		return aolai.delete(Job.TABLE, where);
	}

	/** 职级 */

	/**
	 * 获取一个默认的职级，无默认职级时先置为空“”
	 */
	private String getDefaultRank(Object compId) {
		Map<String, Object> where = new HashMap<>();
		where.put(Rank.COMP, compId);
		where.put(Rank.MARK, Const.S_1);
		Map<String, String> result = aolai.findOne(Rank.TABLE, where);
		return result.containsKey(Const.STS) ? "" : result.get(Rank.ID);
	}

	/**
	 * 根据条件查询所有职级数据：rankId,rankName,rankCount,rankMark
	 */
	public Map<String, Object> getRankList() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Rank.COMP);

		// 正则匹配名称
		FlowUtil.likeCond(where, Rank.NAME);
		// 查询所有职级数据
		return utils.success(rankGroupCount(where));
	}

	/**
	 * 归集每个职级包含的员工数（每个员工对应一个职级）
	 */
	private List<Map<String, String>> rankGroupCount(Map<String, Object> where) {
		log.debug("getRankList...{}", where);
		// 查询职级一览
		String[] args = { null, Rank.TABLE, "getRankList", Const.S_0 };
		List<Map<String, String>> items = aolai.findAll(where, null, args);
		if (!items.isEmpty()) {
			Map<String, Object> cond = new HashMap<>();
			cond.put(Emp.COMP, where.get(Rank.COMP));
			cond.put(Emp.STATE, Const.S_1); // 在职员工
			// 按员工职级empRank归集在职员工数量
			String[] keys = { Emp.RANK };
			List<Map<String, String>> list = aolai.groupCount(null, Emp.TABLE, keys, cond);
			// 职级对应员工人数
			Map<String, String> tmp = utils.list2Map(list, Emp.RANK, "CNT");
			// 对应职级的员工人数
			for (Map<String, String> item : items) {
				String cnt = tmp.get(item.get(Rank.ID));
//				if (cnt != null)
				item.put(Rank.COUNT, cnt == null ? "" : cnt);
			}
		}
		return items;
	}

	/**
	 * 增加一条职级数据
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> addRank() {
		// 请求参数、当前用户信息
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户所属公司
		data.put(Rank.COMP, user.get(User.COMP));
		// 检查是否有重复数据（重名数据）
		String[] keys = { Rank.ID, Rank.COMP, Rank.NAME };
		if (aolai.exists(Rank.TABLE, utils.sameIf(data, keys)))
			throw new FailedException(ConfUtil.DUPL_NAME);

		// 当前操作用户
		data.put(Rank.OPUID, user.get(User.ID));

		// 清除已设定的“默认”值，改为“非默认”
		clearDefaultRank(data);

		// 新增一条职级数据
		return addRank(data);
	}

	/**
	 * 非当前数据全部改为“非默认”，rankMark=0.非默认 1.默认
	 */
	private void clearDefaultRank(Map<String, Object> data) {
		if (Const.S_1.equals(data.get(Rank.MARK))) {
			Map<String, Object> cond = new HashMap<>();
			cond.put(Rank.COMP, data.get(Rank.COMP));
			cond.put(Rank.MARK, data.get(Rank.MARK));
			// 排除自己，其他都改为：0.非默认职级
			Map<String, Object> fields = new HashMap<>();
			fields.put(Rank.MARK, Const.S_0);
			aolai.update(Rank.TABLE, fields, cond);
		}
	}

	/**
	 * 新增职级数据，允许尝试2次
	 */
	private Map<String, String> addRank(Map<String, Object> data) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Rank.COMP, data.get(Rank.COMP));
		// 连续尝试2次
		int times = FlowUtil.TRY_TIMES;
		while (times-- > 0) {
			// 没有编码时，自动生成一个编码
			int id = aolai.getMaxCode(null, Rank.TABLE, Rank.ID, cond);
			data.put(Rank.ID, String.format("%02d", id));
			try {
				// 新增一条岗位职责数据
				return aolai.addRecord(Rank.TABLE, data);
			} catch (FailedException e) {
				throw new FailedException(e.getMessage());
			} catch (Exception e) {
				// 再尝试一次
				log.error("addRank...{}", e.getMessage());
			}
		}
		// 尝试两次后失败，系统有并发冲突
		throw new FailedException(FlowUtil.DO_FAILED);
	}

	/**
	 * 修改一条职级数据
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> setRank() {

		// 请求参数、当前操作用户
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		data.put(Rank.COMP, user.get(User.COMP));

		// 检查是否有重复数据（重名数据）
		if (data.containsKey(Rank.NAME)) {
			String[] keys = { Rank.ID, Rank.COMP, Rank.NAME };
			if (aolai.exists(Rank.TABLE, utils.sameIf(data, keys)))
				throw new FailedException(ConfUtil.DUPL_NAME);
		}
		// 当前操作用户
		data.put(Rank.OPUID, user.get(User.ID));

		// 清除已设定的“默认”值，改为“非默认”
		clearDefaultRank(data);

		// 更新职级数据
		return aolai.update(Rank.TABLE, data, utils.sameId(data, Rank.KEY));
	}

	/**
	 * 删除一条职级数据
	 */
	public Map<String, String> delRank() {
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Rank.COMP);

		Map<String, Object> cond = new HashMap<>();
		cond.put(Emp.COMP, where.get(Rank.COMP));
		cond.put(Emp.RANK, where.get(Rank.ID));
		// 数据存在，且有绑定的人员数
		if (aolai.exists(Emp.TABLE, cond))
			throw new FailedException(ConfUtil.CANT_REMOVE);

		// 删除一条职级数据
		return aolai.delete(Rank.TABLE, where);
	}

}
