package cn.ps1.soar.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;

import javax.servlet.http.HttpServletRequest;

import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.ps1.aolai.entity.Dept;
import cn.ps1.aolai.service.AolaiService;
import cn.ps1.aolai.service.UtilsService;
import cn.ps1.soar.dao.FlowDao;
import cn.ps1.soar.engine.FlowEngine;
import cn.ps1.soar.entity.*;
import cn.ps1.soar.utils.FlowUtil;

/**
 * 业务流程的控制节点处理
 * 
 * @author Aolai
 * @version 1.0 $Date: 2024.01.20
 * @since openjdk-1.8
 */
@Service
public class EventEmitter {

	private static Logger log = LoggerFactory.getLogger(EventEmitter.class);

	@Autowired
	private AolaiService aolai;
	@Autowired
	private UtilsService utils;

	@Autowired
	private BizService bizSvc;
	@Autowired
	private FlowService flowSvc;
	@Autowired
	private OrgnService orgnSvc;
	@Autowired
	private TaskService taskSvc;

	@Autowired
	private FlowDao flowDao;
	@Autowired
	private HttpServletRequest req;

	// 业务流程驱动引擎
	private FlowEngine flowEngine;

	/**
	 * 暂存一个初始新任务为“草稿”，“待发起”的审批流程<br>
	 * 必填参数：{taskFlowNo,taskEmpId,taskDept,taskMode,taskFormData}<br>
	 * 可选参数：[,taskId,candidates]
	 */
	public Map<String, Object> saveTask() {

		// 补充初始参数：补充“秘书”empSecty、公司taskComp、标题taskTitle
		Map<String, Object> params = taskSvc.initTaskParams();
		// 获取“发起人”（非操作员）empId、empUid及所在部门、岗位等
		Map<String, String> sponsor = getSponsorInfo(params);

		// 发起“新流程”的json对象格式转换，前台参数有时“字符串”有时“对象”
		setJsonData(params, Task.FORMDATA);
		// 如果携带“taskId”，则沿用已有任务编号，没有就新建一个
		checkReadyTask(params);

		// 保存为“草稿”，此处仅把原“草稿”、“撤回”、“驳回”改变状态
		// 又补充了：taskState（等待中）
		// 补充信息：taskEmpId、taskEmpName、taskOpUid、taskSecty
		taskSvc.addNewTask(params, sponsor, FlowUtil.STATE_READY);

		// 提交成功后，返回任务编号
		return utils.success(params.get(Task.ID));
	}

	/**
	 * 当用户新发起“新流程”时，根据流程编号选出第一个审批节点的“候选人”
	 * <p>
	 * 如有需要前端确认的多个候选人，返回前端确认处理后再“提交”。
	 * <p>
	 * 如果没有需要确认的多个候选人，则直接提交处理，并返回结果。<br>
	 * <p>
	 * 注意：选中“审核人”后再提交时，必须携带 candidates参数，否则无效
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object newWorkTask() {
		// 参数：{taskFlowNo,taskEmpId,taskDept,taskMode,taskFormData[,taskId,candidates]}

		// 补充初始参数：补充“秘书”empSecty、公司taskComp、标题taskTitle
		Map<String, Object> params = taskSvc.initTaskParams();
		// 获取“发起人”（非操作员）empId、empUid及所在部门、岗位等
		Map<String, String> sponsor = getSponsorInfo(params);

		// 发起“新流程”的json对象格式转换，前台参数有时“字符串”有时“对象”
		// 携带“taskId”时，前面initTaskParams()中已删除taskMode
		if (params.containsKey(Task.MODE))
			setJsonData(params, Task.MODE);

		Map<String, Object> formData = setJsonData(params, Task.FORMDATA);
		// 表单项中追加流程“发起人”的员工及部门等信息，以备用
		formData.put(Emp.INFO, sponsor);

		// 当前用户（发起人）提交一个“新流程”
		initFlowEngine(params, formData);
		return startTask(params, sponsor);
	}

	/**
	 * 表单json对象格式转换：字符串转成对象返回，对象转成字符串保存
	 */
	private Map<String, Object> setJsonData(Map<String, Object> params,
			String fieldKey) {
		// 发起“新流程”的json对象信息
		Object obj = params.get(fieldKey);
		// 如果前台传过来是字符串，则不处理，如果是对象，则转成字符串
		if (obj instanceof String) {
			return utils.json2Map(obj);
		}
		// 转成字符串，保存到临时表单项中
		params.put(fieldKey, utils.obj2Str(obj));
		return utils.obj2Map(obj);
	}

	/**
	 * 获取“发起人”信息，新发起流程时，通过“当前用户”或“秘书”找出“发起人”
	 */
	private <T> Map<String, String> getSponsorInfo(Map<String, T> params) {
		Map<String, Object> where = new HashMap<>();
		// 发起、审批时，必须有“taskEmpId”参数
		if (!params.containsKey(Task.EMPID))
			throw new FailedException(); // 无效参数

		// 查询条件：发起人的empId
		where.put(Orgn.EMPID, params.get(Task.EMPID));

		// 新发起流程时，默认发起人为“当前用户”或“秘书”
		// 鉴权：如果携带“秘书”参数，为秘书代办，通过“秘书”找出“发起人”
		if (params.containsKey(Emp.SECTY)) {
			// 审批时，不再关心是“当前用户”或“秘书”（注意不能用taskSecty）
			where.put(Emp.SECTY, params.get(Emp.SECTY));
		}
		// 国际化支持
		where.put(Const.I18N, params.get(Const.I18N));

		// 锁定发起任务的公司及部门
		where.put(Orgn.COMP, params.get(Task.COMP));
		where.put(Orgn.DEPT, params.get(Task.DEPT));

		// 返回“发起人”信息：empId、empUid等
		List<Map<String, String>> empList = orgnSvc.getEmpBy(where);
		if (empList.isEmpty())
			throw new FailedException(ConfUtil.DENY_ACCESS);
		return empList.get(0);
	}

	/**
	 * 流程引擎初始化，设置当前工作流程的所有“待审批”、“审批中”节点
	 */
	private void initFlowEngine(Map<String, Object> params, Map<String, Object> formData) {
		// 1.初始化工作流，设置当前工作流程的所有“待审批”、“审批中”节点
		// 这里如果携带了taskId，则绑定taskId的
		flowEngine = new FlowEngine(formData, flowSvc.getFlowNodes(params));
		flowEngine.setUserLang(params.get(Const.I18N));
	}

	/**
	 * 发起“新流程”时，根据流程编号（或任务编号）的获取所有节点，进入工作审批流程<br>
	 * <p>
	 * 参数：{taskFlowNo,taskDept,taskMode,taskFormData[,taskId,candidates]}
	 */
	private Map<String, Object> startTask(Map<String, Object> params,
			Map<String, String> sponsor) {
		// 1.初始化工作流，设置当前工作流程的所有“待审批”、“审批中”节点

		// 2.通过引擎自动选出第一个审批节点及全部“候选人”
		// 节点上只有一个候选人时，设为节点审批人
		// 并设置节点信息：workEmpId,workEmpName,workAgent,workOpUid
		flowEngine.findFirstNode();

		// 3.当前第一个节点有多个“候选人”时，返给前台选择确认
		Map<String, Object> result = getCandidates(params);
		if (result != null)
			return result;

		// 如果携带“taskId”，则沿用已有任务编号，没有就新建一个
		if (checkReadyTask(params)) {
			// 发起新流程，如果包含taskId，说明是驳回后再发起，需更新workIdx++
			incrApproveIdx(params);
		}

		// 4.保存“审批中”的工作节点，并更新节点状态
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 更新待审批节点（参考数据来源：getFlowNodes）
		taskSvc.saveWorkNodes(params, workNodes);

		// 5.更新流程表的“待审批”或“就绪”的中途节点taskNodeNo
		// 补充信息：taskNodeNo、这里不可能完成（taskState）
		flowEngine.setWaitingNode(params);

		// 6.增加一条新“草稿”，原“草稿”、“撤回”、“驳回”状态，此处仅改变“状态”
		// 又补充了：taskState（等待中）
		// 补充信息：taskEmpId、taskEmpName、taskOpUid、taskSecty
		taskSvc.addNewTask(params, sponsor, FlowUtil.STATE_WAITING);

		// 7.发送通知消息
		flowSvc.notifyAll(workNodes);

		// 流程处理后，执行该类别的“后续操作”
		// 此处兼容发起人和审批人为一个节点，直接完成场景
		// 如果审批人和发起人是同一个人不自动跳过，此处isSuccess不为true，所以不影响
		if (flowEngine.isSucceed()) {
			// 流程发起后审批人如果是自己并且已自动完成，则自动审批通过
			params.put(Task.STATE, FlowUtil.STATE_COMPLETE);
			flowSvc.followProcess(params);
		}

		// 提交成功后，返回任务编号
		return utils.success(params.get(Task.ID));
	}

	/**
	 * 找出候选人列表
	 */
	private Map<String, Object> getCandidates(Map<String, Object> params) {
		// 按步骤处理： candidates.size() > 0时，多个候选人
		// 若无候选人，前面已返回错误提示，仅一个候选人时则跳过处理
		List<Map<String, Object>> candidates = flowEngine.getCandidates();
		// candidates.isEmpty()，系统已经自动推送了唯一候选人
		// candidates.size() > 0，一个或多个节点多个候选人
		if (!candidates.isEmpty()) {
			// 判断是否携带了candidates参数
			Object selectee = params.get(FlowUtil.CANDIDATES);
			// 如果未携带参数，则返回当前节点对应“候选人”列表
			if (selectee == null) {
				return utils.success(candidates);
			} else if (!matchCandidate(candidates, selectee)) {
				// 匹配前台已选中当前节点的其中一位“候选人”
				// 新设候选人数据：workEmpId,workEmpName,workAgent,workOpUid
				throw new FailedException(); // 无效参数
			}
		}
		return null;
	}

	/**
	 * 匹配提交已“选中”的审批节点的审批“候选人”
	 * 
	 * @param candidates 备选候选人
	 * @param selectee 前台选中的人选
	 * @return 是否能对应匹配上
	 */
	private boolean matchCandidate(List<Map<String, Object>> candidates,
			Object selectee) {
		// 已提交“选中”的审批“候选人”： [{nodeNo:"010103",empId:"wixsi4s8"},..]
		List<Map<String, Object>> nodes = utils.obj2List(selectee);
		// 前台选中的人选要与后台候选节点保持一致，这里多节点主要考虑是并行节点
		if (candidates.size() != nodes.size())
			return false;

		// 遍历当前缓存的“审批中”的节点，匹配成功的审批人计数
		int approverCount = 0;
		Map<String, String> node;
		List<Map<String, String>> empList;
		// 遍历每个审批节点映射的全部“候选人”
		for (Map<String, Object> candidate : candidates) {
			node = utils.obj2Map(candidate.get(FlowUtil.MORM_NODE));
			empList = utils.obj2List(candidate.get(FlowUtil.CANDIDATES));
			
			// 如果提交的“节点”未在“待审批”节点列表中，则跳过
			// {nodeNo:"010103",empId:"wixsi4s8"}
			String nodeNo = node.get(Node.NO);
			Map<String, Object> params = utils.findIn(nodes, Node.NO, nodeNo);
			if (params == null)
				continue;

			// 逐个按“节点”与“候选人”列表映射中匹配，必须匹配上才行
			// {nodeNo:"010103",empId:"wixsi4s8"}
			Map<String, String> emp = utils.findIn(empList, Emp.ID, (String) params.get(Emp.ID));
			if (emp == null) {
				// 如果某节点提交的审核人为空，则判断是否批量提交多人
				// 批量处理某节点提交的审核人
				List<String> list = utils.obj2List(params.get(FlowUtil.APPROVERS));
				int count = 0;
				for (String empId : list) {
					emp = utils.findIn(empList, Emp.ID, empId);
					if (emp == null)
						continue;
					/**
					 * 如果提交推送的“候选人”能匹配到审批节点的“候选人”<br>
					 * 新设节点上的审批人信息：workEmpId,workEmpName,workAgent,workOpUid..<br>
					 * empId,empName,empUid,empAgent,empMobi,empMail,empWechat
					 */
					flowEngine.setNodeApprover(node, emp);
					// 3.3.把克隆节点设为“就绪”待办
//					flowEngine.addWorkNode(node, FlowUtil.STATE_READY);
					count++;
				}
				if (count == 0)
					return false;
			} else {
				flowEngine.setNodeApprover(node, emp);
			}
			approverCount++;
		}
		// 只有数量一致才可以，这里多节点主要考虑是并行节点
		return candidates.size() == approverCount;
	}

	/**
	 * 校验发起的任务（taskId）是否重复越权操作了其他taskId
	 */
	private boolean checkReadyTask(Map<String, Object> params) {
		// 未携带taskId，则返回一个新taskId
		if (params.get(Task.ID) == null) {
			params.put(Task.ID, Digest.uuid8() + params.get(Task.EMPID));
			return false;
		} else {
			// 如果前台携带了taskId，根据taskId获取工作流“草稿”
			Map<String, String> taskInfo = taskSvc.getTaskInfo(params);

			// 若草稿中有“秘书”，则秘书可以再发起流程（每人只能设一个秘书）
			// 此时params中的Emp.SECTY就是当前登录用户userId
			Object userId = params.get(Emp.SECTY);
			if (!userId.equals(taskInfo.get(Task.UID))
					&& !userId.equals(taskInfo.get(Task.SECTY))) {
				// 参数无效：当前登录用户既非“秘书”、也非“自己”
				throw new FailedException("", Task.UID);
			}

			// 无效任务：非“草稿”、“撤回”、“驳回”状态（0\5\4）时
			String taskState = taskInfo.get(Task.STATE);
			if (!FlowUtil.STATE_READY.equals(taskState)
					&& !FlowUtil.STATE_CANCEL.equals(taskState)
					&& !FlowUtil.STATE_REJECT.equals(taskState)) {
				// 其他状态的任务不允许有重新提交的情况：1\2\3\9
				throw new FailedException("", Task.STATE);
			}
			// 携带了taskId
			return true;
		}
	}

	/**
	 * 选择满足规则的当前节点的审批候选人，根据条件选取当前node的审批候选人
	 * <p>
	 * 当前节点全部信息：
	 * nodeComp,nodeNo,nodeFlowNo,nodeName,nodeParent,nodeStyle,nodeTier
	 * ,nodeType,nodeStep,nodeRules,nodeEvent,taskComp,taskId
	 * ,taskFlowNo,workComp,workTaskId,workNodeNo,workState,workComment
	 * ,workESign,workUpdate,workIdx..
	 */
	public List<Map<String, String>> getCandidatesBy(Map<String, String> node,
			Map<String, Object> formData) {
		// 根据empInfo获取发起人及部门信息：
		// deptId,deptName,deptType,deptTier,deptCost
		Map<String, String> sponsor = utils.obj2Map(formData.get(Emp.INFO));
		return getCandidates(node, sponsor);
	}

	/**
	 * 根据“发起人”属性，获取流程中每个节点的审批人
	 */
	private List<Map<String, String>> getCandidates(Map<String, String> node,
			Map<String, String> sponsor) {
		String deptId = sponsor.get(Dept.ID);

		// 规则一览：当前审批节点的岗位规则列表，多条审批岗位规则如下：
		// [{Node.JOB: "eAbpfsBK", Node.SCOPE: {"all": "1"}},..]
		List<Map<String, Object>> rules = utils.json2List(node.get(Node.RULES));

		// 匹配规则一览表，一个员工可匹配多个岗位
		List<Map<String, String>> list = new ArrayList<>();
		for (Map<String, Object> rule : rules) {
			Map<String, Object> where = new HashMap<>();
			where.put(Const.I18N, flowEngine.getUserLang());
			/**
			 * 只能是（nodeJob、employee）二选一<br>
			 * A.仅配置一个“审批人员”的情况
			 */
			if (rule.containsKey(FlowUtil.EMPLOYEE)) { 
				// 二选一：选定一个“审批人”
				String empId = (String) rule.get(FlowUtil.EMPLOYEE);
				// 这里需要“去重”处理：列表中不存时(==null)处理
				if (utils.findIn(list, Emp.ID, empId) == null) {
					// 如果直接绑定某位员工时
					where.put(Emp.COMP, node.get(Node.COMP));
					where.put(Emp.ID, rule.get(FlowUtil.EMPLOYEE));
					// 审批人携带了“主要任职部门”信息
					list.addAll(orgnSvc.getEmployee(where));
				}
				continue;
			}
			/**
			 * 只能是（nodeJob、employee）二选一<br>
			 * B.如果配置了“审批岗位”nodeJob，先根据“岗位”获取候选人一览
			 */
			where.put(Orgn.COMP, node.get(Node.COMP));
			// 注意这里的传递参数为：Orgn.JOB，用作去匹配orgnJobs
			where.put(Orgn.JOB, rule.get(Node.JOB));

			// 查询该岗位的员工一览（默认按部门倒序）
			List<Map<String, String>> empList = orgnSvc.getEmpBy(where);
			if (empList.isEmpty())
				continue;
			log.debug("approver...{}", empList.get(0));

			/**
			 * 若要选出“候选人”，还需要匹配“节点规则”相关配置
			 */
			Map<String, String> scope = utils.obj2Map(rule.get(Node.SCOPE));
			// 虽然用了for循环，实际key只有一个：
			// {"all": "1"}、或{deptId: "001005"}、或{"deptLevel": "1"}
			for (String key : scope.keySet()) {
				// 注意员工有重复分布在多个部门
				for (Map<String, String> approver : empList) {
					// “候选人”所在部门
					String dept = approver.get(Dept.ID);

					// 1.“发起人”部门要在“候选人”部门管辖内，否则无审批权
					// 候选人非上级部门，如：sponsorDeptId=“001005003”，emp.deptId=“001004”
					// 未限制nodeScope为{"all": "1"}时，必须确保审批人是发起者同级或上级部门，否则跳过
					if (Const.ALL.equals(key) && !deptId.startsWith(dept))
						continue; // 不匹配则忽略

					// 2.“候选人”要匹配“节点规则”才可以审批
					// 这里key可能是发起部门：“默认(all)”，或其上级主管部门的岗位：
					// 限定部门等级(deptTier)、指定部门(deptId)、部门类型(deptType)、成本中心(deptCost)等

					if (Const.ALL.equals(key) || matchRules(scope, key, approver, deptId)) {
						/**
						 * 增加一位能匹配角色的候选人
						 */
						addCandidate(list, approver, rule.get(Node.JOB));
						log.debug("approver...{}", approver);
					}

				}
			}
		}
		// 返回候选人列表（含员工的岗位信息）
		log.debug("Candidates...{}", list);
		return orgnSvc.setJobsName(list);
	}

	/**
	 * 匹配规则
	 */
	private boolean matchRules(Map<String, String> scope, String key,
			Map<String, String> emp, String deptId) {
		// 这里关闭对deptTier的支持
		// 已弃用deptTier属性，前端支持deptLevel传参
		String val = String.valueOf(scope.get(key));
		if (Dept.LEVEL.equals(key)) {// || "deptTier".equals(key)) {
			// "deptLevel"
			int tier = Integer.parseInt(val);
			return emp.get(Dept.ID).length() / FlowUtil.leafW() == tier;
		} else if (Dept.TYPE.equals(key)) {
			// "deptType"
			return Const.S_0.equals(val) || emp.get(key).equals(val);
		} else if ("sameDept".equals(key)) {
			return deptId.equals(emp.get(Dept.ID));
		} else if (scope.containsKey(key) && val.equals(emp.get(key))) {
			return true;
		}
		return false;
	}

	/**
	 * 增加一位能匹配的角色候选人
	 */
	private void addCandidate(List<Map<String, String>> list,
			Map<String, String> approver, Object job) {
		// 能匹配上的角色
		Map<String, String> jobs = new HashMap<>();
		// 这里也需要“去重”处理
		Map<String, String> candidate = utils.findIn(list, Emp.ID,
				approver.get(Emp.ID));

		if (candidate == null) {
			// 二选一后用nodeJob去匹配orgnJobs
			list.add(approver);
		} else {
			// 追加同一人多次可匹配的岗位
			approver = candidate;
			jobs = utils.json2Map(approver.get(Orgn.JOBS));
		}
		jobs.put((String) job, Const.S_1);
		approver.put(Orgn.JOBS, utils.obj2Str(jobs));
	}

	/**
	 * 条件驱动，根据条件返回的结果再继续处理
	 */
	public int getActionIdx(Map<String, String> node, Map<String, Object> form) {
//		StringBuilder sb = new StringBuilder();
		// {"event": ["{formMoney} >= 10000", "{formMoney} <= 10"], "joint": "OR"}
		Map<String, Object> rules = utils.json2Map(node.get(Node.RULES));
		StringJoiner jt = new StringJoiner(utils.quote(rules.get(Const.JOINT), " "));
		List<String> event = utils.obj2List(rules.get("event"));
		// ["{formMoney} >= 10000", "{formMoney} <= 10"]
		for (String ev : event) {
			String[] arr = ev.split("\\{ *");
			for (String str : arr) {
				String[] key = str.split(" *\\}");
//				if ("".equals(str)) // 首个字符一般为空
				if (key.length == 1 || !isValidExpr(key[1])) // 首个字符一般为空
					continue;

				// 注意这里form 属性的值需要防止SQL注入，数字转为字符：formMoney="20000"
				// String val = String.valueOf(form.get(key[0]));

				// 改为form表单支持多层嵌套的对象：formMoney
				String val = getFormValue(form, key[0]);
		
				// 表单的值需要能够匹配汉字，空字符串也认为是正常的表单数据，需要校验
				if ("".equals(val) || utils.isMatch(val, "\\w+")
						|| utils.isMatch(val, "[%0-9a-zA-Z\\u4e00-\\u9fa5]+")) {
					// 此处需要增加单引号，如不增加字符串或者汉字会报错
					// 如果是纯数字比较，则不用单引号，否则比较会不正确
					if (utils.isNumeric(val)) {
						jt.add(val + key[1]);
					} else {
						jt.add(utils.quote(val, "'") + key[1]);
					}
				}

			}

		}
		log.debug("> getActionIdx...{}", event);
		try {
			// 这里执行自定义SQL，必须增加try-catch捕获异常
//			return flowDao.getActionIdx(sb.toString());
			return flowDao.getActionIdx(jt.toString());
		} catch (Exception e) {
			log.warn("> getActionIdx...{}", event);
			return -1;
		}
	}

	/**
	 * 获取表单的属性值：form表单支持多层嵌套的对象<br>
	 * 例如：testForm.topTitle.userAge
	 */
	private String getFormValue(Object obj, String keyStr) {
		String[] keys = keyStr.split("\\.");
		for (String key : keys) {
			log.debug("> getFormValue...{}", obj);
			if (obj instanceof Map) {
				Map<String, Object> map = utils.obj2Map(obj);
				// 例如：key=topTitle时，还有一层key=userAge
				obj = map.get(key);
			} else {
				obj = null;
			}
/*			if (obj == null) {
				obj = "";
//				log.warn("> getFormValue...{}", key);
//				flowEngine.isLostWay();
			}*/
		}
		return obj == null ? "" : String.valueOf(obj);
	}

	/**
	 * 正则校验表达式
	 */
	private boolean isValidExpr(String expr) {
		if (expr.length() > 0) {
			// 从表达式拆分识别出关键字，如：“ in (”
			String[] arr = expr.split(" +"); // 以空格“”分割
			for (String str : arr) {
				// 必须能匹配规定范围内的表达式，或满足JSON类型操作的表达式格式："->>'$.inspur.eyun'"
				if (str.length() == 0
						|| utils.findIn(ConfUtil.SQL_EXPR, str.toLowerCase())
						|| utils.isNumeric(str)
						|| utils.isMatch(str, "['%]*\\w+[%']*")
						|| utils.isMatch(str, "'[%0-9a-zA-Z\\u4e00-\\u9fa5]+'")) // 匹配单引号，开头结尾，中间包含百分号、数字以及汉字
					continue;
				// 若有未匹配的表达式，则返回无效
				log.error("invalid expr...{}", str);
				return false;
			}
		}
		return true;
	}

	/**
	 * 当用户完成当前节点审批时，根据流程编号列出下一个审批候选人
	 * <p>
	 * 如有需要前端确认的候选人，返回前端确认处理后再“提交”。
	 * <p>
	 * 如果没有需要确认的候选人，则直接提交处理，并返回结果。
	 * <p>
	 * 当用户确认选中“审核人”后，必须携带candidates参数，否则无效 <br>
	 * 注意：选中“审核人”后再提交时，必须携带 candidates参数，否则无效 <br>
	 * params={taskId,workNodeNo,workComment[,candidates,informSb]}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object approveNode() {
		// 先补充参数：taskComp\workAgent
		Map<String, Object> params = taskSvc.getWorkParams();
		// 再读取待办的任务节点信息, 并补充params参数：workEmpId、taskFlowNo
		Map<String, String> task = taskSvc.getReadyTask(params);

		// 支持多语言i18n
		// 当前用户有审批权限。查询“发起人”的条件：taskEmpId或taskOpUid
		task.put(Const.I18N, (String) params.get(Const.I18N));
		// 返回发起人信息：empId、empUid等
		Map<String, String> sponsor = getSponsorInfo(task);

		// 发起人启动“流程”时填写的表单数据
		Map<String, Object> formData = utils.json2Map(task.get(Task.FORMDATA));
		// 表单项中追加流程“发起人”的员工及部门等信息，以备用
		formData.put(Emp.INFO, sponsor);

		// 当前用户提交“通过”状态
		initFlowEngine(params, formData);
		return approveNode(params, task);
	}

	/**
	 * 完成当前节点审批，根据流程编号列出审批候选人<br>
	 * 参数：{taskId,taskFlowNo,workNodeNo,workComment[,candidates]}
	 */
	private Object approveNode(Map<String, Object> params, Map<String, String> task) {

		// 2.从当前节点递归遍历全部节点及“候选人”，更新节点“完成”状态、并选出下一个节点
		flowEngine.confirmNode(params);

		// 3.选出当前第一个有效节点有多个“候选人”时，返给前台进行交互确认
		Map<String, Object> result = getCandidates(params);
		if (result != null)
			return result;

		// 4.保存“审批中”的工作节点，并更新节点状态、审批意见
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 更新待审批节点（参考数据来源：getFlowNodes）
		taskSvc.saveWorkNodes(params, workNodes);

		// 5.更新流程表的“待审批”或“就绪”的中途节点taskNodeNo
		// 补充信息：taskNodeNo、若完成（taskState）
		flowEngine.setWaitingNode(params);
		// 5.1.更新任务信息及状态
		taskSvc.setTaskState(params);

		// 6.处理需要跳过（忽略）的其他并行节点（事务失败不影响）
		skipParallelNodes(params);

		// 7.会签节点“未满签”时，节点状态更新后需要二次校验
		// 当jointlyNode==null返回未完成false时跳过处理
		// 会签“父节点”jointlyNode!=null、存在“未通过”（或驳回）的节点属正常
		// 若不存在，即“会签”完成，则不正常需要“回滚”重新处理
		if (taskSvc.jointlyCompleted(flowEngine.getJointlyNode())) {
			// 并发冲突，需要终止事务重新处理
			throw new FailedException("isRollback");
			// 如果“未满签”时，有可能兄弟节点“已驳回”，无不良影响
		}

		// 8.发送通知消息
		flowSvc.notifyAll(workNodes);

		// 流程处理后，执行该类别的“后续操作”
		if (flowEngine.isSucceed()) {
			 task.put(Task.STATE, FlowUtil.STATE_COMPLETE);
			 flowSvc.followProcess(task);
		}

		return utils.success();
	}

	/**
	 * 处理需要跳过（忽略）的其他并行节点（事务失败不影响）
	 * <p>
	 * 跳过“父节点”下没有处理的并行子节点状态为‘3’
	 */
	private void skipParallelNodes(Map<String, Object> where) {
		Set<String> skipNodes = flowEngine.getSkipNodes();
		if (!skipNodes.isEmpty()) {
			log.debug("> skipParallelNodes...{}", skipNodes);
			Map<String, Object> cond = utils.sameId(where, Task.KEY);
			// 更新状态：就绪‘0’->>跳过‘3’
			cond.put("stateReady", FlowUtil.STATE_READY); // 原状态WORK_STATE
			cond.put("stateSkip", FlowUtil.STATE_SKIP); // 更新WORK_STATE
			// 跳过的节点
			cond.put("skipNodes", skipNodes);
			// BRK_APPRO JOIN BRK_NODE
			// 更新需要跳过（忽略）的其他并行节点
			flowDao.skipParallelNodes(flowSvc.workJoinNode(), cond);
		}
	}

	/**
	 * 当用户完成当前节点驳回
	 * 参数：{taskId,workNodeNo,workComment[,candidates]}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object rejectNode() {
		// 先补充参数：taskComp\workAgent
		Map<String, Object> params = taskSvc.getWorkParams();
		// 再读取待办的任务节点信息, 并补充params参数：workEmpId、taskFlowNo
		Map<String, String> task = taskSvc.getReadyTask(params);

		// 1.然后处置当前审批就绪节点：设为驳回
		handleNode(params, FlowUtil.STATE_REJECT);
		return rejectNode(params, task);
	}

	/**
	 * 处置当前“就绪”节点，并返回当前节点信息：<br>
	 * 移交STATE_HANDOVER\加签STATE_COMPLETE\驳回STATE_REJECT\追问
	 */
	private Map<String, String> handleNode(Map<String, Object> params, String workState) {
		// 提交“加签”“驳回”“移交”状态时，不必关心表单内容。
		// 1.初始化流程引擎，这里也用到了：taskFlowNo
		initFlowEngine(params, null);

		// 当前“审批中”的节点及当前用户审批人
		Object workNo = params.get(Work.NODENO);
		Object empId = params.get(Work.EMPID);
		// 遍历节点，找到当前“就绪”节点
		Map<String, String> theNode = null;
		for (Map<String, String> node : flowEngine.getFlowNodes()) {
			// 仅处理“就绪”节点，忽略非“就绪”节点
			if (!FlowUtil.STATE_READY.equals(node.get(Work.STATE)))
				continue;

			// 定位当前“审批中”的节点，即状态“就绪”且分配给当前用户的节点
			// 校验workEmpId以防止越权操作
			if (empId.equals(node.get(Work.EMPID)) && node.get(Node.NO).equals(workNo)) {
				// 非“代理人”审批时，需清空“代理人”
				setAgentEmpty(params, node.get(Work.UID));

				// 处置当前“就绪”节点，更新状态：
				// STATE_COMPLETE\STATE_REJECT\STATE_HANDOVER
				flowEngine.addWorkNode(node, workState);
				// 返回当前的节点信息
				theNode = node;
//			} else {
			} else if (FlowUtil.STATE_REJECT.equals(workState)) {
				// “驳回”时，忽略（跳过）当前父节点下未处理的（会签或并行）节点
				// “移交”“加签”时，无需此处理，“追问”时，需要等待返回信息再处理
				flowEngine.addSkipNode(node.get(Node.PARENT));
			}
		}
		if (theNode == null)
			throw new FailedException(FlowUtil.IS_LOST_WAY);
		return theNode;
	}

	/**
	 * 当用户完成当前节点驳回：taskId,taskFlowNo,workNodeNo,workComment
	 */
	private Object rejectNode(Map<String, Object> params, Map<String, String> task) {
		// 3.保存“审批中”的工作节点，并更新节点状态、驳回意见
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 更新待审批节点（参考数据来源：getFlowNodes()）
		taskSvc.saveWorkNodes(params, workNodes);

		// 4.驳回后，把所有已审批过的节点workIdx++
		// 驳回后的可能再发起审批
		// 驳回后如果审批节点workidx++，会导致查不出驳回状态数据，所以此处先注释掉，改到再发起时处理。-- zhengdali
		// incrApproveIdx(params);

		// 5.在流程中暂存“1.等待中”或“0.就绪”的中途节点
		params.put(Task.STATE, FlowUtil.STATE_REJECT);
		// 5.1.更新任务信息及状态（状态=0\1草稿或等待中的任务）
		taskSvc.setTaskState(params);

		// 6.处理需要跳过（忽略）的并行节点（事务失败不影响）
		skipParallelNodes(params);

		// 7.发送通知消息
//		flowSvc.notifyAll(workNodes);

		// 流程拒绝后，执行该类别的“后续操作”
		task.put(Task.STATE, FlowUtil.STATE_REJECT);
		flowSvc.followProcess(task);

		return utils.success();
	}

	/**
	 * 未使用代理审批：非“代理人”审批时，需清空“代理人”
	 */
	public void setAgentEmpty(Map<String, Object> params, String opUid) {
		// 非“代理人”审批时，需清空“代理人”
		Object agent = params.get(Work.AGENT);
		if (agent != null && agent.equals(opUid))
			params.put(Work.AGENT, "");
	}

	/**
	 * 针对工作流中审批节点的处理<br>
	 * 自动递增审批节点表的审批序号（避免主键冲突：从大到小降序处理）
	 */
	private int incrApproveIdx(Map<String, Object> params) {
		Map<String, Object> where = taskSvc.workIf(params);
		// 补充追问场景，补充预选节点编号条件
		where.put(Work.NODENO, params.get(Chose.NODENO));
//		where.put(Work.STATE, FlowUtil.STATE_COMPLETE);
		return flowDao.incrApproveIdx(Work.TABLE, where);
	}

	/**
	 * 在当前节点追问，回到指定的节点 <br>
	 * 参数：{taskId,workNodeNo,workComment,choseNodeNo}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object probeNode() {
		// 先补充参数：taskComp\workAgent
		Map<String, Object> params = taskSvc.getWorkParams();
		// 再读取待办的任务节点信息, 并补充params参数：workEmpId、taskFlowNo
		taskSvc.getReadyTask(params);

		// 1.回退到追问的指定节点choseNodeNo，获取原审批人
		List<Map<String, String>> nodes = taskSvc.getNodeApprovers(params);
		// 注意：会签节点暂不支持追问
		if (nodes.isEmpty() || nodes.size() > 1)
			throw new FailedException();

		// 2.然后处置当前审批就绪节点：设为“等待”
		handleNode(params, FlowUtil.STATE_WAITING);

		// 3.退回后再发起，原审批记录自动更新workIdx++
		incrApproveIdx(params);

		// 4.克隆追问的其他节点，为新节点设置一个审批人，状态改为“就绪”
		for (Map<String, String> newNode : nodes) {
			setTraceInfo(newNode, FlowUtil.OP_PROBE, params);
			// 4.1.把克隆节点设为“就绪”待办
			flowEngine.addWorkNode(newNode, FlowUtil.STATE_READY);
		}
		return handover(params);
	}

	/**
	 * 便于跟踪，加签、移交、追问需补充的转移信息
	 */
	private void setTraceInfo(Map<String, String> newNode, String opType,
			Map<String, Object> params) {
//		newNode.remove(Work.CREATE);
//		newNode.put(Work.CREATE, utils.today(Const.DTF));
		// 当前发起追问的节点和审批人
		Map<String, Object> trace = new HashMap<>();
		trace.put(Node.NO, params.get(Work.NODENO));
		trace.put(Emp.ID, params.get(Work.EMPID));
		trace.put(Const.TYPE, opType);
		newNode.put(Work.TRACE, utils.obj2Str(trace));
		// 审批意见
		Object workNote = params.get(Work.NOTE);
		if (utils.isEmpty(workNote))
			workNote = params.get(Work.COMMENT);
		newNode.put(Work.NOTE, (String) workNote);
		// 清除拷贝的评论和签名信息
		newNode.put(Work.COMMENT, "");
		newNode.put(Work.ESIGN, "");
	}

	/**
	 * 在节点前驱加签，相当于在前面追加了一个审批人<br>
	 * 参数：{taskId,workNodeNo,workComment,employee}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object frontNode() {
		return handover(FlowUtil.STATE_WAITING, FlowUtil.OP_PROBE);
	}

	/**
	 * 在当前节点后加签，相当于在后面追加了一个审批人 <br>
	 * 参数：{taskId,workNodeNo,workComment,employee}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object behindNode() {
		return handover(FlowUtil.STATE_COMPLETE, FlowUtil.OP_ASIGN);
	}

	/**
	 * 当前节点任务移交给他人处理：{taskId,workNodeNo,employee}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object forgoNode() {
		return handover(FlowUtil.STATE_SKIP, FlowUtil.OP_FORGO);
	}

	/**
	 * 当前节点任务移交给他人处理：{taskId,workNodeNo,employee}
	 */
	private Object handover(String workState, String opType) { //
		// 先补充参数：taskComp\workAgent
		Map<String, Object> params = taskSvc.getWorkParams();
		
		// 再读取待办的任务节点信息, 并补充params参数：workEmpId、taskFlowNo
		taskSvc.getReadyTask(params);

		// 1.然后处置当前审批就绪节点：设为移交、加签
		Map<String, String> theNode = handleNode(params, workState);
		// 先处理当前节点,保存后将idx+1，清空workNode by yujianfang
//		handover(params);
//		incrApproveIdx(params);
//		flowEngine.getWorkNodes().clear();
		// 2.克隆当前审批节点，为新节点设置一个审批人
		cloneNodeApprover(theNode, params, opType);
		
		return handover(params);
	}


	/**
	 * 在当前节点上加签、或移交，相当于追加一个审批人<br>
	 * 加签后，理论上仅一个“就绪”的节点，移交后，理论上还是多个节点
	 */
	private Object handover(Map<String, Object> params) {
		// 3.获取所有节点
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 4.更新审批节点状态、审批意见，并加签或移交
		taskSvc.saveWorkNodes(params, workNodes);

		// 5.发送通知消息
		flowSvc.notifyAll(workNodes);
		// 返回成功
		return utils.success();
	}

	/**
	 * 为克隆的审批节点设置一个审批人
	 */
	private void cloneNodeApprover(Map<String, String> theNode, Map<String, Object> params,
			String opType) {
		Object approver = params.get(FlowUtil.EMPLOYEE);
		// 不能加签、转交给当前用户
		if (approver.equals(params.get(Work.EMPID)))
			throw new FailedException("", FlowUtil.EMPLOYEE);

		// 3.先克隆同一个审批节点
		Map<String, String> newNode = utils.obj2Map(theNode);
		setTraceInfo(newNode, opType, params);

		// 3.1.克隆的节点上设一个（前台选定的“employee”）审批人
		Map<String, Object> where = new HashMap<>();
		where.put(Const.I18N, flowEngine.getUserLang());
		where.put(Emp.COMP, newNode.get(Node.COMP));
		where.put(Emp.ID, approver);
		List<Map<String, String>> empList = orgnSvc.getEmployee(where);
		// 根据empId获取用户，理论上仅一个满足此条件的用户
		if (empList.isEmpty())
			throw new FailedException();

		// 3.2.把克隆的节点设为“就绪”待办
		// 新设节点上的审批人信息：workEmpId,workEmpName,workAgent,workOpUid..
		// empId,empName,empUid,empAgent,empMobi,empMail,empWechat
		flowEngine.setNodeApprover(newNode, empList.get(0));

		// 3.3.把克隆节点设为“就绪”待办
		flowEngine.addWorkNode(newNode, FlowUtil.STATE_READY);
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 */
	public Map<String, Object> getBizFlows() {
		// 请求参数（orgnDept或orgnEmpId）、及当前用户信息
		Map<String, Object> params = bizSvc.getBizParams();
		// 返回当前用户可发起的“流程”（或携带bizNo查询）
		return utils.success(getBizFlows(params));
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 * <p>
	 * 返回当前发起人可用的业务流程 bizNo与flowNo的对应映射
	 */
	private Map<String, String> getBizFlows(Map<String, Object> where) {
		// 当前员工及所在部门、角色、职务（含部门类型、部门层级、成本中心等信息）
		// 参数需转换一下
		where.put(Orgn.COMP, where.get(Biz.COMP));
		// 支持多语言i18n
		List<Map<String, String>> empList = orgnSvc.getEmpBy(where);
		Map<String, Object> params = utils.sameNo(where, Biz.KEY);
		if (!utils.isEmpty(where.get(Biz.STATE)))
			params.put(Biz.STATE, where.get(Biz.STATE));

		// 获取当前用户可以发起的业务流程（或携带bizNo查询）
		return bizSvc.getBizFlows(params, empList);
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 */
	public Map<String, Object> getBizModes() {
		// 请求参数（orgnDept、orgnEmpId、bizNo）、及当前用户信息
		Map<String, Object> params = bizSvc.getBizParams();

		// 用户可发起的流程：orgnDept、orgnEmpId、empUid（可能是秘书）
		Map<String, String> bizMap = getBizFlows(params);

		// 返回当前用户可发起的“业务”
		return utils.success(bizSvc.getBizModes(bizMap, params));
	}

	/**
	 * 流程概览：展示全节点及审批人<br>
	 * 两种情况：一种还未生成taskId之前，一种是有taskId之后
	 */
	public Object flowPreview() {
		// 设置初始参数，设置当前操作员信息，补充参数：taskComp
		Map<String, Object> params = taskSvc.newTaskParams();
		Map<String, String> sponsor;
		if (params.containsKey(Task.ID)) {
			// 获取工作流，条件：{taskId, taskComp}
			// 注意task中添加了返回信息taskComp
			Map<String, String> task = taskSvc.getTaskInfo(params);
			params.put(Task.FLOWNO, task.get(Task.FLOWNO));
			// 获取“发起人（非操作员）”信息
			sponsor = getSponsorInfo(task); // task中补充taskComp后
		} else {
			// 必须条件：{ taskFlowNo, taskEmpId, taskDept }
			String[] keys = { Task.FLOWNO, Task.EMPID, Task.DEPT };
			if (!utils.availParams(params, keys))
				throw new FailedException();
			// 获取“发起人（非操作员）”信息
			sponsor = getSponsorInfo(params);
		}
		// 获取全部审批节点，条件：{[taskId, ]taskComp, taskFlowNo}
		List<Map<String, String>> nodes = flowSvc.getFlowNodes(params);
		// TODO: 升级aolai版本后可以不用传递i18n参数
		flowEngine = new FlowEngine(null, null);
		flowEngine.setUserLang(params.get(Const.I18N));
		// 筛选有审批人的普通节点
		List<Map<String, Object>> list = new ArrayList<>();
		for (Map<String, String> node : nodes) {
			// 普通节点=“0”时，选出所有候选人列表
			if (!FlowUtil.STYLE_NORMAL.equals(node.get(Node.STYLE)))
				continue;
			Map<String, Object> item = utils.map2Obj(node);
			// 摘取审批人的关键信息
			List<Map<String, String>> approvers = getCandidates(node, sponsor);
			item.put(FlowUtil.CANDIDATES, getApprovers(approvers));
			list.add(item);
		}
		return utils.success(list);
	}

	/**
	 * 摘取审批人的关键信息
	 */
	private List<Map<String, String>> getApprovers(List<Map<String, String>> candidates) {
		List<Map<String, String>> approvers = new ArrayList<>();
		for (Map<String, String> item : candidates) {
			Map<String, String> emp = new HashMap<>();
			emp.put(Dept.NAME, item.get(Dept.NAME));
			emp.put(Emp.NAME, item.get(Emp.NAME));
			emp.put(Orgn.JOBNAMES, item.get(Orgn.JOBNAMES));
			emp.put(Emp.ID, item.get(Emp.ID));
			emp.put(Emp.UID, item.get(Emp.UID));
			approvers.add(emp);
		}
		return approvers;
	}

	/**
	 * 读取预选好某节点上的审批人
	 */
	public List<Map<String, String>> choseApprovers(Map<String, String> node) {
		// 跳转到taskSvc处理
//		if (FlowUtil.canPreSelect() && node.get(Task.ID) != null)
		if (node.get(Task.ID) != null)
			return taskSvc.choseApprovers(node);
		return new ArrayList<>();
	}

}
