package cn.ps1.soar.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.ps1.aolai.entity.Dept;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.service.AolaiService;
import cn.ps1.aolai.service.UtilsService;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.soar.entity.Biz;
import cn.ps1.soar.entity.Emp;
import cn.ps1.soar.entity.Form;
import cn.ps1.soar.entity.Mode;
import cn.ps1.soar.entity.Node;
import cn.ps1.soar.entity.Orgn;
import cn.ps1.soar.entity.Task;
import cn.ps1.soar.entity.Tmpl;
import cn.ps1.soar.utils.FlowUtil;

/**
 * 业务模式的维护管理
 * 
 * @author Aolai
 * @version 1.0 $Date: 2024.01.07
 * @since openjdk-1.8
 */
@Service
public class BizService {

	private static Logger log = LoggerFactory.getLogger(BizService.class);

	@Autowired
	private AolaiService aolai;
	@Autowired
	private UtilsService utils;
	@Autowired
	private HttpServletRequest req;

	/** 业务模式与流程的关联关系处理 */

	/**
	 * 根据业务模式（bizNo）查询触发条件（业务模式关联的流程）数据一览
	 */
	public Object getBizList() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Biz.COMP);
		// 查询条件：bizComp、bizNo
		return utils.success(getBizList(where));
	}

	/**
	 * 根据业务模式（bizNo）查询触发条件（业务模式关联的流程）数据一览 <br>
	 * 查询条件：bizComp、bizNo
	 */
	List<Map<String, String>> getBizList(Map<String, Object> where) {

		// 如果携带了JSON查询条件，则按需查询
		String[] bizRules = { Biz.SCOPE, Biz.JOBS };
		for (String key : bizRules) {
			Map<String, String> map = utils.obj2Map(where.get(key));
			if (map.isEmpty())
				continue;
			where.remove(key);

			// 以下是有效JSON数值处理
			// 这里map只有一个键值对，如：bizJobs=“{"jobNo00":Const.S_1}”

			Iterator<String> iterator = map.keySet().iterator();
			String fieldKey  = iterator.next();
//			String val = utils.sqlVal(map.get(fieldKey));
			String[] values = { Const.S_1, map.get(fieldKey) }; // { '1', 'X' }

			// key: ( json_extract({bizJobs},'$.all') ={} OR
			// json_extract({bizJobs},'$.jobNo00') ={})
			fieldKey = utils.jsonExt(key, fieldKey) + "{})";
			key = "(" + utils.jsonExt(key, Const.ALL) + "{} OR";
			where.put(key + fieldKey, values);
		}
		FlowUtil.invertCond(where, Biz.STATE);
		log.debug("getBizList...{}", where);

		// 排序条件：按优先级号排序，数字越大优先级越高
		Map<String, String> order = new LinkedHashMap<>();
		order.put(Biz.NO, "ASC"); // 查询条件之一
		order.put(Biz.LEVEL, "DESC");
		String[] args = { null, Biz.TABLE, "", Const.S_0 };
		return aolai.findAll(where, order, args);
	}

	/**
	 * 查询当前用户可发起的业务流程映射表，返回 bizNo与flowNo的对应映射
	 */
	Map<String, String> getBizFlows(Map<String, Object> params,
			List<Map<String, String>> empList) {
		Map<String, String> bizMap = new HashMap<>();
		// 遍历所有的业务规则，含所有bizNo
		List<Map<String, String>> bizList = getBizList(params);

		Map<String, String> bizScope, bizJobs, empJobs;
		// 之前是"deptTier"，已修改为deptLevel
		String[] keys = { Dept.TYPE, Dept.LEVEL };
		// 遍历所有业务规则
		for (Map<String, String> biz : bizList) {
			// 已有匹配好的业务规则，则忽略后续处理（忽略优先级低的规则）
			if (bizMap.containsKey(biz.get(Biz.NO)))
				continue;
			bizJobs = utils.json2Map(biz.get(Biz.JOBS));
			bizScope = utils.json2Map(biz.get(Biz.SCOPE));
			// 分别匹配岗位职责及范围
			// 如果允许全部岗位参与，则忽略后续处理
			boolean isScope = false, isJob = false;
			if (bizJobs.containsKey(Const.ALL))
				isJob = true;

			// 根据员工所在部门及岗位去匹配规则，优先匹配：deptType,deptId
			for (Map<String, String> emp : empList) {
				// 部门类型=“0”时：0.全部组织
				// 公司或部门的流程，下属子部门的员工都可以发起
				if (utils.isNil(bizScope.get(Dept.TYPE))) {
					isScope = true;
				} else if (bizScope.get(Dept.ID) != null
						&& emp.get(Dept.ID).startsWith(bizScope.get(Dept.ID))) {
					// 这里有可能：bizScope.get(Dept.ID)=“”，支持全部部门
					isScope = true;
				} else {
					// 然后再匹配：deptLevel,deptCost等
					for (String key : keys) {
						// 只要有一项能匹配即可跳出本层循环
						if (emp.get(key).equals(bizScope.get(key))) {
							isScope = true;
							break;
						}
					}
				}
				if (!isJob) {
					// 前面非“全部岗位职责”没匹配上，这里岗位职责再匹配一次
					empJobs = utils.json2Map(emp.get(Orgn.JOBS));
					for (String jobNo : empJobs.keySet()) {
						if (bizJobs.containsKey(jobNo)) {
							isJob = true;
							break;
						}
					}
				}
				// 只有两个条件同时满足时，该业务规则才能有效，该员工可以发起该流程
				if (isJob && isScope) {
					bizMap.put(biz.get(Biz.NO), biz.get(Biz.FLOWNO));
				}
			}
		}
		// 返回 bizNo与flowNo的对应映射
		return bizMap;
	}

	/**
	 * 增加一条触发条件（业务模式关联的流程）
	 * <p>
	 * 根据业务规则添加触发条件，新建一条数据：bizNo,bizScope,bizJobs[,bizLevle]
	 */
	public Map<String, String> addBiz() {
		// 请求参数、当前用户信息
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户所属公司
		data.put(Biz.COMP, user.get(User.COMP));

		Map<String, Object> cond = utils.sameNo(data, Biz.KEY);
		// 设置优先级，优先触发使用条件，数字越大优先级越高
		if (!data.containsKey(Biz.LEVEL)) {
			int level = aolai.getMaxCode(null, Biz.TABLE, Biz.LEVEL, cond);
			data.put(Biz.LEVEL, level);
		}
		cond.put(Biz.STATE, Const.S_1);
		cond.put(Biz.LEVEL, data.get(Biz.LEVEL));
		// 设置触发条件规则的校验条件
		setCheckRuleCond(data, cond);
		// 检查数据是否重复数据
		if (aolai.exists(null, Biz.TABLE, cond))
			throw new FailedException(ConfUtil.DUPL_DATA);

		// 设置一个默认值状态值“1”
		utils.resetState(data, Biz.STATE);
		// 这里必定是“编辑过”或“自定义”的业务流程
		// 状态标识bizMark=“0”时，bizFlowNo必定是从上级继承来的
		data.put(Biz.MARK, Const.S_1);

		/**
		 * 每个bizNo对应多个触发条件，每个触发条件可以设置一个bizFlowNo
		 * 
		 * 1.新建bizNo在inheritMode()中处理，默认设同名流程bizFlowNo，编辑时克隆一个预置的全局流程节点
		 * 
		 * 2.新建或从上级继承的bizFlowNo，默认标识都是bizMark=“0”， 节点数据编辑后更新为：bizMark=“1”
		 * 
		 * 3.编辑新建或从上级继承的bizFlowNo时，会先查询（克隆）一份默认全局节点数据，编辑后直接保存即可
		 * 
		 * 4.单纯新增一条业务的触发条件（关联流程），系统自动生成一个bizFlowNo编号与已有的bizNo关联即可
		 * 
		 */
		// 当前操作用户
		data.put(Biz.OPUID, user.get(User.ID));
		return addBizFlowNo(data);
	}

	/**
	 * 新增一条触发条件数据（新建一条工作流），允许尝试2次
	 */
	private Map<String, String> addBizFlowNo(Map<String, Object> data) {
		// 尝试2次
		int times = FlowUtil.TRY_TIMES;
		while (times-- > 0) {
			// 设置一个流程编码，自动生成一个编码
			data.put(Biz.FLOWNO, Digest.uuid8());
			try {
				// 新增一条触发条件数据
				return aolai.addRecord(null, Biz.TABLE, data);
			} catch (Exception e) {
				// 再尝试一次
				log.error("addBizFlowNo...{}", e.getMessage());
			}
		}
		// 尝试两次后失败，系统有并发冲突
		throw new FailedException(FlowUtil.DO_FAILED);
	}

	/**
	 * 修改一条触发条件（业务模式关联的流程），或关闭触发条件
	 */
	public Map<String, String> setBiz() {
		// 请求参数、当前操作用户
		Map<String, Object> data = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		data.put(Biz.COMP, user.get(User.COMP));

		// 检查重复数据：排除当前业务流程：bizFlowNo
		Map<String, Object> cond = utils.sameNo(data, Biz.KEY);
		cond.put(utils.pHolder(Biz.FLOWNO, Const.NEQ), data.get(Biz.FLOWNO));
		if ("1".equals(String.valueOf(data.get(Biz.STATE)))) {
			cond.put(Biz.STATE, Const.S_1);
			cond.put(Biz.LEVEL, data.get(Biz.LEVEL));
		} else {
			cond.put(utils.pHolder(Biz.STATE, Const.NEQ), data.get(Biz.STATE));
		}

		// 只有修改了"bizScope"或 "bizJobs"，才检查数据是否重复
		boolean mustCheck = setCheckRuleCond(data, cond);
		if (mustCheck && aolai.exists(null, Biz.TABLE, cond))
			throw new FailedException(ConfUtil.DUPL_DATA);
		// 如果是启用流程，则需要判断已启用的流程是否有重复数据
	
		
		// 修改一条业务模式关联的流程
		cond = utils.sameNo(data, Biz.KEY);
		cond.put(Biz.FLOWNO, data.get(Biz.FLOWNO));
		// 校验条件：系统预置的数据不能修改
		cond.put(utils.pHolder(Biz.STATE, Const.NEQ), Const.S_2);

		// 矫正状态值，重置数据状态
		utils.resetState(data, Biz.STATE);
		// 当前操作用户
		data.put(Biz.OPUID, user.get(User.ID));
		// 修改一条业务模式关联的流程
		return aolai.update(null, Biz.TABLE, data, cond);
	}

	/**
	 * 设置触发条件规则的校验条件
	 */
	private boolean setCheckRuleCond(Map<String, Object> data,
			Map<String, Object> cond) {
		boolean mustCheck = false;
		
		String[] bizRules = { Biz.SCOPE, Biz.JOBS };
		for (String key : bizRules) {
			Map<String, String> map = utils.obj2Map(data.get(key));
			if (map.isEmpty())
				continue;
			// 重新赋值
			data.put(key, utils.obj2Str(map));
			// 以下是有效JSON数值处理
//			key = "({" + key + "}->>'$.";
/*			for (Map.Entry<String, String> e : map.entrySet()) {
				String[] values = { utils.sqlVal(e.getValue()) };
//				cond.put(key + e.getKey() + "' ={})", values);
				key = "( " + utils.jsonExt(key, e.getKey()) + "{})";
				cond.put(key, values);
				break;
			}
*/
			Iterator<String> iterator = map.keySet().iterator();
			String fieldKey  = iterator.next();
			String fieldValue = map.get(fieldKey);
//			String[] values = { utils.sqlVal(map.get(fieldKey)) }; // { '1' }
			// fieldKey: ( json_extract({bizJobs},'$.jobNo00') ={})
			fieldKey = "( " + utils.jsonExt(key, fieldKey) + "{})";
//			cond.put(fieldKey, values);
			cond.put(fieldKey,fieldValue);
			mustCheck = true;
		
		}
		// 如禁用，则无需校验重复
		if(Const.STR_0.equals(data.get(Biz.STATE))) 
			return false;
		return mustCheck;
	}

	/**
	 * 删除一条触发条件（业务模式关联的流程）：bizNo,bizFlowNo
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> delBiz() {
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Biz.COMP);
		return delBiz(where);
	}

	/**
	 * 删除一条触发条件（业务模式关联的流程）：bizNo,bizFlowNo
	 */
	private Map<String, String> delBiz(Map<String, Object> where) {
		// 已发起过流程的，不能再删除
		if (bizIsStarted(where))
			return utils.failed(ConfUtil.CANT_REMOVE);
		// 未发起过任务的bizNo，可以删除flowNo流程节点
		Map<String, String> result = delFlowNodes(where);
		log.debug("delFlowNodes...{}", result);

		// 系统预置的数据不能删除
		where.put(utils.pHolder(Biz.STATE, Const.NEQ), Const.S_2);

		// 删除flowNo流程节点后，删除一条触发条件（业务规则）
		// if (!aolai.exists(null, Biz.TABLE, where))
		// throw new FailedException(FlowUtil.CANT_REMOVE);
		return aolai.delete(null, Biz.TABLE, where);
	}

	/**
	 * 业务模式（工作流），是否存在已发起的任务
	 */
	private <T> boolean bizIsStarted(Map<String, T> where) {
		// 已经在用的业务流程提示不能删除
		Map<String, Object> cond = FlowUtil.flowCond(where, Biz.KEY, Task.KEY);
		return taskIsExist(cond);
	}

	/**
	 * 任务是否存在
	 */
	private boolean taskIsExist(Map<String, Object> cond) {
		return aolai.exists(null, Task.TABLE, cond);
	}

	/**
	 * 是否存在已发起的任务：key="modeTmplNo"\"modeFormNo"
	 */
	private <T> boolean taskIsStarted(Map<String, T> where, String key) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Task.COMP, where.get(Mode.COMP));
//		String[] keys = { "{", Task.MODE, "}->>'$.", key, "' =" };
//		cond.put(utils.arr2Str(keys), where.get(key));
		cond.put(utils.jsonExt(Task.MODE, key), where.get(key));
		return taskIsExist(cond);
	}

	/**
	 * 根据流程编号bizFlowNo删除工作流节点
	 */
	private <T> Map<String, String> delFlowNodes(Map<String, T> where) {
		// 已经在用的业务流程提示不能删除
		Map<String, Object> cond = FlowUtil.flowCond(where, Biz.KEY, Node.KEY);
		return aolai.delete(null, Node.TABLE, cond);
	}

	/** 业务模式分组 */

	/**
	 * 根据条件查询业务模式分组一览
	 */
	public Map<String, Object> getModeList() {
		// 请求参数、当前用户信息
		Map<String, Object> where = utils.jsonParams(req);
		utils.setUserComp(req, where, Mode.COMP);
		List<Map<String, String>> list = getModeList(where);
		return utils.success(list);
	}

	/**
	 * 根据条件查询业务模式分组一览
	 */
	List<Map<String, String>> getModeList(Map<String, Object> where) {
		// 正则匹配名称
		FlowUtil.likeCond(where, Mode.NAME);
		// 这里需要返回前台所有数据
		// 默认查询有效的数据
		FlowUtil.invertCond(where, Mode.STATE);

		// 放弃超级顶点的用法
		// 忽略超级顶点
		// where.put("{modeId} !=", Const.S_0);
		String[] args = { null, Mode.TABLE, "getModeList", Const.S_0 };
		return aolai.findAll(where, null, args);
	}

	/**
	 * 增加一条业务模式或分组：modeName,modeType,modePid
	 */
	public Map<String, String> addMode() {
		// 请求参数、当前用户信息
		Map<String, Object> mode = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户及所属公司
		mode.put(Mode.COMP, user.get(User.COMP));
		mode.put(Mode.OPUID, user.get(User.ID));
		// 检查是否有重名
		String[] keys = { Mode.ID, Mode.COMP, Mode.NAME, Mode.PID };
		if (aolai.exists(Mode.TABLE, utils.sameIf(mode, keys)))
			throw new FailedException(ConfUtil.DUPL_NAME);
		utils.resetState(mode, Mode.TYPE); // 类型：0.分组 1.业务

		// 继承父级数据：新增数据时自动生成ID，然后继承（含触发条件）父级数据
		if (!Const.S_0.equals(mode.get(Mode.PID))) {
			// 继承父级数据：modeFormNo,modeTmplNo,modeBizNo
			Map<String, String> parent = getParentMode(mode);
			for (Map.Entry<String, String> e : parent.entrySet()) {
				// 已有的属性不必继承，没有的属性需要继承
				if (!mode.containsKey(e.getKey()))
					mode.put(e.getKey(), e.getValue());
			}
		}
		// 设置一个默认状态值：“1”启用
		mode.put(Mode.STATE, Const.S_1);
		// 新增一条业务模式或分组数据
		return addMode(mode);
	}

	/**
	 * 新增业务模式或分组数据，允许尝试2次
	 */
	private Map<String, String> addMode(Map<String, Object> data) {
		// 尝试2次
		int times = FlowUtil.TRY_TIMES;
		while (times-- > 0) {
			// {MODE_COMP==0, MODE_PID==02, MODE_NAME==交通费}
			// 在Pid同一级自动生成编码，每层编码一律按2位计算
			String modeId = newModeId(data);
			data.put(Mode.TIER, modeId.length() / FlowUtil.MODE_TIER_W);
			data.put(Mode.ID, modeId);
			// 设置第一层级的初始数据
			oneTierModeData(data, modeId);
			try {
				// 新增一条岗位数据
				return aolai.addRecord(null, Mode.TABLE, data);
			} catch (Exception e) {
				// 再尝试一次
				log.error("addMode...{}", e.getMessage());
			}
		}
		// 尝试两次后失败，系统有并发冲突
		throw new FailedException(FlowUtil.DO_FAILED);
	}

	/**
	 * 设置第一层级的初始数据
	 */
	private void oneTierModeData(Map<String, Object> data, String id) {
		// 默认第一层“关闭”继承，二层以上“开启”继承
		// 默认第一层不继承上级数据
		if (id.length() == FlowUtil.MODE_TIER_W) {
			// 第一层关闭继承
			data.put(Mode.INFORM, Const.S_0);
			data.put(Mode.INFLOW, Const.S_0);
			data.put(Mode.INTMPL, Const.S_0);

			// 默认一个新表单编号、业务编号、模板编号
			data.put(Mode.FORMNO, Form.NO + id); // 默认编号
			data.put(Mode.BIZNO, Biz.NO + id); // 默认编号
			data.put(Mode.TMPLNO, Tmpl.NO + id); // 默认编号

			// 插入一条默认的全局可用的触发条件规则
			// bizNo,bizScope,bizJobs,bizMark='0',bizOpUid
			Map<String, Object> bizData = new HashMap<>();
			bizData.put(Biz.NO, data.get(Mode.BIZNO));
			bizData.put(Biz.COMP, data.get(Mode.COMP));
			bizData.put(Biz.OPUID, data.get(Mode.OPUID));
			bizData.put(Biz.FLOWNO, "flowNo" + id); // 默认编号
			bizData.put(Biz.SCOPE, "{\"" + Dept.TYPE + "\":\"0\"}"); // 默认全部组织
			bizData.put(Biz.JOBS, "{\"all\":\"1\"}"); // 默认全部岗位
			aolai.addRecord(null, Biz.TABLE, bizData);
		}
	}

	/**
	 * 查询父层级的业务数据
	 */
	private Map<String, String> getParentMode(Map<String, Object> mode) {
		// 查询父层级的业务模式
		Map<String, Object> cond = FlowUtil.parent(Mode.KEY, mode);
		// 查全局超级节点数据
		if (Const.S_0.equals(cond.get(Mode.ID)))
			cond.put(Mode.COMP, Const.S_0);
		Map<String, String> parent = aolai.findOne(null, Mode.TABLE, cond,
				"getModeInfo");
		// 父层不存在或是“叶子”节点都无效
		if (parent.containsKey(Const.STS) || Const.S_1.equals(parent.get(Mode.TYPE)))
			throw new FailedException();
		return parent;
	}

	/**
	 * 移动一条业务模式到其另一个分组，满足条件必须在同一个大的业务分组之下
	 */
	public Map<String, String> moveMode() {
		Map<String, Object> mode = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		mode.put(Mode.COMP, user.get(User.COMP));
		// 当前数据
		Map<String, Object> cond = utils.sameId(mode, Mode.KEY);
		Map<String, String> self = aolai.findOne(null, Mode.TABLE, cond);

		// 检查同一父级下是否有重名数据
		mode.put(Mode.NAME, self.get(Mode.NAME));
		String[] keys = { null, Mode.COMP, Mode.NAME, Mode.PID };
		if (aolai.exists(null, Mode.TABLE, utils.sameIf(mode, keys)))
			throw new FailedException(ConfUtil.DUPL_NAME);

		// 继承父级数据
		Map<String, String> parent = getParentMode(mode);
		Map<String, Object> fields = new HashMap<>();
		// 当前操作用户
		fields.put(Mode.OPUID, user.get(User.ID));
		fields.put(Mode.TAG, parent.get(Mode.TAG));
		fields.put(Mode.PID, parent.get(Mode.ID)); // 目标父集
		if (Const.S_1.equals(self.get(Mode.INFORM)))
			fields.put(Mode.FORMNO, parent.get(Mode.FORMNO));
		if (Const.S_1.equals(self.get(Mode.INTMPL)))
			fields.put(Mode.TMPLNO, parent.get(Mode.TMPLNO));
		if (Const.S_1.equals(self.get(Mode.INFLOW)))
			fields.put(Mode.BIZNO, parent.get(Mode.BIZNO));

		// 在父级modePid下自动生成编码，默认每层编码2位字符
		String modeId = newModeId(mode);
		fields.put(Mode.ID, modeId); // 重置当前编码
		// 重置当前层级
		fields.put(Mode.TIER, modeId.length() / FlowUtil.MODE_TIER_W);

		// 修改当前数据编码
		return aolai.update(null, Mode.TABLE, fields, cond);
	}

	/**
	 * 在Pid同一级自动生成编码
	 */
	private String newModeId(Map<String, Object> data) {
		// 在Pid同一级自动生成编码，每层编码一律按2位计算
		Map<String, Object> cond = FlowUtil.brother(Mode.KEY, data);
		int id = aolai.getMaxCode(null, Mode.TABLE, Mode.ID, cond);
		return utils.idCode(data.get(Mode.PID), id, FlowUtil.MODE_TIER_W);
	}

	/**
	 * 删除一条业务模式或分组：modeId
	 */
	@Transactional
	public Map<String, String> delMode() {
		Map<String, Object> where = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		where.put(Mode.COMP, user.get(User.COMP));

		// 不能删除系统预置数据：modeComp\modeId
		Map<String, String> mode = getMode(where);
		if (Const.S_2.equals(mode.get(Mode.STATE)))
			throw new FailedException(ConfUtil.CANT_REMOVE);
		else if (mode.containsKey(Const.STS))
			throw new FailedException();

		// 下级还有子业务，父业务类型也不能删除
		if (Const.S_0.equals(mode.get(Mode.TYPE))) {
			Map<String, Object> cond = FlowUtil.family(Mode.KEY, where);
			if (aolai.count(null, Mode.TABLE, cond) > 1)
				throw new FailedException(ConfUtil.CANT_REMOVE);
		}
		// taskIsStarted 已经发起过流程不允许删除
		// 此处解决如果业务类型继承父级，发起流程后可以删除业务类型bug
		if (taskIsStarted(mode, Mode.ID)) {
			throw new FailedException(ConfUtil.CANT_REMOVE);
		}
		// 删除冗余触发条件
		if (!delLostData(Mode.BIZNO, mode))
			throw new FailedException(ConfUtil.CANT_REMOVE);
		// 删除冗余表单
		delLostData(Mode.FORMNO, mode);
		// 删除冗余模板
		delLostData(Mode.TMPLNO, mode);
		// 删除一条业务模式或分组信息
		return aolai.delete(null, Mode.TABLE, where);
	}

	/**
	 * 根据（modeId）模式编码查询一条业务模式
	 */
	public Map<String, String> getMode(Map<String, Object> where) {
		Map<String, Object> cond = utils.sameId(where, Mode.KEY);
		return aolai.findOne(null, Mode.TABLE, cond);
	}

	/**
	 * 修改一条业务模式或分组数据，其中modeState=0禁用\1启用\2系统预制
	 */
	public Map<String, String> setMode() {
		// 请求参数、当前操作用户
		Map<String, Object> params = utils.jsonParams(req);
		// 当前操作用户及所属公司：注意区分params和data
		Map<String, String> user = utils.userSelf(req);
		params.put(Mode.COMP, user.get(User.COMP));

		// 只能修改以下数据，修改表单、流程、模板另调其他接口（setInheritMode）
		String[] keys = { Mode.NAME, Mode.TAG, Mode.IDX, Mode.ICON };
		Map<String, Object> data = new HashMap<>();
		for (String key : keys) {
			// 忽略冗余参数
			if (params.containsKey(key))
				data.put(key, params.get(key));
		}
		if (data.isEmpty())
			return utils.invalidParams();

		String userId = user.get(User.ID);
		data.put(Mode.OPUID, userId);

		// 检查在同一父层内不能有重名
		if (data.containsKey(Mode.NAME)) {
			keys = new String[] { Mode.ID, Mode.PID, Mode.COMP, Mode.NAME };
			if (aolai.exists(null, Mode.TABLE, utils.sameIf(params, keys)))
				throw new FailedException(ConfUtil.DUPL_NAME);
		}
		// 同时更新当前级&子级标签“modeTag”
		if (data.containsKey(Mode.TAG)) {
			// 如修改本层级业务标签，则子级的业务标签全部修改
			Map<String, Object> cond = FlowUtil.family(Mode.KEY, params);
			Map<String, Object> fields = new HashMap<>();
			fields.put(Mode.OPUID, userId);
			fields.put(Mode.TAG, data.get(Mode.TAG));
			aolai.update(null, Mode.TABLE, fields, cond);
		}
		// 修改一条业务模式或分组数据，更新当前数据
		Map<String, Object> where = utils.sameId(params, Mode.KEY);
		aolai.update(null, Mode.TABLE, data, where);
		return utils.success();
	}

	/**
	 * 检查自己本身是否被修改，如父级被禁用，则自己不允许修改
	 * @param params
	 */
	private void checkCanModifyState(Map<String,Object> params ) {
		// 如果修改了modeState，则需判断父级的modeState是否为禁用，如果父级为禁用则不允许启用
		Map<String, Object> parent = FlowUtil.parent(Mode.KEY, params);
		// modeState 为 1 或者2都代表父级状态正常
//		String[] keys = { Mode.STATE, Mode.STATE };
		String[] values = { Const.S_1, Const.S_2 };
		parent.put(utils.pHolder(Mode.STATE, Const.IN), values);
		if (!aolai.exists(Mode.TABLE, parent))
			throw new FailedException(ConfUtil.CANT_UPDATE);
	}

	/**
	 * 修改一条业务模式或分组数据的状态
	 */
	public Map<String, String> setModeState() {
		// 请求参数、当前操作用户
		Map<String, Object> params = utils.jsonParams(req);
		// 当前操作用户及所属公司：注意区分params和data
		Map<String, String> user = utils.userSelf(req);
		params.put(Mode.COMP, user.get(User.COMP));
		String userId = user.get(User.ID);
		utils.resetState(params, Mode.STATE);

		// 如果从“禁用”切换到“启用”，仅“启用”当前级，子级保持原状态
		// 只有从“启用”切换到“禁用”，子业务类型也全部“禁用”
		Map<String, Object> fields = new HashMap<>();
		fields.put(Mode.OPUID, userId);
		fields.put(Mode.STATE, params.get(Mode.STATE));

		// 修改一条业务模式或分组数据，更新当前数据
		Map<String, Object> where = utils.sameId(params, Mode.KEY);
		Map<String, String> result;
		if (Const.S_0.equals(params.get(Mode.STATE))) {
			
			// 如修改本层级业务标签，则子级的业务标签全部修改
			Map<String, Object> cond = FlowUtil.family(Mode.KEY, params);
			// 原状态为“启用”的才可以修改为“禁用”
			// 同时更新当前级&子级
			// 为了预防重复修改预制数据的状态
			cond.put(Mode.STATE, Const.S_1);
			result = aolai.update(null, Mode.TABLE, fields, cond);
		} else {
			// 此处如果父级未启用，则本身也不允许启用
			checkCanModifyState(params);
			where.put(Mode.STATE, Const.S_0);
			result = aolai.update(null, Mode.TABLE, fields, where);
		}
		if (utils.isSuccess(result))
			return utils.success();
		return utils.invalidParams();
	}

	/**
	 * 只有二级、三级才可以开启或关闭继承模式：modeId,modePid
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> setInheritMode() {
		Map<String, Object> mode = utils.jsonParams(req);
		// 当前操作用户
		Map<String, String> user = utils.userSelf(req);
		// 当前操作用户及所属公司
		mode.put(Mode.COMP, user.get(User.COMP));
		mode.put(Mode.OPUID, user.get(User.ID));

		// 三种情况：继承表单modeInForm、继承流程modeInFlow、继承模板modeInTmpl
		// 获取父级的数据，再克隆
		if (mode.containsKey(Mode.INFORM) && mode.containsKey(Mode.FORMNO)) {
			// 继承表单
			return inheritMode(mode, Mode.INFORM, Mode.FORMNO);

		} else if (mode.containsKey(Mode.INFLOW)
				&& mode.containsKey(Mode.BIZNO)) {
			// 继承流程
			return inheritMode(mode, Mode.INFLOW, Mode.BIZNO);

		} else if (mode.containsKey(Mode.INTMPL)
				&& mode.containsKey(Mode.TMPLNO)) {
			// 继承打印模板
			return inheritMode(mode, Mode.INTMPL, Mode.TMPLNO);

		} else {
			// 参数无效
			return utils.invalidParams();
		}
	}

	/**
	 * 只有二级、三级才可以开启或关闭继承模式
	 * <P>
	 * 新建数据时，系统默认“启用”继承使用拷贝父级的表单数据
	 * <P>
	 * 此处设为“禁用”（即关闭“启用”）时，拷贝BizNo\TmplNo\FormNo表中插入一条新编号数据，
	 * <P>
	 * 用新编号更新当前业务（Mode）表数据，同时更新“启用”继承的下级数据，全删不再用的数据
	 * <P>
	 * 再“启用”继承时，使用拷贝父级表单数据，同时更新其“启用”继承的下级数据，并删除旧数据
	 */
	private Map<String, String> inheritMode(Map<String, Object> mode,
			String inKey, String numKey) {
		// 当前层级数据
		// modeId,modePid,modeInFlow,modeBizNo[,modeComp,modeOpUid]
		Map<String, Object> fields = new HashMap<>();
		fields.put(Mode.OPUID, mode.get(Mode.OPUID));

		// 继承表单（modeInForm），含所有开启继承的子级（“modeInForm”）
		// 分别三种情况处理：inKey = “modeInForm”、“modeInFlow”、“modeInTmpl”
		// 每次仅单独“启用”一种情况：modeFormNo、modeBizNo、modeTmplNo

		// 只有二级、三级才可以启用继承，如：modeInFlow=“1”
		boolean isInherit = Const.S_1.equals(mode.get(inKey));
		// 继承模板：继承父级业务规则（删除自定义的规则）
		if (isInherit) {
			/**
			 * 清除冗余业务规则，已发起过任务的bizNo，提示不能再删除
			 * 否则，未发起过任务的bizNo，先删除flowNo流程节点，再删除业务规则bizNo
			 */
			if (!delLostData(numKey, mode))
				throw new FailedException("cantInherit");

			// 直接继承parent的数值（如：modeBizNo）
			Map<String, String> parent = getParentMode(mode);
			fields.put(numKey, parent.get(numKey)); // modeBizNo

		} else {
			// 自定义模板：并复制数据到新模板
			// 每次“禁用”时，必须携带对应的modeBizNo参数值，否则提示“无效参数”
			// 例如：如果“禁用”继承的是"modeBizNo"，需要克隆新建一个bizNo
			fields.put(numKey, cloneMode(numKey, mode)); // modeBizNo
		}

		// 为保持幂等性：批处理只更新下级“启用”的子级
		// 理论上3层没有下级，因此不需要操作子级（操作子集会报modePid超长）
		int len = String.valueOf(mode.get(Mode.ID)).length();
		// 目前设计最大支持3层
		if (len / FlowUtil.MODE_TIER_W < 3) {
			Map<String, Object> cond = FlowUtil.child(Mode.KEY, mode);
			cond.put(inKey, Const.S_1); // modeInFlow“开启”继承的
			aolai.update(null, Mode.TABLE, fields, cond);
		}

		// 再处理当前级，使用拷贝的数据，更新当前数据为新建itemNo
		Map<String, Object> where = utils.sameId(mode, Mode.KEY);
		// 增加当前级（二或三级）对应约束条件
		where.put(inKey, isInherit ? Const.S_0 : Const.S_1);

		// 设为“开启”、“关闭”
		fields.put(inKey, isInherit ? Const.S_1 : Const.S_0);
		return aolai.update(null, Mode.TABLE, fields, where);
	}

	/**
	 * 清除冗余业务规则数据：modeId,modePid,modeBizNo(modeFormNo\modeTmplNo)
	 * 
	 * @param <T>
	 */
	private <T> boolean delLostData(String numKey, Map<String, T> mode) {
		// 一条自定义的modeNo对应一条bizNo，子级mode也继承该bizNo
		// “启用”继承，子集也将继承爷爷级bizNo，并弃用原有自定义的bizNo
		Map<String, Object> where;
		if (Mode.BIZNO.equals(numKey)) {
			// 如果是继承的父级，则不做删除处理
			if (Const.S_1.equals(mode.get(Mode.INFLOW)))
				return true;

			// 查询触发条件一览：bizNo\bizFlowNo\bizMark=Const.S_1
			where = FlowUtil.modeCond(mode, utils.initCap(Biz.KEY), Biz.KEY);
			List<Map<String, String>> items = getBizList(where);

			for (Map<String, String> biz : items) {
				// 自定义的触发条件，只有bizMark=“1”的才可能是发起过流程
				if (Const.S_1.equals(biz.get(Biz.MARK))) {
					// 已发起过流程的，不能再删除
					if (bizIsStarted(biz))
						return false;
					// 未发起过任务的bizNo，可以删除flowNo流程节点
					delFlowNodes(biz);
				}
			}
			// 删除flowNo流程节点后，删除多条触发条件（业务规则）
			aolai.delete(null, Biz.TABLE, where);
		} else if (Mode.TMPLNO.equals(numKey) && !taskIsStarted(mode, numKey)) {
			// 如果是继承的父级，则不做删除处理
			if (Const.S_1.equals(mode.get(Mode.INTMPL)))
				return true;
			// 删除冗余
			String uTmpl = utils.initCap(Tmpl.KEY);
			where = FlowUtil.modeCond(mode, uTmpl, Tmpl.KEY);
			aolai.delete(null, Tmpl.TABLE, where);
		} else if (Mode.FORMNO.equals(numKey) && !taskIsStarted(mode, numKey)) {
			// 如果是继承的父级，则不做删除处理
			if (Const.S_1.equals(mode.get(Mode.INFORM)))
				return true;
			// 删除冗余
			String uForm = utils.initCap(Form.KEY);
			where = FlowUtil.modeCond(mode, uForm, Form.KEY);
			aolai.delete(null, Form.TABLE, where);
		}
		return true;
	}

	/**
	 * 克隆一条数据，新建bizNo，并复制业务模式关联的流程
	 * <p>
	 * 从modeBizNo\modeFormNo\modeTmplNo对应表的拷贝一条既有数据
	 */
	private String cloneMode(String numKey, Map<String, Object> mode) {
		// 新建itemNo，克隆一条数据，并复制原始数据
		// 分别从modeBizNo\modeTmplNo\modeFormNo对应表中克隆既有数据
		if (Mode.BIZNO.equals(numKey)) {
			// 根据modeBizNo克隆业务流程触发条件
			// return cloneBizData(mode);
			String uBiz = utils.initCap(Biz.KEY);
			return cloneData(mode, Biz.TABLE, uBiz, Biz.KEY);
		} else if (Mode.FORMNO.equals(numKey)) {
			// 根据modeFormNo克隆流程表单
			String uForm = utils.initCap(Form.KEY);
			return cloneData(mode, Form.TABLE, uForm, Form.KEY);
		} else if (Mode.TMPLNO.equals(numKey)) {
			// 根据modeTmplNo克隆打印模板
			String uTmpl = utils.initCap(Tmpl.KEY);
			return cloneData(mode, Tmpl.TABLE, uTmpl, Tmpl.KEY);
		}
		return Digest.uuid8();
	}

	/**
	 * 克隆数据（打印模板、流程表单等）
	 */
	private String cloneData(Map<String, Object> mode, String table,
			String src, String key) {
		Map<String, Object> cond = FlowUtil.modeCond(mode, src, key);
		// 含预置数据
		cond.put(utils.pHolder(key + FlowUtil.STATE, Const.NEQ), Const.S_0);
		// 获取被克隆的原始数据
		String[] args = { null, table, "", Const.S_0 };
		List<Map<String, String>> list = aolai.findAll(cond, null, args);
		String newItemNo = Digest.uuid8();
		if (!list.isEmpty()) {
			// 进行格式转换及调整
			List<Map<String, Object>> items = new ArrayList<>();
			for (int i = 0; i < list.size(); i++) {
				Map<String, String> item = list.get(i);
				item.put(key + FlowUtil.NO, newItemNo);
				item.put(key + FlowUtil.STATE, Const.S_1); // 新数据设为非预置“1”
				if (Biz.KEY.equals(key))
					putBizItem(item, i);

				// 更新当前操作用户
				item.put(key + FlowUtil.OPER, (String) mode.get(Mode.OPUID));
				// 进行格式转换
				items.add(utils.map2Obj(item));
			}
			try {
				aolai.batchAdd(null, table, items, null, false);
			} catch (Exception e) {
				throw new FailedException(FlowUtil.DO_FAILED);
			}
		}
		// 返回新业务编号
		return newItemNo;
	}

	/**
	 * 设置新建的触发条件的属性值
	 */
	private void putBizItem(Map<String, String> item, int incr) {
		// 新克隆数据默认设置为初始值：“未编辑”
		item.put(Biz.MARK, Const.S_0);
		item.put(Biz.FLOWNO, Digest.uuid8(incr));
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 */
	List<Map<String, String>> getBizModes(Map<String, String> bizMap,
			Map<String, Object> params) {
		// 根据业务触发条件，根据优先级，获取有优先权限的业务流程
		List<Map<String, String>> flowList = new ArrayList<>();
		if (!bizMap.isEmpty()) {
			// 获取所有业务流程的管辖范围和岗位职责
			Map<String, Object> where = new HashMap<>();
			where.put(Mode.COMP, params.get(Biz.COMP));
			where.put(utils.pHolder(Mode.STATE, Const.NEQ), Const.S_0);

			// 根据modeId模糊匹配下级mode
			if(!utils.isEmpty(params.get(Mode.ID))){
				String key = utils.pHolder(Mode.ID, "LIKE");
				where.put(key, params.get(Mode.ID) + "%");
			}
			// modeType
			if (!utils.isEmpty(params.get(Mode.TYPE))) {
				where.put(Mode.TYPE, params.get(Mode.TYPE));
			}

			// 模式一览
			List<Map<String, String>> modeList = getModeList(where);
			for (Map<String, String> mode : modeList) {
				String bizNo = mode.get(Mode.BIZNO);
				if (bizMap.containsKey(bizNo)) {
					mode.put(Biz.FLOWNO, bizMap.get(bizNo));
					flowList.add(mode);
				}
			}
		}
		return flowList;
	}

	/**
	 * 公共：查询业务的请求参数条件
	 */
	Map<String, Object> getBizParams() {
		// 请求参数（orgnDept或orgnEmpId）、及当前用户信息
		Map<String, Object> params = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		params.put(Biz.COMP, user.get(User.COMP));

//		// 默认查询当前用户或“秘书”
//		params.put(Emp.SECTY, user.get(User.ID));

		// 如果请求参数中存在 orgnEmpId ,则查询该用户的数据，否则查询登录人的数据
		// 此处解决秘书发起权限被放大问题
		if (params.containsKey(Orgn.EMPID)) {
			params.put(Emp.ID, params.get(Orgn.EMPID));
		} else {
			// 默认查询当前用户
			params.put(Emp.UID, user.get(User.ID));
		}

		return params;
	}

}
