package cn.ps1.soar.engine;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import cn.ps1.aolai.service.UtilsService;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.aolai.utils.SpringContext;
import cn.ps1.soar.entity.Emp;
import cn.ps1.soar.entity.Node;
import cn.ps1.soar.entity.Task;
import cn.ps1.soar.entity.Work;
import cn.ps1.soar.service.EventEmitter;
import cn.ps1.soar.utils.FlowUtil;

/**
 * 工作流程（任务）处理引擎
 * 
 * @author Aolai
 * @version 1.0 $Date: 2023.06.21
 *
 */
public class FlowEngine {

	// 当前的工作流程节点（FLOWNODE）一览表
	// 注意：来自FLOWNODE，与APPRONODE关联
	private List<Map<String, String>> flowNodes = new ArrayList<>();
	// 当前的审批流中·审批节点一览表
	private List<Map<String, String>> workNodes = new ArrayList<>();
	// 审批节点上的候选人列表
	private List<Map<String, Object>> candidates = new ArrayList<>();
	// 审批表单内容
	private Map<String, Object> formData = new HashMap<>();
	// 需要跳过忽略的并行节点
	private Set<String> skipNodes = new HashSet<>();
	// 控制会签的父节点
	private Map<String, String> jointlyNode = null;

	// 节点的事件驱动服务类
	private static EventEmitter event = (EventEmitter) SpringContext.getBean("eventEmitter");
	// 基础工具类
	private static UtilsService utils = (UtilsService) SpringContext.getBean("utilsService");

	// 需要单独处理的节点STATE_COMPLETE
	private boolean isSucceed = false; // 顺利完成全部流程
	/** 动态用户语言 */
	private Object userLang = "ZH";

	/**
	 * 引擎启动后，选出工作流中第一个审批节点上的候选人列表
	 */
	public void findFirstNode() {
		// 新发起的流程，会先确认节点列表是否存在
		if (flowNodes.isEmpty())
			throw new FailedException(FlowUtil.NODE_IS_EMPTY);
		/**
		 * 选出第一个节点的处理过程，默认第一个节点是“串行”节点
		 */
		pickWorkNode(flowNodes.get(0), FlowUtil.STYLE_SERIAL);
	}

	/**
	 * 根据当前节点5种类型分别处理，直到递归找到第一个普通节点
	 * <p>
	 * 0.普通 1.串行(子流程) 2.条件 3.会签 5.并行
	 * <p>
	 * 当前节点全部信息：
	 * nodeComp,nodeNo,nodeFlowNo,nodeName,nodeParent,nodeStyle,nodeTier
	 * ,nodeType,nodeStep,nodeRules,nodeEvent,taskComp,taskId,taskFlowNo
	 * ,workComp,workTaskId,workNodeNo,workState,workComment
	 * ,workESign,workUpdate,workIdx..
	 */
	private void pickWorkNode(Map<String, String> theNode, String pStyle) {
		// 审批节点会根据当前节点类型去分别处理
		String style = theNode.get(Node.STYLE);

		// 只要第一个节点是个普通节点=“0”就行，选出所有候选人列表
		// 并把当前节点状态改为“就绪”
		if (FlowUtil.STYLE_NORMAL.equals(style)) {
			// 这里先设置审批状态为“就绪”=“0”，并缓存当前“审批中”的节点
//			addWorkNode(theNode, FlowUtil.STATE_READY);

			/**
			 * 此方法addWorkNode()移到了pickCandidates()中处理
			 * 选出当前节点审批候选人列表（这里也可能是“死胡同”）
			 */
			pickCandidates(theNode, pStyle);
			return;
		}

		// 当前节点不是普通节点时，暂时置为“等待中”=“1”
		addWorkNode(theNode, FlowUtil.STATE_WAITING);

		// 再获取下一级所有子节点
		List<Map<String, String>> childs = getChildNode(theNode);
		// 如果分支节点下无子节点，则流程有问题
		if (childs.isEmpty())
			isLostWay(); // 遇到“死胡同”了

		// 如果当前节点是串行节点“1”，则跳转到下级第一个子节点
		if (FlowUtil.STYLE_SERIAL.equals(style)) {
			// 当前节点是串行节点“1”，则跳转到下级第一个子节点
			pickWorkNode(childs.get(0), style);

		} else if (FlowUtil.STYLE_CONDITION.equals(style)) {
			// 如果当前节点是条件节点“2”，根据条件分支处理
			int actionIdx = event.getActionIdx(theNode, formData);

			// 如果“条件”节点无法跳转，则流程有问题
			if (actionIdx == -1 || actionIdx >= childs.size())
				isLostWay(); // 遇到“死胡同”了
			// 分支“子节点”数量必须大于返回值才行，根据返回值跳转不同分支
			pickWorkNode(childs.get(actionIdx), style);

		} else if (FlowUtil.STYLE_JOINTLY.equals(style)
				|| FlowUtil.STYLE_PARALLEL.equals(style)) {
			// 如果当前节点是会签节点“3” or 并行节点“5”，则遍历所有子节点
			for (Map<String, String> child : childs) {
				pickWorkNode(child, style);
			}

		}
	}

	/**
	 * 当前“普通”节点的审批候选人，参数pStyle为父节点的处理模式
	 * <p>
	 * 当前节点全部信息：
	 * nodeComp,nodeNo,nodeFlowNo,nodeName,nodeParent,nodeStyle,nodeTier
	 * ,nodeType,nodeStep,nodeRules,nodeEvent,taskComp,taskId,taskFlowNo
	 * ,workComp,workTaskId,workNodeNo,workState,workComment
	 * ,workESign,workUpdate,workIdx..
	 */
	private void pickCandidates(Map<String, String> node, String pStyle) {
		/**
		 * 当前审批节点的所有审批“候选人”
		 */
		List<Map<String, String>> candidateList = event.getCandidatesBy(node, formData);

		// 这里先设置审批状态为“就绪”=“0”，并缓存当前“审批中”的节点
//		addWorkNode(node, FlowUtil.STATE_READY);

		// 如果一个节点仅一个候选人时，直接推给此人待审核
		if (candidateList.size() == 1) {
			/**
			 * 新设节点上的候选人：workEmpId,workEmpName,workAgent,workOpUid..<br>
			 * empId,empName,empUid,empAgent,empMobi,empMail,empWechat
			 */
			setNodeApprover(node, candidateList.get(0));

			// 这里先设置审批状态为“就绪”=“0”，并缓存当前“审批中”的节点
//			addWorkNode(node, FlowUtil.STATE_READY);

		} else if (candidateList.size() > 1) {
			/**
			 * 根据节点签发规则进一步处理：0.任一人 1.仅一人 2.多人 3.全员会签 <br>
			 */
			String issue = node.get(Node.ISSUE);
			if (FlowUtil.ISSUE_ONE.equals(issue) || FlowUtil.ISSUE_MORE.equals(issue)) {
				// 优先选择FF_CHOSE表中的数据，
				// 包含：empId,empName,empUid,empAgent,empMobi,empMail,empWechat
				List<Map<String, String>> list = event.choseApprovers(node);
				if (list.isEmpty()) {
					// 有多个候选人时，返回前台节点信息以及候选人列表
					Map<String, Object> candidate = new HashMap<>();
					candidate.put(FlowUtil.MORM_NODE, node);
					candidate.put(FlowUtil.CANDIDATES, candidateList);
					// 返给前台的参数parentType
					candidate.put("parentType", pStyle);
					candidates.add(candidate);
					return;
				}
				candidateList = list;
			}
			// 先删除前面新添加的最后一个节点
//			workNodes.remove(workNodes.size() - 1);
			// 扩展为支持同时推送多人的情况
			for (Map<String, String> approver : candidateList) {
				// 复制同一个节点，绑定多个审批人
				Map<String, String> sameNode = new HashMap<>(node);
				setNodeApprover(sameNode, approver);
				// 这里先设置审批状态为“就绪”=“0”，并缓存当前“审批中”的节点
//				addWorkNode(sameNode, FlowUtil.STATE_READY);
			}
		} else {
			// 遇到“死胡同”了
			isLostWay();
		}
	}

	/**
	 * 设置节点上的审批（候选）人信息，还需在addWorkNodes()克隆一次<br>
	 * <p>
	 * 新设候选人数据：workEmpId,workEmpName,workAgent,workOpUid<br>
	 * empId,empName,empUid,empAgent,empMobi,empMail,empWechat
	 */
	public void setNodeApprover(Map<String, String> node, Map<String, String> emp) {
		node.put(Work.EMPID, emp.get(Emp.ID));
		node.put(Work.EMPNAME, emp.get(Emp.NAME));
		// 这里必须填empUid，用于审批人登录后校验权限
		node.put(Work.UID, emp.get(Emp.UID));
		/**
		 * 如果设置了“代理人”，则推送“代理人”<br>
		 * 另外根据三种消息通知方式：邮件、手机、微信，推送提醒消息
		 */
		String[] keys = { "Agent", "Mobi", "Mail", "Wechat" };
		for (String key : keys)
			node.put(Work.KEY + key, emp.get(Emp.KEY + key));

		// 3.3.把克隆节点设为“就绪”待办
		addWorkNode(node, FlowUtil.STATE_READY);
	}

	/**
	 * 定位当前的“审批节点”，状态“就绪”且分配给当前用户审批的节点<br>
	 * 再寻找工作流程的下一个节点，这里一定是flowNodes.size()>0
	 */
	public void confirmNode(Map<String, Object> params) {
		// 新发起的流程，会先确认节点列表是否存在
		// 已“审批中”的工作节点号必然存在，不必确认节点是否存在
		Object workNo = params.get(Work.NODENO);
		for (Map<String, String> node : flowNodes) {
			// 一个节点有多个审批人，需校验workEmpId以防止越权操作
			// 前面getReadyTask()中已经验证了workNodeNo(workEmpId\workState)
			if (!node.get(Node.NO).equals(workNo))
				continue;
			// 定位到节点审批人
			// 提交选中“审批人”时，已在setNodeApprover()设置了：
			// workEmpId,workEmpName,workAgent,workOpUid

			String workState = node.get(Work.STATE);
			if (params.get(Work.EMPID).equals(node.get(Work.EMPID))) {
				// 只要节点状态是“就绪”就可以审批
				if (FlowUtil.STATE_READY.equals(workState)) {
					// 非“代理人”审批时，需清空“代理人”
					event.setAgentEmpty(params, node.get(Work.UID));

					/**
					 * 完成当前节点，并找出下一个合适的“待审批”节点
					 * 如果遇到会签节点则返回，继续处理下一步
					 */
					pickNextNode(node);

				} else if (FlowUtil.STATE_SKIP.equals(workState)) {
					// 状态为“3”时提示“错过审批机会”：其他并行节点先批，被跳过了。
					throw new FailedException(FlowUtil.LOST_CHANCE);
				}
			} else if (FlowUtil.STATE_READY.equals(workState)
					&& !FlowUtil.ISSUE_FULL.equals(node.get(Node.ISSUE))) {
				// 跳过处理其他并行审批节点，忽略会签节点
				addWorkNode(node, FlowUtil.STATE_SKIP);
			}
		}
		// 至少应该有一个当前节点，否则提示“无效参数”（"invalidParams"）
		if (workNodes.isEmpty())
			throw new FailedException();
	}

	/**
	 * 遍历节点列表，完成当前节点，并找出下一个合适的“待审批”节点
	 */
	private void pickNextNode(Map<String, String> theNode) {
		// 当前节点“完成”审批
		addWorkNode(theNode, FlowUtil.STATE_COMPLETE);

		// 当前节点被“追问”或“加签”，需要回到关联节点继续处理
		Map<String, String> relatedNode = getRelatedNode(theNode);
		if (relatedNode != null) {
			addWorkNode(relatedNode, FlowUtil.STATE_READY);
			return;
		}
		// 节点内实现会签，如果未全部审批完成，则需等待其他人审批
		if (FlowUtil.ISSUE_FULL.equals(theNode.get(Node.ISSUE)) && !jointlyApproved(theNode)) {
			// 未完返回，继续等待
			jointlyNode = theNode;
			return;
		}

		// 根据“父节点”类型，跳转到下一步待处理的节点
		Map<String, String> parent = getParentNode(theNode);
		if (parent == null)
			return; // 无父节点时，处理完毕

		String pStyle = parent.get(Node.STYLE);
		if (FlowUtil.STYLE_SERIAL.equals(pStyle)) {
			// “父节点”是“串行节点”时，需要遍历兄弟节点
			List<Map<String, String>> nodes = getChildNode(parent);
			// 找到当前节点的下个“兄弟”节点
			String prevNo = theNode.get(Node.NO); // 上个节点的编号
			for (int i = 0; i < nodes.size() - 1; i++) {
				// 只有一个workIdx=0的节点
				String nodeNo = nodes.get(i).get(Node.NO);
				Map<String, String> nextNode = nodes.get(i + 1);

//				if (nodeNo != null && nodeNo.equals(prevNo)) {
				// 当前节点编号与已执行的节点编号要匹配，且下个节点编号还不能与之前编号一致
				// 推送多人同时审批或加签后，会同时有多个节点审批人
				if (prevNo.equals(nodeNo) && !prevNo.equals(nextNode.get(Node.NO))) {
					/**
					 * 选择下一个串行兄弟节点，且只有一个workIdx=0的节点
					 */
					pickWorkNode(nextNode, pStyle);
					return;
				}
			}

			// 没找到下一个“兄弟”审批节点，则需要回归“父节点”找“兄弟”
			// 如果“父节点”是超级节点，则整个工作流程结束
			isSucceed = Const.S_0.equals(parent.get(Node.PARENT));

			// 完成或继续下一个
			if (isSucceed) {
				addWorkNode(parent, FlowUtil.STATE_COMPLETE);
			} else {
				// 再找“父节点”的“兄弟”节点处理
				pickNextNode(parent);
			}

		} else if (FlowUtil.STYLE_CONDITION.equals(pStyle)) {
			// 父节点是“条件”节点，由系统自动完成
			pickNextNode(parent);

		} else if (FlowUtil.STYLE_JOINTLY.equals(pStyle)) {

			// 如果父节点是“会签节点”时，系统自动判断子节点是否都完成审批
			if (jointlyApproved(parent, theNode)) {
				// 所有子节点都完成，则父节点也自动完成后，再找“父节点”的“兄弟”节点
				pickNextNode(parent);
			} else {
				// 未完成
				// 但有可能出现多人“并发”，这里实现幂等性，需要单独处理
//				jointlyNode = parent.get(Node.NO);
				jointlyNode = parent;
			}
		} else if (FlowUtil.STYLE_PARALLEL.equals(pStyle)
				&& FlowUtil.STATE_WAITING.equals(parent.get(Work.STATE))) {

			// 父节点为“并行”节点时：考虑可能并发，需要实现幂等性处理
//			if (FlowUtil.STATE_WAITING.equals(parent.get(Work.STATE))) {
				// 当前父节点下，需忽略（跳过）未处理的节点
				addSkipNode(parent.get(Node.NO));
				// “并行节点”时，系统自动完成
				pickNextNode(parent);
//			}
			// 注意：其他并发节点完成后，只完成当前节点即可，无需再处理父节点
		}

	}

	/**
	 * 判断会签“父节点”下的每个节点是否都审批完毕
	 */
	private boolean jointlyApproved(Map<String, String> parent, Map<String, String> theNode) {
//		List<Map<String, String>> nodes = getChildNode(parent);
		// 遍历同父“兄弟”节点，排除当前节点编号
		String parentNo = parent.get(Node.NO);
		String nodeNo = theNode.get(Node.NO);
//		for (Map<String, String> node : nodes) {
		for (Map<String, String> node : flowNodes) {
			String state = node.get(Work.STATE);
			// 未批完的同父“兄弟”节点（排除当前节点）
			if (node.get(Node.PARENT).equals(parentNo) && !node.get(Node.NO).equals(nodeNo)
					&& (FlowUtil.STATE_READY.equals(state) || FlowUtil.STATE_WAITING.equals(state)))
				// 只要一个“兄弟”节点未完成，则未完成
				return false;
		}
		// 全部审批完（含当前节点）
		return true;
	}

	/**
	 * 判断当前会签“节点”下的每个审批人是否都审批完毕
	 */
	private boolean jointlyApproved(Map<String, String> theNode) {
//		List<Map<String, String>> nodes = getChildNode(parent);
		// 遍历“当前”节点下的审批人
		String nodeNo = theNode.get(Node.NO);
		for (Map<String, String> node : flowNodes) {
			String state = node.get(Work.STATE);
			// 当前节点下未批完的其他会签审批人
			if (node.get(Node.NO).equals(nodeNo)
					&& (FlowUtil.STATE_READY.equals(state) || FlowUtil.STATE_WAITING.equals(state)))
				// 只要一个“兄弟”节点未完成，则未完成
				return false;
		}
		// 全部审批完（含当前节点）
		return true;
	}

	/**
	 * 根据当前节点，找出下级子节点
	 * <p>
	 * 当前节点全部信息：
	 * nodeComp,nodeNo,nodeFlowNo,nodeName,nodeParent,nodeStyle,nodeTier
	 * ,nodeType,nodeStep,nodeRules,nodeEvent,taskComp,taskId,taskFlowNo
	 * ,workComp,workTaskId,workNodeNo,workState,workComment
	 * ,workESign,workUpdate,workIdx..
	 */
	private List<Map<String, String>> getChildNode(Map<String, String> parentNode) {
		// 根据当前节点，从工作流程中选出子节点
		List<Map<String, String>> list = new ArrayList<>();
		// 这里需判断空值“”
		String parentNo = parentNode.get(Node.NO);
		for (Map<String, String> node : flowNodes) {
			if (node.get(Node.PARENT).equals(parentNo))
				list.add(node);
		}
		return list;
	}

	/**
	 * 根据当前节点，找出上级父节点
	 */
	private Map<String, String> getParentNode(Map<String, String> theNode) {
		// 从工作节点列表定位父节点，这里不需要判断空值
//		String nodeNo = theNode.get(Node.PARENT);
//		for (Map<String, String> node : flowNodes) {
//			if (node.get(Node.NO).equals(nodeNo))
//				return node;
//		}
//		return null;
		return utils.findIn(flowNodes, Node.NO, theNode.get(Node.PARENT));
	}

	/**
	 * 找出当前审批节点的关联（追问或前加签）节点
	 */
	private Map<String, String> getRelatedNode(Map<String, String> theNode) {
		String trace = theNode.get(Work.TRACE);
		if (trace != null) {
			Map<String, String> map = utils.json2Map(trace);
			if (FlowUtil.OP_PROBE.equals(map.get(Const.TYPE))) {
				// 再跳转回去：{ "type": "3", "empId": "QiTdKv", nodeNo: "0101" }
				for (Map<String, String> node : flowNodes) {
					if (map.get(Node.NO).equals(node.get(Node.NO))
							&& map.get(Emp.ID).equals(node.get(Work.EMPID))) {
						return node;
					}
				}
//				return utils.findIn(flowNodes, Node.NO, map.get(Node.NO));
			}
		}
		return null;
	}

	/**
	 * 设置当前选择节点的审批状态，并缓存当前审批节点
	 */
	public void addWorkNode(Map<String, String> theNode, String workState) {
		// 更新当前节点的审批状态
		theNode.put(Work.STATE, workState);
		// 暂时缓存当前审批节点
		workNodes.add(theNode);
	}

	/**
	 * 更新流程表的“待审批”或“就绪”的中途节点taskNodeNo<br>
	 * 相关参数：{taskFlowNo,taskDept,taskTitle,taskFormData,..candidates}
	 */
	public void setWaitingNode(Map<String, Object> params) {
		/**
		 * 当前节点全部信息： 数据源：getFlowNodes
		 */
		Map<String, String> theNode = workNodes.get(0);
		for (Map<String, String> node : workNodes) {
			// 遍历每个“待审批”节点状态，要么“就绪”=“0”，要么“等待中”=“1”
			String state = node.get(Work.STATE);
			// 找出第一个就绪的待审批节点
			// 为兼容发起人第一次发起时去除自身节点，改为：找出最后一个就绪的待审批节点，
			if (FlowUtil.STATE_READY.equals(state)
					|| FlowUtil.STATE_WAITING.equals(state)) {
				theNode = node;
				// break;
			}
		}
		// 如果状态已经完成
		if (isSucceed)
			params.put(Task.STATE, FlowUtil.STATE_COMPLETE);
		// 更新流程上“待审批”任务节点taskNodeNo
		params.put(Task.NODENO, theNode.get(Node.NO));
	}

	/** 构造函数 */
	public FlowEngine(Map<String, Object> formData,
			List<Map<String, String>> flowNodes) {
		this.formData = formData;
		this.flowNodes = flowNodes;
	}

	/** 遇到死胡同了 */
	public void isLostWay() {
		// 如果走到“死胡同”迷路了，说明该“流程”有问题，需要维护完善流程
		throw new FailedException(FlowUtil.IS_LOST_WAY); // 迷路了
	}

	/** 已完成全部流程 */
	public boolean isSucceed() {
		return isSucceed;
	}

	/**
	 * 需忽略（跳过）未处理的节点
	 */
	public boolean addSkipNode(String nodeNo) {
		return skipNodes.add(nodeNo);
	}

	/** 需要跳过（忽略）的节点集合 */
	public Set<String> getSkipNodes() {
		return skipNodes;
	}

	/** 当前工作流的全部节点 */
	public List<Map<String, String>> getFlowNodes() {
		return flowNodes;
	}

	/** 当前审批节点一览表 */
	public List<Map<String, String>> getWorkNodes() {
		return workNodes;
	}

	/** 审批候选人列表 */
	public List<Map<String, Object>> getCandidates() {
		return candidates;
	}

	/** 需要重复检查的控制会签的“父”节点 */
	public Map<String, String> getJointlyNode() {
		return jointlyNode;
	}

	/** 当前用户语言 */
	public Object getUserLang() {
		return userLang;
	}

	/** 设置用户语言 */
	public void setUserLang(Object userLang) {
		this.userLang = userLang;
	}

}
