package cn.ps1.soar.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;

import javax.servlet.http.HttpServletRequest;

import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.ps1.aolai.entity.Dept;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.service.UtilsService;
import cn.ps1.soar.dao.FlowDao;
import cn.ps1.soar.engine.FlowEngine;
import cn.ps1.soar.entity.*;
import cn.ps1.soar.utils.FlowUtil;

/**
 * 业务流程的控制节点处理
 * 
 * @author Aolai
 * @version 1.0 $Date: 2024.01.20
 * @since openjdk-1.8
 */
@Service
public class EventEmitter {

	private static Logger LOG = LoggerFactory.getLogger(EventEmitter.class);

	@Autowired
	private UtilsService utils;

	@Autowired
	private BizService bizSvc;
	@Autowired
	private FlowService flowSvc;
	@Autowired
	private OrgnService orgnSvc;
	@Autowired
	private TaskService taskSvc;

	@Autowired
	private FlowDao flowDao;

	// 业务流程驱动引擎
	private FlowEngine flowEngine;

	private static final String EMP_INFO = "empInfo";
	private static final String CANDIDATES = "candidates";

	/**
	 * 暂存一个初始新任务为“草稿”，“待发起”的审批流程
	 */
	public Map<String, Object> saveTask(HttpServletRequest req) {
		// params={taskFlowNo,taskEmpId,taskDept,taskMode,taskFormData[,taskId,candidates]}
	
		// 补充增加暂时“秘书”参数empSecty
		Map<String, Object> params = taskSvc.newTaskParams(req);
		// 获取“发起人”（非操作用户）empId、empUid及所在部门、岗位等
		Map<String, String> sponsor = getSponsorInfo(params);
	
		// 发起“新流程”的表单信息格式转换
		setJsonData(params, Task.FORMDATA);
		// 如果携带了“taskId”，则继续沿用已有的任务编号，没有就新建一个
		checkReadyTask(params);
	
		// 保存为“草稿”，原“草稿”、“撤回”、“驳回”状态，此处仅改变“状态”
		// 又补充了：taskState（处理中）
		// 补充信息：taskEmpId、taskEmpName、taskOpUid、taskSecty
		taskSvc.addNewTask(params, sponsor, FlowUtil.STATE_READY);
		
		// 提交成功后，返回任务编号
		return utils.success(params.get(Task.ID));
	}

	/**
	 * 当用户新发起“新流程”时，根据流程编号选出第一个审批节点的“候选人”
	 * <p>
	 * 如有需要前端确认的多个候选人，返回前端确认处理后再“提交”。
	 * <p>
	 * 如果没有需要确认的多个候选人，则直接提交处理，并返回结果。
	 * <p>
	 * 注意：选中“审核人”后再提交时，必须携带 candidates参数，否则无效
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object newWorkTask(HttpServletRequest req) {
		// params={taskFlowNo,taskEmpId,taskDept,taskMode,taskFormData[,taskId,candidates]}

		// 补充增加暂时“秘书”参数empSecty
		Map<String, Object> params = taskSvc.newTaskParams(req);
		// 获取“发起人”（非操作用户）empId、empUid及所在部门、岗位等
		Map<String, String> sponsor = getSponsorInfo(params);

		// 发起“新流程”的表单信息格式转换
		// 当携带“taskId”时，taskMode在前面newTaskParams(req)中已删除
		if (params.containsKey(Task.MODE))
			setJsonData(params, Task.MODE);

		Map<String, Object> formData = setJsonData(params, Task.FORMDATA);
		// 表单项中追加流程“发起人”的员工及部门等信息，以备用
		formData.put(EMP_INFO, sponsor);

		// 当前用户（发起人）提交一个“新流程”
		initFlowEngine(params, formData);
		return startTask(params, sponsor);
	}

	/**
	 * 数据格式转换
	 */
	private Map<String, Object> setJsonData(Map<String, Object> params,
			String key) {
		// 发起“新流程”的表单信息
		Object obj = params.get(key);
		// 如果前台传过来是字符串，则不处理，如果是对象，则转成字符串
		if (obj instanceof String)
			obj = utils.json2Map((String) obj);
		else
			// 转成字符串，保存到临时表单项中
			params.put(key, utils.obj2Str(obj));

		return utils.obj2Map(obj);
	}

	/**
	 * 获取发起人信息
	 */
	private <T> Map<String, String> getSponsorInfo(Map<String, T> params) {
		Map<String, Object> where = new HashMap<>();
		// 发起、审批时，必须有“taskEmpId”参数
		if (params.containsKey(Task.EMPID))
			where.put(Orgn.EMPID, params.get(Task.EMPID)); 
		else
			throw new FailedException(); // 无效参数

		// 新发起流程时，默认查询当前用户或“秘书”
		if (params.containsKey(Emp.SECTY)) {
			// 审批时，不关心当前用户或秘书（注意不能用taskSecty）
			where.put(Emp.SECTY, params.get(Emp.SECTY));
		}
		// 国际化支持
		where.put(Const.I18N, params.get(Const.I18N));

		// 锁定发起任务的公司及部门
		where.put(Orgn.COMP, params.get(Task.COMP));
		where.put(Orgn.DEPT, params.get(Task.DEPT));

		// 返回发起人信息：empId、empUid等
		List<Map<String, String>> empList = orgnSvc.getEmpBy(where);
		if (empList.size() == 0)
			throw new FailedException(ConfUtil.DENY_ACCESS);
		return empList.get(0);
	}

	/**
	 * 流程引擎初始化
	 */
	private void initFlowEngine(Map<String, Object> params,
			Map<String, Object> formData) {
		// 1.初始化工作流，当前工作流程设置的所有“待审批”、“审批中”节点
		// 这里如果携带了taskId，则绑定taskId的
		flowEngine = new FlowEngine(formData, flowSvc.getFlowNodes(params));
		flowEngine.setUserLang(params.get(Const.I18N));
	}

	/**
	 * 发起“新流程”时，根据流程编号（或任务编号）的获取所有节点，进入工作审批流程
	 * <p>
	 * params={taskFlowNo,taskDept,taskMode,taskFormData[,taskId,candidates]}
	 */
	private Map<String, Object> startTask(Map<String, Object> params,
			Map<String, String> sponsor) {

		// 2.通过引擎自动选出第一个有效的审批节点及“候选人”
		// 如果只有一个候选人时，直接设为节点审批人
		// 设置节点上的候选人数据：workEmpId,workEmpName,workAgent,workOpUid
		flowEngine.findFirstNode();

		// 3.选出当前第一个有效节点有多个“候选人”时，返给前台进行交互确认
		Map<String, Object> result = getCandidates(params);
		if (result != null)
			return result;

		// 如果携带了“taskId”，则继续沿用已有的任务编号，没有就新建一个
		if (checkReadyTask(params)) {
			// 发起新流程，如果包含taskId，说明是驳回后再发起，必须更新idx
			incrApproveIdx(params);
		}

		// 4.保存“审批中”的工作节点，并更新节点状态
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 更新待审批节点（数据源参考：getFlowNodes）
		taskSvc.saveWorkNodes(params, workNodes);

		// 5.在流程中暂存“待审批”或“就绪”的中途节点
		// 补充信息：taskNodeNo、这里不可能完成（taskState）
		flowEngine.setWaitingNode(params);

		// 6.增加一条新“草稿”，原“草稿”、“撤回”、“驳回”状态，此处仅改变“状态”
		// 又补充了：taskState（处理中）
		// 补充信息：taskEmpId、taskEmpName、taskOpUid、taskSecty
		taskSvc.addNewTask(params, sponsor, FlowUtil.STATE_WAITING);

		// 7.发送通知消息
		flowSvc.notifyAll(workNodes);

		// 流程处理后，执行该类别的“后续操作”
		// 此处兼容发起人和审批人为一个节点，直接完成场景
		// 如果审批人和发起人是同一个人不自动跳过，此处isSuccess不为true，所以不影响
		if (flowEngine.isSucceed()) {
			// 流程发起后审批人如果是自己并且已自动完成，则自动审批通过
			params.put(Task.STATE, FlowUtil.STATE_COMPLETE);
			flowSvc.followProcess(params);
		}

		// 提交成功后，返回任务编号
		return utils.success(params.get(Task.ID));
	}

	/**
	 * 找出候选人列表
	 */
	private Map<String, Object> getCandidates(Map<String, Object> params) {
		// 按步骤处理：
		// 仅candidates.size() > 0时，才可能是多个候选人
		// 无候选人时前面已返回错误提示，仅一个候选人时跳过
		List<Map<String, Object>> candidates = flowEngine.getCandidates();
		// candidates.size() == 0，系统自动推送唯一候选人
		// candidates.size() > 0，多个或一个节点多个候选人
		if (candidates.size() > 0) {
			// 判断是否携带了candidates参数
			// 如果未携带参数，则返回当前节点对应“候选人”列表
			if (!params.containsKey(CANDIDATES))
				return utils.success(candidates);

			// 携带参数表示前台已确认，并已“选中”当前节点的其中一位“候选人”
			// 新设候选人数据：workEmpId,workEmpName,workAgent,workOpUid
			if (!pickedCandidate(candidates, params.get(CANDIDATES)))
				throw new FailedException(); // 无效参数
		}
		return null;
	}

	/**
	 * 设置提交已“选中”的审批节点的审批“候选人”
	 * 
	 * @param candidates 备选候选人
	 * @param approvers 前台选中的人选
	 * @return 是否能对应匹配上
	 */
	private boolean pickedCandidate(List<Map<String, Object>> candidates,
			Object approvers) {
		// 每个审批节点的全部“候选人”映射
		Map<String, List<Map<String, String>>> candidatesMap = candidatesOfNode(candidates);
		// 已提交“选中”的审批“候选人”： [{workNodeNo:"010103",empId:"wixsi4s8"},..]
		List<Map<String, String>> list = utils.obj2List(approvers);

		// 前台选中的人选要与后台候选节点保持一致，这里多节点主要考虑是并行节点
		if (candidates.size() != list.size())
			return false;

		// 遍历当前缓存的“审批中”的节点，匹配成功的审批人计数
		int approverCount = 0;
		for (Map<String, String> node : flowEngine.getWorkNodes()) {
			// 忽略非审批节点
			if (!FlowUtil.STYLE_NORMAL.equals(node.get(Node.STYLE))) 
				continue;

			// 前台选中的“审批人”节点，携带了“nodeNo”参数
			String nodeNo = node.get(Node.NO);

			// 如果提交的“节点”未在“待审批”节点列表中，则跳过
			Map<String, String> approver = utils.findIn(list, Node.NO, nodeNo);
			if (approver == null)
				continue;

			// 逐个按“节点”与“候选人”列表映射中匹配，必须匹配上才行
			List<Map<String, String>> empList = candidatesMap.get(nodeNo);
			for (Map<String, String> emp : empList) {
				// 如果提交推送的“候选人”能匹配到审批节点的“候选人”
				if (emp.get(Emp.ID).equals(approver.get(Emp.ID))) { 
					/**
					 * 新设候选人：workEmpId,workEmpName,workAgent,workOpUid
					 */
					flowEngine.setNodeApprover(node, emp);

					// 只要能匹配到了一个节点候选人即可
					approverCount++;
					break;
				}
			}
		}
		// 只有数量一致才可以，这里多节点主要考虑是并行节点
		return candidates.size() == approverCount;
	}

	/**
	 * 校验发起的任务（taskId）是否重复越权操作了其他taskId
	 */
	private boolean checkReadyTask(Map<String, Object> params) {
		// 未携带taskId，则返回一个新taskId
		if (params.get(Task.ID) == null) {
			params.put(Task.ID, Digest.uuid8() + params.get(Task.EMPID));
			return false;
		} else {
			// 如果前台携带了taskId，根据taskId获取工作流“草稿”
			Map<String, String> taskInfo = taskSvc.getTaskInfo(params);
//			if (utils.isFailed(taskInfo)) {
//				throw new FailedException();
//			}

			// 若草稿中有“秘书”，则秘书可以再发起（每人只能设一个秘书）
			// 注意：这里params中的Emp.SECTY为当前登录用户userId
			Object userId = params.get(Emp.SECTY);
			if (!userId.equals(taskInfo.get(Task.OPUID))
					&& !userId.equals(taskInfo.get(Task.SECTY))) {
				throw new FailedException();
			}

			// 如果不是“草稿”、“撤回”、“驳回”状态（0\5\4）时
			String taskState = taskInfo.get(Task.STATE);
			if (!FlowUtil.STATE_READY.equals(taskState)
					&& !FlowUtil.STATE_CANCEL.equals(taskState)
					&& !FlowUtil.STATE_REJECT.equals(taskState)) {
				// 其他状态的任务不允许有重新提交的情况：1\2\3\9
				throw new FailedException();
			}

			return true;
		}
	}

	/**
	 * 某个审批节点上的候选人
	 */
	private Map<String, List<Map<String, String>>> candidatesOfNode(
			List<Map<String, Object>> candidates) {
		Map<String, String> nodeMap;
		Map<String, List<Map<String, String>>> candidatesMap = new HashMap<>();
		List<Map<String, String>> empList;
		for (Map<String, Object> candidate : candidates) {
			nodeMap = utils.obj2Map(candidate.get(FlowUtil.MORM_NODE)); 
			// 这里candidates=empList
			empList = utils.obj2List(candidate.get(CANDIDATES));
			candidatesMap.put(nodeMap.get(Node.NO), empList);
		}
		return candidatesMap;
	}

	/**
	 * 选择满足规则的当前节点的审批候选人，根据条件选取当前node的审批候选人
	 * <p>
	 * 当前节点全部信息：
	 * nodeComp,nodeNo,nodeFlowNo,nodeName,nodeParent,nodeStyle,nodeTier
	 * ,nodeType,nodeStep,nodeRules,nodeEvent,taskComp,taskId
	 * ,taskFlowNo,workComp,workTaskId,workNodeNo,workState,workComment
	 * ,workESign,workUpdate,workIdx..
	 */
	public List<Map<String, String>> pickCandidates(Map<String, String> node,
			Map<String, Object> formData, List<Map<String, String>> flowNodes) {
		List<Map<String, String>> list = new ArrayList<>();

		// 根据empInfo获取发起人及部门信息：
		// deptId,deptName,deptType,deptTier,deptCost
		Map<String, String> sponsor = utils.obj2Map(formData.get(EMP_INFO));
		String deptId = sponsor.get(Dept.ID);

		// 当前审批节点的岗位规则列表，多条审批岗位规则如下：
		// [{Node.JOB: "eAbpfsBK", Node.SCOPE: {"all": "1"}},..]
		List<Map<String, Object>> rules;
		rules = utils.json2List(node.get(Node.RULES));// 规则一览表

		// 匹配规则一览表，一个员工可匹配多个岗位
		for (Map<String, Object> rule : rules) {
			Map<String, Object> where = new HashMap<>();
			where.put(Const.I18N, flowEngine.getUserLang());
			/**
			 * 只能是（nodeJob、employee）二选一<br>
			 * A.仅配置一个“审批人员”的情况
			 */
			if (rule.containsKey(FlowUtil.APPROVER)) { 
				// 二选一：选定一个“审批人”
				// 这里需要“去重”处理：列表中不存时才处理
				String empId = (String) rule.get(FlowUtil.APPROVER);
				if (utils.findIn(list, Emp.ID, empId) == null) {
					// 如果直接绑定某位员工时
					where.put(Emp.COMP, node.get(Node.COMP));
					where.put(Emp.ID, rule.get(FlowUtil.APPROVER));
					// 审批人携带了“主要任职部门”信息
					list.addAll(orgnSvc.getEmployee(where));
				}
				continue;
			}
			/**
			 * 只能是（nodeJob、employee）二选一<br>
			 * B.如果配置了“审批岗位”，先根据“岗位”获取候选人一览
			 */
			where.put(Orgn.COMP, node.get(Node.COMP));
			// 注意这里的传递参数为：Orgn.JOB，用作去匹配orgnJobs
			where.put(Orgn.JOB, rule.get(Node.JOB));

			// 查询该岗位的员工一览（默认按部门倒序）
			List<Map<String, String>> empList = orgnSvc.getEmpBy(where);
			if (empList.size() == 0)
				continue;
			LOG.debug("getEmpBy...{}", empList.get(0));

			/**
			 * 若要选出“候选人”，还需要匹配“节点规则”相关配置
			 */
			Map<String, String> scope = utils.obj2Map(rule.get(Node.SCOPE));
			// 虽然用了for循环，实际key只有一个：{"all": "1"}、{deptId: "001005"}
			for (String key : scope.keySet()) {
				// 注意员工有重复分布在多个部门
				for (Map<String, String> emp : empList) {
					// “候选人”所在部门
					String dept = emp.get(Dept.ID);

					// 1.“发起人”部门要在“候选人”部门管辖内，否则无审批权
					// 候选人非上级部门，如：sponsorDeptId=“001005003”，emp.deptId=“001004”
					if (Const.ALL.equals(key) && !deptId.startsWith(dept))
						continue; // 不匹配则忽略

					// 2.“候选人”要匹配“节点规则”才可以审批
					// 这里key可能是发起部门：“默认(all)”，或其上级主管部门的岗位：
					// 限定部门等级(deptTier)、指定部门(deptId)、部门类型(deptType)、成本中心(deptCost)等

					if (Const.ALL.equals(key)
							|| matchRules(scope, key, emp, deptId)) {
						/**
						 * 增加一位能匹配的角色候选人
						 */
						addCandidate(list, emp, rule.get(Node.JOB));
					}

				}
			}
		}
		// 返回候选人列表（含员工的岗位信息）
		LOG.debug("Candidates...{}", list);
		return orgnSvc.setJobsName(list);
	}

	/**
	 * 匹配规则
	 */
	private boolean matchRules(Map<String, String> scope, String key,
			Map<String, String> emp, String deptId) {
		// 这里关闭对deptTier的支持
		// 已弃用deptTier属性，前端支持deptLevel传参
		String val = String.valueOf(scope.get(key));
		if (Dept.LEVEL.equals(key)) {// || "deptTier".equals(key)) {
			// "deptLevel"
			int tier = Integer.parseInt(val);
			return emp.get(Dept.ID).length() / FlowUtil.leafW() == tier;
		} else if (Dept.TYPE.equals(key)) {
			// "deptType"
			return Const.STR_0.equals(val) || emp.get(key).equals(val);
		} else if ("sameDept".equals(key)) {
			return deptId.equals(emp.get(Dept.ID));
		} else if (scope.containsKey(key) && val.equals(emp.get(key))) {
			return true;
		}
		return false;
	}

	/**
	 * 增加一位能匹配的角色候选人
	 */
	private void addCandidate(List<Map<String, String>> list,
			Map<String, String> emp, Object job) {
		// 能匹配上的角色
		Map<String, String> jobs = new HashMap<>();
		// 这里也需要“去重”处理
		Map<String, String> candidate = utils.findIn(list, Emp.ID,
				emp.get(Emp.ID));
				
		if (candidate == null) {
			// 二选一后用nodeJob去匹配orgnJobs
			list.add(emp);
		} else {
			// 追加同一人多次可匹配的岗位
			emp = candidate;
			jobs = utils.json2Map(emp.get(Orgn.JOBS));
		}
		jobs.put((String) job, Const.STR_1);
		emp.put(Orgn.JOBS, utils.obj2Str(jobs));
		LOG.debug("pickCandidates...{}", emp);
	}

	/**
	 * 条件驱动，根据条件返回的结果再继续处理
	 */
	public int getActionIdx(Map<String, String> node, Map<String, Object> form) {
//		StringBuilder sb = new StringBuilder();
		// {"event": ["{formMoney} >= 10000", "{formMoney} <= 10"], "joint": "OR"}
		Map<String, Object> rules = utils.json2Map(node.get(Node.RULES));
		StringJoiner jt = new StringJoiner(" " + rules.get("joint") + " ");
		List<String> event = utils.obj2List(rules.get("event"));
		// ["{formMoney} >= 10000", "{formMoney} <= 10"]
		for (String ev : event) {
			String[] arr = ev.split("\\{ *");
			for (String str : arr) {
				String[] key = str.split(" *\\}");
//				if ("".equals(str)) // 首个字符一般为空
				if (key.length == 1 || !isValidExpr(key[1])) // 首个字符一般为空
					continue;

				// 注意这里form 属性的值需要防止SQL注入，数字转为字符：formMoney="20000"
				// String val = String.valueOf(form.get(key[0]));

				// 改为form表单支持多层嵌套的对象：formMoney
				String val = getFormValue(form, key[0]);
		
				// 表单的值需要能够匹配汉字，空字符串也认为是正常的表单数据，需要校验
				if ("".equals(val) || utils.isMatch(val, "\\w+")
						|| utils.isMatch(val, "[%0-9a-zA-Z\\u4e00-\\u9fa5]+")) {
//					if (sb.length() > 0)
//						sb.append(" " + rules.get("joint") + " ");
					// 此处需要增加单引号，如不增加字符串或者汉字会报错
//					sb.append("'" + val + "'" + key[1]);
					jt.add("'" + val + "'" + key[1]);
				}

			}

		}
		LOG.debug("> getActionIdx...{}", event);
		try {
			// 这里执行自定义SQL，必须增加try-catch捕获异常
//			return flowDao.getActionIdx(sb.toString());
			return flowDao.getActionIdx(jt.toString());
		} catch (Exception e) {
			LOG.warn("> getActionIdx...{}", event);
			return -1;
		}
	}

	/**
	 * 获取表单的属性值：form表单支持多层嵌套的对象<br>
	 * 例如：testForm.topTitle.userAge
	 */
	private String getFormValue(Object obj, String keyStr) {
		String[] keys = keyStr.split("\\.");
		for (String key : keys) {
			LOG.debug("> getActionIdx...{}", obj);
			if (obj instanceof Map) {
				Map<String, Object> map = utils.obj2Map(obj);
				// 例如：key=topTitle时，还有一层key=userAge
				obj = map.get(key);
			} else {
				obj = null;
			}
/*			if (obj == null) {
				obj = "";
//				LOG.warn("> getActionIdx...{}", key);
//				flowEngine.isLostWay();
			}*/
		}
		return obj == null ? "" : String.valueOf(obj);
	}

	/**
	 * 正则校验表达式
	 */
	private boolean isValidExpr(String expr) {
		if (expr.length() > 0) {
			// 从表达式拆分识别出关键字，如：“ in (”
			String[] arr = expr.split(" +"); // 以空格“”分割
			for (String str : arr) {
				// 必须能匹配规定范围内的表达式，或满足JSON类型操作的表达式格式："->>'$.inspur.eyun'"
				if (str.length() == 0
						|| utils.findIn(ConfUtil.SQL_EXPR, str.toLowerCase())
						|| utils.isNumeric(str)
						|| utils.isMatch(str, "['%]*\\w+[%']*")
						|| utils.isMatch(str, "'[%0-9a-zA-Z\\u4e00-\\u9fa5]+'")) // 匹配单引号，开头结尾，中间包含百分号、数字以及汉字
					continue;
				// 若有未匹配的表达式，则返回无效
				LOG.error("invalid expr...{}", str);
				return false;
			}
		}
		return true;
	}

	/**
	 * 当用户完成当前节点审批时，根据流程编号列出下一个审批候选人
	 * <p>
	 * 如有需要前端确认的候选人，返回前端确认处理后再“提交”。
	 * <p>
	 * 如果没有需要确认的候选人，则直接提交处理，并返回结果。
	 * <p>
	 * 当用户确认选中“审核人”后，必须携带candidates参数，否则无效 <br>
	 * 注意：选中“审核人”后再提交时，必须携带 candidates参数，否则无效 <br>
	 * params={taskId,workNodeNo,workComment[,candidates]}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object approveNode(HttpServletRequest req) {
		Map<String, Object> params = getWorkParams(req);
		// 获取节点后又补充params参数：workEmpId、taskFlowNo
		Map<String, String> task = taskSvc.getReadyTask(params);
		// 支持多语言i18n
		// 当前用户有审批权限。查询“发起人”的条件：taskEmpId或taskOpUid
		task.put(Const.I18N, (String) params.get(Const.I18N));
		// 返回发起人信息：empId、empUid等
		Map<String, String> sponsor = getSponsorInfo(task);

		// 发起人启动“流程”时填写的表单数据
		Map<String, Object> formData = utils.json2Map(task.get(Task.FORMDATA));
		// 表单项中追加流程“发起人”的员工及部门等信息，以备用
		formData.put(EMP_INFO, sponsor);

		// 当前用户提交“通过”状态
		initFlowEngine(params, formData);
		return approveNode(params, task);
	}

	/**
	 * 完成当前节点审批，根据流程编号列出审批候选人
	 * params={taskId,taskFlowNo,workNodeNo,workComment[,candidates]}
	 */
	private Object approveNode(Map<String, Object> params,
			Map<String, String> task) {

		// 2.从当前节点递归遍历全部节点及“候选人”，更新当前节点状态、并选出下一个工作节点
		flowEngine.findNextNode(params);

		// 3.选出当前第一个有效节点有多个“候选人”时，返给前台进行交互确认
		Map<String, Object> result = getCandidates(params);
		if (result != null)
			return result;

		// 4.保存“审批中”的工作节点，并更新节点状态、审批意见
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 更新待审批节点（数据源参考：getFlowNodes）
		taskSvc.saveWorkNodes(params, workNodes);

		// 5.在流程中暂存“待审批”或“就绪”的中途节点
		// 补充信息：taskNodeNo、若完成（taskState）
		flowEngine.setWaitingNode(params);
		// 5.1.更新任务信息及状态
		taskSvc.setTaskState(params);

		// 6.处理需要跳过（忽略）的其他并行节点（事务失败不影响）
		skipParallelNodes(params);

		// 7.会签节点“未满签”时，节点状态更新后需要二次校验
		// 当jointlyNode==null返回未完成false时跳过处理
		// 会签“父节点”jointlyNode!=null、存在“未通过”（或驳回）的节点属正常
		// 若不存在，即“会签”完成，则不正常需要“回滚”重新处理
		if (taskSvc.jointlyCompleted(flowEngine.getJointlyNode())) {
			// 并发冲突，需要终止事务重新处理
			throw new FailedException("isRollback");
			// 如果“未满签”时，有可能兄弟节点“已驳回”，无不良影响
		}

		// 8.发送通知消息
		flowSvc.notifyAll(workNodes);

		// 流程处理后，执行该类别的“后续操作”
		if (flowEngine.isSucceed()) {
			 task.put(Task.STATE, FlowUtil.STATE_COMPLETE);
			 flowSvc.followProcess(task);
		}

		return utils.success();
	}

	/**
	 * 处理需要跳过（忽略）的其他并行节点（事务失败不影响）
	 * <p>
	 * 跳过“父节点”下没有处理的并行子节点状态为‘3’
	 */
	private void skipParallelNodes(Map<String, Object> where) {
		Set<String> skipNodes = flowEngine.getSkipNodes();
		if (skipNodes.size() > 0) {
			LOG.debug("> skipParallelNodes...{}", skipNodes);
			Map<String, Object> cond = utils.sameId(where, Task.KEY);
			// 更新状态：就绪‘0’->>跳过‘3’
			cond.put("stateReady", FlowUtil.STATE_READY); // 原状态WORK_STATE
			cond.put("stateSkip", FlowUtil.STATE_SKIP); // 更新WORK_STATE
			// 跳过的节点
			cond.put("skipNodes", skipNodes);
			// BRK_APPRO JOIN BRK_NODE
			// 更新需要跳过（忽略）的其他并行节点
			flowDao.skipParallelNodes(flowSvc.workNodeTables(), cond);
		}
	}

	/**
	 * 公共：工作流任务的参数条件
	 */
	private Map<String, Object> getWorkParams(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		Map<String, String> user = utils.userSelf(req);
		params.put(Task.COMP, user.get(User.COMP));
		// 根据taskId、workNodeNo获取审批节点及表单信息
		// 这里userId可能是“代理人”
		params.put(Work.AGENT, user.get(User.ID));
		return params;
	}


	/**
	 * 在当前节点加签，相当于追加了一个审批人 <br>
	 * params={taskId,workNodeNo,workComment[,candidates]}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object addApprover(HttpServletRequest req) {
		Map<String, Object> params = getWorkParams(req);
		// 获取节点后又补充params参数：workEmpId、taskFlowNo
		Map<String, String> task = taskSvc.getReadyTask(params);

		// 当前用户提交“加签”用户，加签时不必关心表单内容。
		// 1.初始化流程引擎，这里也用到了：taskFlowNo
		initFlowEngine(params, null);
		// 2.处置当前审批就绪节点：加签、驳回
		handleNode(params, FlowUtil.STATE_COMPLETE);
		return addApprover(params, task);
	}

	/**
	 * 在当前节点加签，相当于追加了一个审批人
	 */
	private Object addApprover(Map<String, Object> params,
			Map<String, String> task) {
		// 3.保存“审批中”的工作节点，并更新节点状态、审批意见
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 理论上有一个满足此时条件的节点
		if (workNodes.size() == 0)
			throw new FailedException(FlowUtil.IS_LOST_WAY);

		// 4.先克隆一个相同节点，设为“就绪”待办
		Map<String, String> node = utils.obj2Map(workNodes.get(0));
		// 在克隆节点上设置一个审批人
		Map<String, Object> where = new HashMap<>();
		where.put(Const.I18N, flowEngine.getUserLang());
		where.put(Emp.COMP, node.get(Node.COMP));
		where.put(Emp.ID, params.get(FlowUtil.APPROVER));

		// 5.携带了加签的“审批人”
		List<Map<String, String>> empList = orgnSvc.getEmployee(where);
		// 理论上有一个满足此时条件的节点
		if (empList.size() == 0)
			throw new FailedException();

		// 6.先克隆一个相同节点，设为“就绪”待办
		flowEngine.setNodeApprover(node, empList.get(0));
		flowEngine.addWorkNode(node, FlowUtil.STATE_READY);

		// 7.更新审批节点状态、审批意见，并加签
		taskSvc.saveWorkNodes(params, workNodes);

		// 8.发送通知消息
		flowSvc.notifyAll(workNodes);

		return utils.success();
	}

	/**
	 * 当用户完成当前节点驳回
	 * params={taskId,workNodeNo,workComment[,candidates]}
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Object rejectNode(HttpServletRequest req) {
		Map<String, Object> params = getWorkParams(req);
		// 获取节点后又补充params参数：workEmpId、taskFlowNo
		Map<String, String> task = taskSvc.getReadyTask(params);

		// 当前用户提交“驳回”状态，取消时不必关心表单内容。
		// 1.初始化流程引擎，这里也用到了：taskFlowNo
		initFlowEngine(params, null);
		// 2.处置当前审批就绪节点
		handleNode(params, FlowUtil.STATE_REJECT);
		return rejectNode(params, task);
	}

	/**
	 * 处置当前审批就绪节点：<br>
	 * 同意(FlowUtil.STATE_COMPLETE)或驳回(FlowUtil.STATE_REJECT)
	 */
	private void handleNode(Map<String, Object> params, String workState) {
		// 2.当前节点驳回，并缓存当前“审批中”的节点
		Object workNodeNo = params.get(Work.NODENO);
		// 遍历节点，找到“就绪”节点
		for (Map<String, String> node : flowEngine.getFlowNodes()) {
			// 仅处理“就绪”节点
			if (!FlowUtil.STATE_READY.equals(node.get(Work.STATE)))
				continue;
			// 定位当前的“审批节点”，状态“就绪”且分配给当前用户审批的节点
			// 校验workEmpId以防止越权操作
			if (node.get(Node.NO).equals(workNodeNo)) {
				// && params.get(Work.EMPID).equals(node.get(Work.EMPID))) {

				// 非“委托代理人”驳回时，暂时清空“代理人”
				setAgentNull(params, node.get(Work.OPUID));

				// 处置当前“就绪”节点，更新状态
				flowEngine.addWorkNode(node, workState);//FlowUtil.STATE_REJECT);
			} else {
				flowEngine.addSkipNode(node.get(Node.PARENT));
			}
		}
	}

	/**
	 * 未使用代理审批：非“委托代理人”审批时，暂时清空“代理人”
	 */
	public void setAgentNull(Map<String, Object> params, String opUid) {
//		if (params.get(Work.AGENT).equals(node.get(Work.OPUID)))
		// 非“委托代理人”审批时，需清空“代理人”
		Object agent = params.get(Work.AGENT);
		if (agent != null && agent.equals(opUid)) // node.get(Work.OPUID)))
			params.put(Work.AGENT, "");
	}

	/**
	 * 当用户完成当前节点驳回
	 */
	private Object rejectNode(Map<String, Object> params,
			Map<String, String> task) {
		// 3.保存“审批中”的工作节点，并更新节点状态、驳回意见
		List<Map<String, String>> workNodes = flowEngine.getWorkNodes();
		// 更新待审批节点（数据来源请参考：getFlowNodes()）
		taskSvc.saveWorkNodes(params, workNodes);

		// 4.驳回后，把所有已审批过的节点workIdx++
		// 驳回后的可能再发起审批
		// 驳回后如果审批节点workidx++，会导致查不出驳回状态数据，所以此处先注释掉，改到驳回后再发起时处理。-- zhengdali
		// incrApproveIdx(params);

		// 5.在流程中暂存“待审批”或“就绪”的中途节点
		params.put(Task.STATE, FlowUtil.STATE_REJECT);
		// 5.1.更新任务信息及状态
		taskSvc.setTaskState(params);

		// 6.处理需要跳过（忽略）的并行节点（事务失败不影响）
		skipParallelNodes(params);

		// 7.发送通知消息
//		flowSvc.notifyAll(workNodes);

		// 流程拒绝后，执行该类别的“后续操作”
		task.put(Task.STATE, FlowUtil.STATE_REJECT);
		flowSvc.followProcess(task);

		return utils.success();
	}

	/**
	 * 针对审批节点与工作流节点<br>
	 * 自动递增审批节点表的审批序号（避免主键冲突：从大到小降序处理）
	 */
	int incrApproveIdx(Map<String, Object> params) {
		Map<String, Object> where = new HashMap<>();
		where.put(Work.COMP, params.get(Task.COMP));
		where.put(Work.TASKID, params.get(Task.ID));
		return flowDao.incrApproveIdx(flowSvc.workTable(), where);
	}

	/**
	 * 根据taskId撤销一个审批中的任务，取消“审批中”的审批流程
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> cancelTask(HttpServletRequest req) {
		Map<String, Object> params = getWorkParams(req);
		Map<String, String> result = taskSvc.cancelTask(params);
		// 撤销后，把所有已审批过的节点workIdx++
		// 撤销后如果审批节点workidx++，会导致查不出撤销状态数据，所以此处先注释掉，改到撤销后再发起时处理。-- zhengdali
//		if (utils.isSuccess(result))
//			incrApproveIdx(utils.sameId(params, Task.KEY));
		return result;
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 */
	public Map<String, Object> getBizFlows(HttpServletRequest req) {
		// 请求参数（orgnDept或orgnEmpId）、及当前用户信息
		Map<String, Object> params = bizSvc.getBizParams(req);
		// 返回当前用户可发起的“流程”（或携带bizNo查询）
		return utils.success(getBizFlows(params));
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 * <p>
	 * 返回当前发起人可用的业务流程 bizNo与flowNo的对应映射
	 */
	private Map<String, String> getBizFlows(Map<String, Object> where) {
		// 当前员工及所在部门、角色、职务（含部门类型、部门层级、成本中心等信息）
		// 参数需转换一下
		where.put(Orgn.COMP, where.get(Biz.COMP));
		// 支持多语言i18n
		List<Map<String, String>> empList = orgnSvc.getEmpBy(where);
		Map<String, Object> params = utils.sameNo(where, Biz.KEY);
		if (!utils.isEmpty(where.get(Biz.STATE)))
			params.put(Biz.STATE, where.get(Biz.STATE));

		// 获取当前用户可以发起的业务流程（或携带bizNo查询）
		return bizSvc.getBizFlows(params, empList);
	}

	/**
	 * 查询当前用户可发起的业务流程一览表
	 */
	public Map<String, Object> getBizModes(HttpServletRequest req) {
		// 请求参数（orgnDept、orgnEmpId、bizNo）、及当前用户信息
		Map<String, Object> params = bizSvc.getBizParams(req);

		// 用户可发起的流程：orgnDept、orgnEmpId、empUid（可能是秘书）
		Map<String, String> bizMap = getBizFlows(params);

		// 返回当前用户可发起的“业务”
		return utils.success(bizSvc.getBizModes(bizMap, params));
	}

}
