package cn.ps1.soar.service;

import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.service.AolaiService;
import cn.ps1.aolai.service.UtilsService;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.soar.entity.Define;
import cn.ps1.soar.entity.Form;
import cn.ps1.soar.entity.Mode;
import cn.ps1.soar.entity.Task;
import cn.ps1.soar.utils.FlowUtil;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 自定义表单的维护管理
 * 
 * @author Aolai
 * @version 1.0 $Date: 2023.06.21
 * @since openjdk-1.8
 */
@Service
public class CFormService {

	private static Logger LOG = LoggerFactory.getLogger(CFormService.class);

	@Autowired
	private AolaiService aolai;
	@Autowired
	private UtilsService utils;

	@Autowired
	private ZeroService zeroSvc;

	/**
	 * 保存自定义表单数据
	 */
	public Map<String, String> saveCustomForm(HttpServletRequest req) {
		List<Map<String, Object>> rows = getFormData(req);
		Map<String, String> user = this.utils.userSelf(req);
		String compId = user.get(User.COMP);
		String formNo = Digest.uuid8();
		// 保存表单数据
		return saveCustomForm(compId, formNo, rows);
	}

	/**
	 * 共通处理表单数据
	 */
	private List<Map<String, Object>> getFormData(HttpServletRequest req) {
		Map<String, Object> data = utils.jsonParams(req);
		List<Map<String, Object>> rows = utils.obj2List(data.get("list"));
		if (rows.size() == 0)
			throw new FailedException();
		return rows;
	}

	/**
	 * 保存自定义表单数据
	 */
	private Map<String, String> saveCustomForm(String compId, String formNo,
			List<Map<String, Object>> rows) {
		String[] keys = ConfUtil.getValid("customForm").split(ConfUtil.COMMA);
		int rowIdx = 0;
		for (Map<String, Object> row : rows) {
			row.put(Form.NO, formNo); // 先补充formNo数据
			if (!utils.availParams(row, keys))
				throw new FailedException();
			row.put(Form.COMP, compId);
			row.put(Form.IDX, rowIdx++);
		}
		Map<String, String> result;
		result = aolai.batchAdd(null, Form.TABLE, rows, null, false);
		if (utils.isSuccess(result))
			return utils.success(formNo);
		return result;
	}

	/**
	 * 判断form是否被task使用过
	 */
	private boolean isFormUsed(Object formNo, String compId) {
		Map<String, Object> taskWhere = new HashMap<>();
		taskWhere.put(utils.jsonExt(Task.MODE, Mode.FORMNO), formNo);
		taskWhere.put(Task.COMP, compId);
		return aolai.exists(Task.TABLE, taskWhere);
	}

	/**
	 * 更新form数据
	 * 
	 * @param rows 这里的form数据非空
	 * @param compId 公司租户
	 */
	private String updateForm(List<Map<String, Object>> rows,
			String compId) {
		Map<String, String> result;
		// 检查引用该表单的业务类型是否被流程占用过，只要占用即认为需要生成新的formNo
		String formNo = (String) rows.get(0).get(Form.NO);

		// task中已用的modeFormNo保持不变
		if (isFormUsed(formNo, compId)) {
			// 只更新被引用过的业务数据Mode
			Map<String, Object> cond = new HashMap<>();
			// 符合这个表单id的都更新成新的表单，包括继承的
			cond.put(Mode.FORMNO, formNo);
			cond.put(Mode.COMP, compId);

			// 被引用了则生成新的formNo
			formNo = Digest.uuid8();
			// 将新生成得formNo替换旧的formNo
			// 这里统一放到saveCustomForm中处理
//			for (Map<String, Object> item : rows) {
//				item.put(Form.NO, formNo); // 新的formNo
//			}
			// 更新新的formNo
			Map<String, Object> data = new HashMap<>();
			data.put(Mode.FORMNO, formNo);
			// 只更新业务数据Mode
			result = aolai.update(Mode.TABLE, data, cond);
		} else {
			// 不更新未被引用过的业务数据Mode
			// 只是正常修改一个表单：先根据formNo删除旧数据，再保存新数据
			Map<String, Object> cond = new HashMap<>();
			cond.put(Form.NO, formNo);
			cond.put(Form.COMP, compId);
			result = aolai.delete(Form.TABLE, cond);
		}
		if (utils.isFailed(result))
			throw new FailedException();
		return formNo;
	}

	/**
	 * 修改一个自定义表单
	 */
	@Transactional(rollbackFor = { Throwable.class })
	public Map<String, String> setCustomForm(HttpServletRequest req) {
		// 这里rows数据已校验非空
		List<Map<String, Object>> rows = getFormData(req);

		Map<String, String> user = this.utils.userSelf(req);
		String compId = user.get(User.COMP);

		// 首先检查表单是否需要生成新的formNo，
		// 如已被流程占用则直接生成新的formNo
		String formNo = updateForm(rows, compId);

		// 正常修改一个表单：需先根据formNo删除已有数据，再保存新数据
//		String formNo = (String) rows.get(0).get(Form.NO);
//		Map<String, Object> where = new HashMap<>();
//		where.put(Form.NO, formNo);
//		where.put(Form.COMP, compId);
//		aolai.delete(Form.TABLE, where);

		// 保存新数据
		return saveCustomForm(compId, formNo, rows);
	}

	/**
	 * 删除一个自定义表单：taskMode.modeFormNo
	 */
	public Map<String, String> delCustomForm(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		Map<String, String> user = this.utils.userSelf(req);
		String compId = user.get(User.COMP);
		String formNo = (String) params.get(Form.NO);

		Map<String, Object> where = new HashMap<>();
		where.put(Task.COMP, compId);
		// TASK_MODE->>'$.modeFormNo'
//		String[] keys = { "{", Task.MODE, "}->>'$.", Mode.FORMNO, "' =" };
//		where.put(utils.arr2Str(keys), formNo);
		where.put(utils.jsonExt(Task.MODE, Mode.FORMNO), formNo);
		if (this.aolai.exists(null, Task.TABLE, where))
			return this.utils.failed(ConfUtil.CANT_REMOVE);

		where = utils.sameNo(params, Form.KEY);
		return aolai.delete(null, Form.TABLE, where);
	}

	/**
	 * 获取一个自定义表单
	 */
	public Map<String, Object> getCustomForm(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		utils.setUserComp(req, params, Form.COMP);
		LOG.debug("-> getCustomForm...{}", params);
		// 默认排序
		Map<String, String> order = new HashMap<>();
		order.put(Form.IDX, "ASC");

		String[] args = { null, Form.TABLE, "getCustomForm", Const.STR_0 };
		return utils.success(this.aolai.findAll(params, order, args));
	}

	/**
	 * 获取自定义表单一览
	 */
	// formName,formId,formField,formTitle,formType,formComponent,formIdx
	public Map<String, Object> getCustomFormList() {
		Map<String, Object> where = new HashMap<>();
		where.put(Form.STATE, Const.STR_1);
		where.put(Form.IDX, 0); // 仅显示一行数据
		Map<String, String> order = new HashMap<>();
		order.put(Form.CREATE, "DESC");
		String[] args = { null, Form.TABLE, "getCustomFormList", Const.STR_0 };
		return utils.success(aolai.findAll(where, order, args));
	}

	/**
	 * 新增保存一个自定义表单部件
	 */
	public Map<String, String> saveFormPart(HttpServletRequest req) {
		Map<String, Object> data = utils.jsonParams(req);
		// addFormPart\addEnumInfo
		data = FlowUtil.putParams(data, "addFormPart", "addEnumInfo");
		utils.setUserComp(req, data, Define.COMP);
		return zeroSvc.addEnumInfo(data);
	}

	/**
	 * 获取一个自定义部件组
	 */
	public Object getFormPart(HttpServletRequest req) {
		Map<String, Object> where = utils.jsonParams(req);
		where = FlowUtil.putParams(where, "getFormPart", "getEnumType");
		utils.setUserComp(req, where, Define.COMP);
		Map<String, String> result = zeroSvc.getEnumInfo(where);
		if (result.containsKey(Const.STS))
			return result;
		result = FlowUtil.putParams(result, "addEnumInfo", "addFormPart");
		return utils.success(result);
	}

	/**
	 * 删除一个自定义部件组
	 */
	public Map<String, String> delFormPart(HttpServletRequest req) {
		Map<String, Object> where = utils.jsonParams(req);
		where = FlowUtil.putParams(where, "delFormPart", "delEnumInfo");
		utils.setUserComp(req, where, Define.COMP);
		return zeroSvc.delEnumInfo(where);
	}

}
