package cn.pengh.util;

import cn.pengh.exception.CustomException;
import com.ctc.wstx.api.WstxOutputProperties;
import com.ctc.wstx.stax.WstxOutputFactory;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.dataformat.xml.XmlFactory;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ClassUtils;

import javax.xml.namespace.NamespaceContext;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.Result;
import java.io.OutputStream;
import java.io.Writer;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 推荐jackson xml
 *
 * @author Created by pengh
 * @datetime 2023/9/15 09:35
 */
public final class XmlUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(XmlUtil.class);
    private static final String CLAZZ_JACKSON_XML = "com.fasterxml.jackson.dataformat.xml.XmlMapper";
    private static final String CLAZZ_XSTREAM_XML = "com.thoughtworks.xstream.XStream";

    private static final ReentrantLock REENTRANT_LOCK = new ReentrantLock();
    private static final int REENTRANT_LOCK_TIMEOUT_SECONDS = 3; //等3s


    public static String toXml(Object object) {
        return toXml(object, false);
    }

    public static String toXml(Object object, boolean cdata) {
        return toXml(object, cdata, false);
    }

    public static String toXml(Object object, boolean cdata, boolean isPretty) {
        try {
            return getXmlInstance().toXml(object, cdata, isPretty);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static <T> T fromXml(String str, Class<T> clazz) {
        try {
            return str == null ? null : getXmlInstance().fromXml(str, clazz);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    // 依赖spring-core的ClassUtils来判断
    private static boolean hasJsonClazz(String clazzName) {
        try {
            return ClassUtils.isPresent(clazzName, XmlUtil.class.getClassLoader());
        } catch (Exception e) {
            LOGGER.error(e.getMessage());
            return false;
        }
    }

    private static <T> XmlInternalFacade<T> getXmlInstance() {
        if (hasJsonClazz(CLAZZ_JACKSON_XML)) {
            return JacksonFacade.getInstance();
        } else if (hasJsonClazz(CLAZZ_XSTREAM_XML)) {
            return XStreamFacade.getInstance();
        }
        throw CustomException.create("无XML依赖适配器，如Jackson、XStream");
    }

    public interface XmlInternalFacade<T> {
        String toXml(Object object, boolean cdata, boolean isPretty) throws Throwable;

        <T> T fromXml(String str, Class<T> clazz) throws Throwable;
    }


    public static class JacksonFacade<T> implements XmlInternalFacade<T> {
        private JacksonFacade() {
            LOGGER.debug("init Jackson..");
        }

        private static final class LazyHolder {
            private static final JacksonFacade INSTANCE = new JacksonFacade<>();
        }

        public static JacksonFacade getInstance() {
            return JacksonFacade.LazyHolder.INSTANCE;
        }

        private transient ObjectMapper xmlMapper;
        private transient ObjectMapper xmlCdataMapper;
        private transient ObjectMapper xmlDeserializationMapper;
        private ObjectMapper getXmlDeserializationInstance() {
            if (xmlDeserializationMapper == null) {
                try {
                    if (!REENTRANT_LOCK.tryLock(REENTRANT_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS)) {
                        LOGGER.warn("init Jackson Deserialization XmlMapper LOCKING");
                    }
                    if (xmlDeserializationMapper != null) {
                        //LOGGER.debug("init Jackson XmlMapper After LOCKING");
                        return xmlDeserializationMapper;
                    }
                    LOGGER.debug("init Jackson Deserialization XmlMapper");
                    xmlDeserializationMapper = new XmlMapper();
                    xmlDeserializationMapper.configure(DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES, false);
                    xmlDeserializationMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                    return xmlDeserializationMapper;
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                } finally {
                    REENTRANT_LOCK.unlock();
                }
            }
            return xmlDeserializationMapper;
        }

        private ObjectMapper getXmlInstance() {
            if (xmlMapper == null) {
                try {
                    if (!REENTRANT_LOCK.tryLock(REENTRANT_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS)) {
                        LOGGER.warn("init Jackson XmlMapper LOCKING");
                        return null;
                    }
                    if (xmlMapper != null) {
                        //LOGGER.debug("init Jackson XmlMapper After LOCKING");
                        return xmlMapper;
                    }
                    LOGGER.debug("init Jackson XmlMapper");
                    XMLOutputFactory factory = new WstxOutputFactory();
                    factory.setProperty(WstxOutputProperties.P_OUTPUT_CDATA_AS_TEXT, true); //忽略@JacksonXmlCData
                    XmlFactory xf = XmlFactory.builder()
                            .xmlOutputFactory(factory) // note: in 2.12 and before "outputFactory()"
                            .build();
                    xmlMapper = new XmlMapper(xf);
                    xmlMapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
                    xmlMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
                    xmlMapper.enable(MapperFeature.USE_STD_BEAN_NAMING);
                    return xmlMapper;
                } catch (InterruptedException e) {
                    e.printStackTrace();
                } finally {
                    REENTRANT_LOCK.unlock();
                }

            }
            return xmlMapper;
        }

        private ObjectMapper getXmlCdataInstance() {
            if (xmlCdataMapper == null) {
                try {
                    if (!REENTRANT_LOCK.tryLock(REENTRANT_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS)) {
                        LOGGER.warn("init Jackson XmlMapper LOCKING");
                        return null;
                    }
                    if (xmlCdataMapper != null) {
                        //LOGGER.debug("init Jackson XmlMapper After LOCKING");
                        return xmlCdataMapper;
                    }

                    LOGGER.debug("init Jackson Cdata XmlMapper");
                    XMLOutputFactory factory = new CDataXmlOutputFactoryImpl();
                    factory.setProperty(WstxOutputProperties.P_OUTPUT_CDATA_AS_TEXT, false);
                    XmlFactory xf = XmlFactory.builder()
                            .xmlOutputFactory(factory) // note: in 2.12 and before "outputFactory()"
                            .build();

                    xmlCdataMapper = new XmlMapper(xf);
                    xmlCdataMapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
                    xmlCdataMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
                    xmlCdataMapper.enable(MapperFeature.USE_STD_BEAN_NAMING);
                    return xmlCdataMapper;
                } catch (InterruptedException e) {
                    e.printStackTrace();
                } finally {
                    REENTRANT_LOCK.unlock();
                }
            }
            return xmlCdataMapper;
        }


        //https://github.com/FasterXML/jackson-dataformat-xml/blob/2.16/src/test/java/com/fasterxml/jackson/dataformat/xml/ser/XmlPrettyPrinterTest.java
        @Override
        public String toXml(Object object, boolean cdata, boolean isPretty) throws Throwable {
            ObjectMapper om = cdata ? getXmlCdataInstance() : getXmlInstance();
            if (cdata) {
                isPretty = false; // 暂不支持cdata pretty
            }
            /*if (cdata) {
                LOGGER.debug("cdata {}",cdata);
                return om.writer()
                        .with(new CdataXmlPrettyPrinter())
                        //.with(ToXmlGenerator.Feature.WRITE_XML_DECLARATION)
                        .writeValueAsString(object);
            }*/
            return isPretty ? om.writerWithDefaultPrettyPrinter().writeValueAsString(object) : om.writeValueAsString(object);
        }

        /*private class CdataXmlPrettyPrinter extends DefaultXmlPrettyPrinter {

            @Override
            public void writeLeafElement(XMLStreamWriter2 sw,
                                         String nsURI, String localName, String text, boolean isCData)
                    throws XMLStreamException
            {
                if (!_objectIndenter.isInline()) {
                    _objectIndenter.writeIndentation(sw, _nesting);
                }
                sw.writeStartElement(nsURI, localName); LOGGER.debug("isCData {}",isCData);
                *//*if(isCData) {
                    sw.writeCData(text);
                } else {
                    sw.writeCharacters(text);
                }*//*
                sw.writeCData(text);
                sw.writeEndElement();
                _justHadStartElement = false;
            }

            @Override
            public void writeLeafElement(XMLStreamWriter2 sw,
                                         String nsURI, String localName,
                                         char[] buffer, int offset, int len, boolean isCData)
                    throws XMLStreamException
            {
                if (!_objectIndenter.isInline()) {
                    _objectIndenter.writeIndentation(sw, _nesting);
                }LOGGER.debug("isCData {}",isCData);
                sw.writeStartElement(nsURI, localName);
                *//*if(isCData) {
                    sw.writeCData(buffer, offset, len);
                } else {
                    sw.writeCharacters(buffer, offset, len);
                }*//*
                sw.writeCData(buffer, offset, len);
                sw.writeEndElement();
                _justHadStartElement = false;
            }
        }*/

        @Override
        public <T> T fromXml(String str, Class<T> clazz) throws Throwable {
            return getXmlDeserializationInstance().readValue(str, clazz);
        }


        //https://gist.github.com/jbcpollak/8312151
        //https://www.oomake.com/question/5417186
        private class CDataXmlOutputFactoryImpl extends XMLOutputFactory {

            XMLOutputFactory f = XMLOutputFactory.newInstance();

            public XMLEventWriter createXMLEventWriter(OutputStream out) throws XMLStreamException {
                return f.createXMLEventWriter(out);
            }

            public XMLEventWriter createXMLEventWriter(OutputStream out, String enc) throws XMLStreamException {
                return f.createXMLEventWriter(out, enc);
            }

            public XMLEventWriter createXMLEventWriter(Result result) throws XMLStreamException {
                return f.createXMLEventWriter(result);
            }

            public XMLEventWriter createXMLEventWriter(Writer w) throws XMLStreamException {
                return f.createXMLEventWriter(w);
            }

            public XMLStreamWriter createXMLStreamWriter(OutputStream out) throws XMLStreamException {
                return new CDataXmlStreamWriter(f.createXMLStreamWriter(out));
            }

            public XMLStreamWriter createXMLStreamWriter(OutputStream out, String enc) throws XMLStreamException {
                return new CDataXmlStreamWriter(f.createXMLStreamWriter(out, enc));
            }

            public XMLStreamWriter createXMLStreamWriter(Result result) throws XMLStreamException {
                return new CDataXmlStreamWriter(f.createXMLStreamWriter(result));
            }

            public XMLStreamWriter createXMLStreamWriter(Writer w) throws XMLStreamException {
                return new CDataXmlStreamWriter(f.createXMLStreamWriter(w));
            }

            public Object getProperty(String name) {
                return f.getProperty(name);
            }

            public boolean isPropertySupported(String name) {
                return f.isPropertySupported(name);
            }

            public void setProperty(String name, Object value) {
                f.setProperty(name, value);
            }

        }

        private class CDataXmlStreamWriter implements XMLStreamWriter {

            private XMLStreamWriter w;

            public CDataXmlStreamWriter(XMLStreamWriter w) {
                this.w = w;
            }

            public void close() throws XMLStreamException {
                w.close();
            }

            public void flush() throws XMLStreamException {
                w.flush();
            }

            public NamespaceContext getNamespaceContext() {
                return w.getNamespaceContext();
            }

            public String getPrefix(String uri) throws XMLStreamException {
                return w.getPrefix(uri);
            }

            public Object getProperty(String name) throws IllegalArgumentException {
                return w.getProperty(name);
            }

            public void setDefaultNamespace(String uri) throws XMLStreamException {
                w.setDefaultNamespace(uri);
            }

            public void setNamespaceContext(NamespaceContext context) throws XMLStreamException {
                w.setNamespaceContext(context);
            }

            public void setPrefix(String prefix, String uri) throws XMLStreamException {
                w.setPrefix(prefix, uri);
            }

            public void writeAttribute(String prefix, String namespaceURI, String localName, String value) throws XMLStreamException {
                w.writeAttribute(prefix, namespaceURI, localName, value);
            }

            public void writeAttribute(String namespaceURI, String localName, String value) throws XMLStreamException {
                w.writeAttribute(namespaceURI, localName, value);
            }

            public void writeAttribute(String localName, String value) throws XMLStreamException {
                w.writeAttribute(localName, value);
            }

            public void writeCData(String data) throws XMLStreamException {
                w.writeCData(data);
            }

            public void writeCharacters(char[] text, int start, int len) throws XMLStreamException {
                w.writeCharacters(text, start, len);
            }

            // All this code just to override this method
            public void writeCharacters(String text) throws XMLStreamException {
                w.writeCData(text);
            }

            public void writeComment(String data) throws XMLStreamException {
                w.writeComment(data);
            }

            public void writeDTD(String dtd) throws XMLStreamException {
                w.writeDTD(dtd);
            }

            public void writeDefaultNamespace(String namespaceURI) throws XMLStreamException {
                w.writeDefaultNamespace(namespaceURI);
            }

            public void writeEmptyElement(String prefix, String localName, String namespaceURI) throws XMLStreamException {
                w.writeEmptyElement(prefix, localName, namespaceURI);
            }

            public void writeEmptyElement(String namespaceURI, String localName) throws XMLStreamException {
                w.writeEmptyElement(namespaceURI, localName);
            }

            public void writeEmptyElement(String localName) throws XMLStreamException {
                w.writeEmptyElement(localName);
            }

            public void writeEndDocument() throws XMLStreamException {
                w.writeEndDocument();
            }

            public void writeEndElement() throws XMLStreamException {
                w.writeEndElement();
            }

            public void writeEntityRef(String name) throws XMLStreamException {
                w.writeEntityRef(name);
            }

            public void writeNamespace(String prefix, String namespaceURI) throws XMLStreamException {
                w.writeNamespace(prefix, namespaceURI);
            }

            public void writeProcessingInstruction(String target, String data) throws XMLStreamException {
                w.writeProcessingInstruction(target, data);
            }

            public void writeProcessingInstruction(String target) throws XMLStreamException {
                w.writeProcessingInstruction(target);
            }

            public void writeStartDocument() throws XMLStreamException {
                w.writeStartDocument();
            }

            public void writeStartDocument(String encoding, String version) throws XMLStreamException {
                w.writeStartDocument(encoding, version);
            }

            public void writeStartDocument(String version) throws XMLStreamException {
                w.writeStartDocument(version);
            }

            public void writeStartElement(String prefix, String localName, String namespaceURI) throws XMLStreamException {
                w.writeStartElement(prefix, localName, namespaceURI);
            }

            public void writeStartElement(String namespaceURI, String localName) throws XMLStreamException {
                w.writeStartElement(namespaceURI, localName);
            }

            public void writeStartElement(String localName) throws XMLStreamException {
                w.writeStartElement(localName);
            }
        }
    }


    public static class XStreamFacade<T> implements XmlInternalFacade<T> {
        private XStreamFacade() {
            LOGGER.debug("init XStream..");
        }

        private static final class LazyHolder {
            private static final XStreamFacade INSTANCE = new XStreamFacade<>();
        }

        public static XStreamFacade getInstance() {
            return XStreamFacade.LazyHolder.INSTANCE;
        }

        @Override
        public String toXml(Object object, boolean cdata, boolean isPretty) {
            return XstreamUtil.toXml(object, cdata, isPretty, null);
        }

        @Override
        public <T> T fromXml(String str, Class<T> clazz) {
            return XstreamUtil.fromXML(str, clazz);
        }
    }


}
