package cn.pengh.util;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.util.List;

public class CurrencyUtil {
    private static final BigDecimal BD0 = BigDecimal.ZERO;
    private static final BigDecimal BD1 = BigDecimal.ONE;
    private static final BigDecimal BD100 = BigDecimal.valueOf(100);
    private static final BigDecimal BD1_NEGATIVE = BigDecimal.valueOf(-1);
    private static final int SCALE = 2;
    private static final String CURRENCY_FORMAT = "#,#00.00#";
    private static final String CURRENCY_FORMAT_NUM = "#,#00.#";

    /**
     * Object转成BigDecimal
     * boolean 转成 0和1
     * char 转成 ASCII码，'a'是97，'A'是65，'0'是48，'1'是49
     *
     * @param money
     * @return BigDecimal
     */
    public static BigDecimal convert(Object money) {
        //Log.trace("convert before: " + money);
        if (money == null)
            return BD0;

        if (money instanceof BigDecimal) {
            return (BigDecimal) money;
        } else if (money instanceof BigInteger) {
            return new BigDecimal((BigInteger) money);
        } else if (money instanceof Double) {
            return BigDecimal.valueOf((Double) money);
        } else if (money instanceof Float) {
            return BigDecimal.valueOf((Float) money);
        } else if (money instanceof Long) {
            return BigDecimal.valueOf((Long) money);
        } else if (money instanceof Integer) {
            return BigDecimal.valueOf((Integer) money);
        } else if (money instanceof Short) {
            return BigDecimal.valueOf((Short) money);
        } else if (money instanceof Byte) {
            return BigDecimal.valueOf((Byte) money);
        } else if (money instanceof Character) {
            return BigDecimal.valueOf((Character) money);
        } else if (money instanceof Boolean) {
            Boolean b = (Boolean) money;
            return b ? BD1 : BD0;
        } else {
            String strMoney = money.toString();
            if (strMoney.length() == 0 || "0".equals(strMoney))
                return BD0;
			/*if (!strMoney.matches("^[\\d|\\-|\\.]+[\\d|\\.|\\/|\\-]+$")) {
				throw new NumberFormatException("请输入合法的数字，谢谢。"+strMoney);
			}*/

            String validateMoney = strMoney.replace(",", "").replace("/", "").replaceAll("-", "");
            if (validateMoney.length() == 0)
                return BD0;
            strMoney = (strMoney.startsWith("-") ? "-" : "") + validateMoney;
            //System.out.println("money = [" + strMoney + "]");
            //Log.trace("convert after: " + money);
            return new BigDecimal(strMoney);
        }
    }

    public static BigDecimal convert(Object money, int scale) {
        return convert(money).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    //元转分
    public static BigDecimal yuan2fen(Object money) {
        return yuan2fen(money, SCALE);
    }

    public static BigDecimal yuan2fen(Object money, int scale) {
        return convert(money).multiply(BD100).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    //分转元
    public static BigDecimal fen2yuan(Object money, int scale) {
        return convert(money).divide(BD100, scale, BigDecimal.ROUND_HALF_UP);
    }

    public static BigDecimal fen2yuan(Object money) {
        return fen2yuan(money, SCALE);
    }


    public static BigDecimal negative(Object money, int scale) {
        return convert(money).setScale(scale, BigDecimal.ROUND_HALF_UP).multiply(BD1_NEGATIVE);
    }


    public static long yuan2fenLong(Object money) {
        return yuan2fen(money).longValue();
    }

    public static int yuan2fenInt(Object money) {
        return yuan2fen(money).intValue();
    }

    public static String yuan2fenStr(Object money) {
        return yuan2fen(money).toString();
    }

    public static String fen2yuanStr(Object money) {
        return fen2yuan(money).toString();
    }

    public static int fen2yuanInt(Object money) {
        return fen2yuan(money).intValue();
    }

    public static String fen2yuanWithFormat(Object money) {
        return format(fen2yuanStr(money));
    }

    public static String fen2yuanWithFormatNoZero(Object money) {
        return formatWithoutZero(fen2yuanStr(money));
    }

    public static Double convertDouble(Object money) {
        return convert(money).doubleValue();
    }

    public static Double convertDouble20(Object money) {
        return convert(money, 20).doubleValue();
    }

    public static String convertStr(Object money) {
        return convert(money).toString();
    }

    public static String convertStr(Object money, int scale) {
        return convert(money, scale).toString();
    }

    public static String convertStrWithoutZero(Object money, int scale) {
        return formatTailZero(convertStr(money, scale));
    }

    //加法

    /**
     * 未scale化
     *
     * @param ds
     * @param <T>
     * @return
     */
    public static <T> BigDecimal add(T... ds) {
        if (ds == null)
            return BD0;

        BigDecimal bigD = BD0;
        for (T t : ds) {
            bigD = bigD.add(convert(t));
        }
        return bigD;
    }

    public static <T> BigDecimal add(int scale, List<T> ls) {
        return add(StringUtil.list2Array(ls)).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    public static <T> BigDecimal add(List<T> ls) {
        return add(SCALE, ls);
    }

    public static BigDecimal add(Object money, Object money2, int scale) {
        return convert(money).add(convert(money2)).setScale(scale, BigDecimal.ROUND_HALF_UP);
        //return add(SCALE,new String[]{ convert(money).toString(), convert(money2).toString()});
    }

    public static BigDecimal add(Object money, Object money2) {
        return add(money, money2, SCALE);
    }


    public static BigDecimal subtract(Object money, Object money2, int scale) {
        return convert(money).subtract(convert(money2)).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    public static BigDecimal subtract(Object money, Object money2) {
        return subtract(money, money2, SCALE);
    }


    //乘法
    public static <T> BigDecimal multiply(T... ds) {
        BigDecimal bigD = BD1;
        for (T d : ds) {
            if (d != null)
                bigD = bigD.multiply(convert(d));
        }
        return bigD;
    }

    public static <T> BigDecimal multiply(int scale, List<T> ls) {
        return multiply(StringUtil.list2Array(ls)).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    public static <T> BigDecimal multiply(List<T> ls) {
        return multiply(SCALE, ls);
    }

    public static BigDecimal multiply(Object money, Object money2, int scale) {
        return convert(money).multiply(convert(money2)).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    public static BigDecimal multiply(Object money, Object money2) {
        return multiply(money, money2, SCALE);
    }

    /**
     * 约定 0/0 == 0
     * son/mum
     *
     * @param son
     * @param mum
     * @return
     */
    public static BigDecimal divide(Object son, Object mum) {
        return divide(son, mum, SCALE);
    }

    /**
     * 约定 0/0 == 0
     *
     * @param son
     * @param mum
     * @param scale
     * @return
     */
    public static BigDecimal divide(Object son, Object mum, int scale) {
        BigDecimal mumBd = convert(mum);
        //分母为0
        return BD0.compareTo(mumBd) == 0 ? BD0 : convert(son).divide(mumBd, scale, BigDecimal.ROUND_HALF_UP);
    }

    public static String rate(Object son, Object mum) {
        return rate(son, mum, SCALE);//2位小数
    }

    public static String rate(Object son, Object mum, int scale) {
        return divide(convert(son).multiply(BD100), convert(mum), scale).toString();
    }

    public static double rate2(Object son, Object mum) {
        return rate2(son, mum, SCALE);//2位小数
    }
    public static double rate2(Object son, Object mum, int scale) {
        return divide(convert(son).multiply(BD100), convert(mum), scale).doubleValue();
    }

    public static String format(Object money) {
        return format(money, CURRENCY_FORMAT);
    }

    public static String formatNum(Object money) {
        return format(money, CURRENCY_FORMAT_NUM);
    }

    public static String format(Object money, String formatReg) {
        return new DecimalFormat(formatReg).format(convert(money).doubleValue()).toString().replaceAll("^0(\\d).(\\d+)$", "$1.$2");
    }

    public static String bytes2HexString(byte[] b) {
        return StringUtil.byteToHex(b);
    }

    public static String formatWithoutZero(Object money) {
        return formatTailZero(format(money, CURRENCY_FORMAT));
    }

    public static String formatTailZero(String money) {
        return money.replaceAll("0+?$", "")//去掉多余的0
                .replaceAll("[.]$", "");//如最后一位是.则去掉;
    }

    // long start = System.nanoTime();
    public static BigDecimal duration(long start) {
        return duration(start, 6);
    }

    // 计算耗时，单位s
    public static BigDecimal duration(long start, int scale) {
        return CurrencyUtil.divide((System.nanoTime() - start), 1e9, scale);
    }


}
