package cn.pengh.http;

import cn.pengh.helper.ClazzHelper;
import cn.pengh.helper.SignHelper;
import cn.pengh.util.CurrencyUtil;
import cn.pengh.util.JsonUtil;
import cn.pengh.util.RandomUtil;
import cn.pengh.util.StringUtil;
import okhttp3.*;
import okhttp3.internal.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 依赖 "com.squareup.okhttp3:okhttp:$okhttp_version",
 *
 * @author Created by pengh
 * @datetime 2020/6/12 11:02
 */
public class OkHttpUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(OkHttpUtil.class);
    private static final String CHARSET = "utf-8";
    private static final String MEDIA_TYPE_JSON = "application/json";
    private static final String MEDIA_TYPE_FORM = "application/x-www-form-urlencoded";
    private static final MediaType CONTENT_TYPE_JSON_UTF8 = MediaType.parse(MEDIA_TYPE_JSON + "; charset=" + CHARSET);
    private static final MediaType CONTENT_TYPE_FORM_UTF8 = MediaType.parse(MEDIA_TYPE_FORM + "; charset=" + CHARSET);

    private static Map<Integer, OkHttpClient> okHttpClientMap = new ConcurrentHashMap<>();
    private static final int INSTANCE_MAX = 15; //本地map里面最多15个okHttpClient实例，当大于则清空所有
    private static final int TIMEOUT = 5000; //默认请求超时5s
    private static final int CONNECT_TIMEOUT_FACTOR = 5; //连接超时，一般 == 超时/5，最低100ms
    private static final int CONNECT_TIMEOUT_MIN = 100; //连接超时，最低100ms
    private static final int CONNECT_MAX = 5; //okHttp默认5个，空闲TCP接连，即连接最多被缓存5个，每一个最多空闲5分钟
    private static final int KEEP_ALIVE_DURATION_SECONDS = 5 * 60; //okHttp默认5分钟，空闲活跃时间

    private static final int REQUEST_MAX = 64; //okHttp默认64个，最大的并发请求数
    private static final int REQUEST_HOST_MAX = 5; //okHttp默认5个，单主机最大请求并发数

    private static class OkHttpClientBuilderLazyHolder {
        private static final OkHttpClient.Builder INSTANCE = new OkHttpClient.Builder();
    }

    private static OkHttpClient getInstance(int timeout) {
        OkHttpClient okHttpClient = okHttpClientMap.get(timeout);
        if (okHttpClient == null) {
            if (okHttpClientMap.size() > INSTANCE_MAX) {
                okHttpClientMap.clear();
            }

            int connectTimeout = timeout / CONNECT_TIMEOUT_FACTOR; //连接超时，一般 == 超时/5，最低100ms
            connectTimeout = connectTimeout < 100 ? CONNECT_TIMEOUT_MIN : connectTimeout;
            int maxIdleConnections = CONNECT_MAX, keepAliveDurationSeconds = KEEP_ALIVE_DURATION_SECONDS;

            /*
            高并发，轻量级，代理多，主要消耗cpu，推荐
             -Dhttp.max.request=200 -Dhttp.max.host.request=10 -Dhttp.pool.core=20
             -Dhttp.max.request=1000 -Dhttp.max.host.request=20 -Dhttp.pool.core=50
            其他默认即可
             */
            int maxRequests = StringUtil.getSystemProperty("http.max.request", REQUEST_MAX),
                    maxRequestsPerHost = StringUtil.getSystemProperty("http.max.host.request", REQUEST_HOST_MAX),
                    corePoolSize = StringUtil.getSystemProperty("http.pool.core", 0);

            Dispatcher dp = new Dispatcher(new ThreadPoolExecutor(corePoolSize, Integer.MAX_VALUE, 60, TimeUnit.SECONDS,
                    new SynchronousQueue<>(), Util.threadFactory("OkHttp Dispatcher", false)));
            dp.setMaxRequests(maxRequests);
            dp.setMaxRequestsPerHost(maxRequestsPerHost);

            LOGGER.debug("OkHttp Dispatcher init, maxRequests:{},maxRequestsPerHost:{},poolCore:{},timeout:{}",
                    maxRequests, maxRequestsPerHost, corePoolSize, timeout);

            okHttpClient = OkHttpClientBuilderLazyHolder.INSTANCE
                    .dispatcher(dp)
                    .retryOnConnectionFailure(true) //默认true，connect失败时重试一次
                    .connectionPool(new ConnectionPool(maxIdleConnections, keepAliveDurationSeconds, TimeUnit.SECONDS))
                    .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
                    .readTimeout(timeout, TimeUnit.MILLISECONDS)
                    .writeTimeout(timeout, TimeUnit.MILLISECONDS)
                    .build();
            okHttpClientMap.put(timeout, okHttpClient);
        }
        return okHttpClient;
    }


    public static String post(String url, Object obj) {
        return post(url, obj, null);
    }

    public static String post(String url, Object obj, int timeout) {
        return post(url, obj, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String post(String url, Object obj, HttpRequest.HttpRequestConfig config) {
        return postFormMap(url, ClazzHelper.KV(obj), config);
    }

    public static String post(String url, Map<String, String> param) {
        return post(url, param, TIMEOUT);
    }

    public static String post(String url, Map<String, String> param, int timeout) {
        return post(url, param, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String get(String url) {
        return get(url, HttpRequest.HttpRequestConfig.createDefault());
    }

    public static String postJson(String url, String json) {
        return postJson(url, json, TIMEOUT);
    }

    public static String postJson(String url, String json, int timeout) {
        return postJson(url, json, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String postJson(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return post(url, json, getMediaTypeJson(config), config);
    }

    public static Object[] postJsonWithCode(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return postWithCode(url, json, getMediaTypeJson(config), config);
    }

    public static String post(String url, Map<String, String> param, HttpRequest.HttpRequestConfig config) {
        return postFormMap(url, ClazzHelper.KV(param), config);
    }

    public static String postFormMap(String url, Map<Object, Object> param, HttpRequest.HttpRequestConfig config) {
        //LOGGER.debug("Params: {}", param);
        /*FormBody.Builder formBodyBuilder = new FormBody.Builder(Charset.forName(config == null ? "utf-8" : config.getCharset()));
        param.forEach((k, v) -> formBodyBuilder.add(k, v));
        return doHttp(new Request.Builder().url(url).post(formBodyBuilder.build()), config);*/
        //解决中文乱码问题
        StringBuilder sb = new StringBuilder();
        /*param.forEach((k, v) -> sb.append(k).append("=").append(v).append("&"));
        String body = sb.toString().replaceAll("(.*)&$", "$1");
        LOGGER.debug("Params: {}", body);*/

        FormBody.Builder formBodyBuilder = new FormBody.Builder(Charset.forName(config == null ? CHARSET : config.getCharset()));
        try {
            param.forEach((k, v) -> formBodyBuilder.add(k.toString(), v.toString()));
        } catch (Exception e) {
            e.printStackTrace();
        }

        FormBody fb = formBodyBuilder.build();
        for (int i = 0; i < fb.size(); i++) {
            sb.append(fb.encodedName(i)).append("=").append(fb.encodedValue(i)).append("&");
        }
        String body = sb.toString().replaceAll("(.*)&$", "$1");
        LOGGER.debug("Params: {}", body);
        return post(url, body, getMediaTypeForm(config), config);
    }

    private static MediaType getMediaTypeForm(HttpRequest.HttpRequestConfig config) {
        return config == null || CHARSET.equalsIgnoreCase(config.getCharset()) ? CONTENT_TYPE_FORM_UTF8 : MediaType.parse(MEDIA_TYPE_FORM + "; charset=" + config.getCharset());
    }

    private static MediaType getMediaTypeJson(HttpRequest.HttpRequestConfig config) {
        return config == null || CHARSET.equalsIgnoreCase(config.getCharset()) ? CONTENT_TYPE_JSON_UTF8 : MediaType.parse(MEDIA_TYPE_JSON + "; charset=" + config.getCharset());
    }


    public static String putJson(String url, String json) {
        return putJson(url, json, null);
    }

    public static String putJson(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return doHttp(new Request.Builder().url(url).put(RequestBody.create(getMediaTypeJson(config), json)), config);
    }

    public static String post(String url, String body, MediaType mediaType, HttpRequest.HttpRequestConfig config) {
        return doHttp(new Request.Builder().url(url).post(RequestBody.create(mediaType, body)), config);
    }

    private static Object[] postWithCode(String url, String body, MediaType mediaType, HttpRequest.HttpRequestConfig config) {
        return doHttpWithCode(new Request.Builder().url(url).post(RequestBody.create(mediaType, body)), config);
    }

    public static String get(String url, HttpRequest.HttpRequestConfig config) {
        return doHttp(new Request.Builder().url(url).get(), config);
    }

    public static String get(String url, Map<String, String> headers) {
        return get(url, HttpRequest.HttpRequestConfig.createDefault().setHeaders(headers));
    }


    private static String doHttp(Request.Builder requestBuilder, HttpRequest.HttpRequestConfig config) {
        Object[] r = doHttpWithCode(requestBuilder, config);
        return r[1] == null ? null : r[1].toString();
    }

    public static Object[] doHttpWithCode(Request.Builder requestBuilder, HttpRequest.HttpRequestConfig config) {
        long start = System.nanoTime();

        int timeout = TIMEOUT;
        if (config != null) {
            timeout = config.getTimeout();
            Map<String, String> headers = config.getHeaders();
            if (headers != null) {
                headers.forEach((k, v) -> {
                    requestBuilder.addHeader(k, v);
                    LOGGER.debug("{}: {}", k, v);
                });
            }
        }

        Request request = requestBuilder.build();
        LOGGER.debug("{} {} {}. Timeout {}ms", request.method(), request.url(), request.body() == null ? "" : request.body().contentType(), timeout);
        //LOGGER.debug("Params: {}", JsonUtil.toJson(request.body()));
        try {
            Response response = getInstance(timeout).newCall(request).execute();
            String res = response.body().string();
            LOGGER.debug("{}", response.headers().toString().replace("\n", " "));
            LOGGER.debug("Response {}: {}", response.code(), res);
            return new Object[]{response.code(), res};
        } catch (Exception e) {
            e.printStackTrace();
            return new Object[]{500, null};
        } finally {
            LOGGER.debug("Time elapsed: {}s", CurrencyUtil.divide(System.nanoTime() - start, 1e9, 6));
        }

    }

    //https://opendocs.alipay.com/open-v3/054q58
    //https://pay.weixin.qq.com/docs/merchant/development/interface-rules/signature-generation.html
    public static Object[] postJsonOSI3(String url, Object json, HttpRequest.HttpRequestConfig config) {
        return postJsonOSI3(url, JsonUtil.toJson(json), config);
    }

    /**
     * @param url
     * @param json
     * @param config
     * @return [code, resBodyStr]
     */
    public static Object[] postJsonOSI3(String url, String json, HttpRequest.HttpRequestConfig config) {
        String schema = config.getSignType() == null ? "SHA256withRSA" : config.getSignType(), timestamp = System.currentTimeMillis() + "", nonce = RandomUtil.getNonceStr();
        String authNotSign = getAuth(config.getAppId(), timestamp, nonce, config.getRootCertSN()), beforeSign = getBeforeSign(HttpUrl.parse(url), authNotSign, json);
        String sign = getSign(beforeSign, schema, config.getRsa2PriKey());
        String authorization = schema + " " + authNotSign + ",sign=" + sign;

        LOGGER.trace("beforeSign:\n{}", beforeSign);

        Map<String, String> headers = new HashMap<>();
        headers.put("Authorization", authorization);

        return postJsonWithCode(url, json, config.setHeaders(headers).setForceStatus200(false));
    }

    private static String getAuth(int appId, String timestamp, String nonce, String sn) {
        return "appId=" + appId + ","
                + (StringUtil.isEmpty(sn) ? "" : "sn=" + sn + ",")
                + "timestamp=" + timestamp + ","
                + "nonce=" + nonce;
    }

    private static String getBeforeSign(HttpUrl url, String authNotSign, String json) {
        return getBeforeSign("POST", url, authNotSign, json);
    }

    private static String getBeforeSign(String method, HttpUrl url, String authNotSign, String json) {
        String uri = url.encodedPath();
        if (url.encodedQuery() != null) {
            uri += "?" + url.encodedQuery();
        }
        return method + "\n"
                + uri + "\n"
                + authNotSign + "\n"
                + (json == null ? "" : json) + "\n";
    }

    public static boolean verifySignOSI3(String url, String authorization, String json, char[] rsaPubKey) {
        String[] arr = authorization.split(" ");
        if (arr.length != 2) {
            return false;
        }
        Map<String, String> m = StringUtil.paramToMap(arr[1], ",");

        String signType = arr[0], sign = m.get("sign"),
                authNotSign = getAuth(Integer.valueOf(m.get("appId")), m.get("timestamp"), m.get("nonce"), m.get("sn")),
                beforeSign = getBeforeSign(HttpUrl.parse(url), authNotSign, json);

        if (TYPE_RSA2.contains(signType)) {
            return SignHelper.verifySHA256WithRSA(beforeSign, rsaPubKey, sign);
        } else if (TYPE_SM3_SM2.contains(signType)) {
            return SignHelper.verifySM3WithSM2(beforeSign, rsaPubKey, sign);
        }
        return SignHelper.verifySHA256WithRSA(beforeSign, rsaPubKey, sign);
    }


    private static String getSign(String beforeSign, String signType, char[] rsa2PriKey) {
        if (TYPE_RSA2.contains(signType)) {
            return SignHelper.signSHA256WithRSA(beforeSign, rsa2PriKey);
        } else if (TYPE_SM3_SM2.contains(signType)) {
            return SignHelper.signSM3WithSM2(beforeSign, rsa2PriKey);
        }
        return SignHelper.signSHA256WithRSA(beforeSign, rsa2PriKey);
    }

    private static final List<String> TYPE_RSA2 = Arrays.asList("SHA256withRSA", "ALIPAY-SHA256withRSA", "WECHATPAY2-SHA256-RSA2048");
    private static final List<String> TYPE_SM3_SM2 = Arrays.asList("ALIPAY-SM3withSM2");

}
