package cn.pengh.helper;

import cn.pengh.crypt.Md5;
import cn.pengh.crypt.asymmetric.AsymmetricFactory;
import cn.pengh.exception.CustomException;
import cn.pengh.util.TemplateUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * @author Created by pengh
 * @datetime 2018/10/11 10:46
 */
public class SignHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(SignHelper.class);

    public static final String CHARSET = "utf-8";
    public static final String SIGN_MD5 = "MD5";
    @Deprecated
    public static final String SIGN_SHA = "SHA";
    @Deprecated
    public static final String SIGN_SHA128 = "SHA128";
    public static final String SIGN_SHA256 = "SHA256";
    public static final String SIGN_SHA384 = "SHA384";
    public static final String SIGN_SHA512 = "SHA512";
    public static final String SIGN_RSA2 = "RSA2";
    public static final String SIGN_RSA = "RSA";
    public static final String SIGN_SHA256_RSA = "SHA256RSA";
    public static final String SIGN_SHA1_RSA = "SHA1RSA";
    public static final String SIGN_HMAC_SHA256 = "HMAC-SHA256";
    private static final String HMAC_SHA256 = "HmacSHA256";


    public static String sign(String signType, String appKey, Object obj) {
        return sign(signType, appKey, obj, null);
    }

    public static String sign(String signType, String appKey, Object obj, List<String> excepts) {
        return sign(signType, appKey, obj, excepts, null);
    }

    /**
     * 默认md5加密
     *
     * @param signType
     * @param appKey
     * @param obj
     * @param excepts
     * @param replaceMap
     * @return
     */
    public static String sign(String signType, String appKey, Object obj, List<String> excepts, Map<String, String> replaceMap) {
        String str = ClazzHelper.getSortedFieldsVal(obj, excepts, replaceMap, true);
        return sign(signType, str, appKey);
    }

    //仅限某些情况下
    public static String sign(String signType, String str) {
        return sign(signType, str, null);
    }

    public static String sign(String signType, String str, String appKey) {
        LOGGER.trace(str);
        String strWithKey = appKey == null ? str : str + "&key=" + appKey;

        switch (signType.toUpperCase()) {
            case SIGN_MD5:
                return CryptHelper.md5(strWithKey).toUpperCase();
            case SIGN_SHA256:
                return CryptHelper.sha256(strWithKey).toUpperCase();
            case SIGN_SHA512:
                return CryptHelper.sha512(strWithKey).toUpperCase();
            case SIGN_SHA384:
                return CryptHelper.sha384(strWithKey).toUpperCase();
            case SIGN_SHA256_RSA:
            case SIGN_RSA2:
                return signSHA256WithRSA(str, appKey);
            case SIGN_RSA:
            case SIGN_SHA1_RSA:
                return signSHA1WithRSA(str, appKey);
            case SIGN_HMAC_SHA256:
                return doHMacSHA256(strWithKey, appKey).toUpperCase();
            case SIGN_SHA:
            case SIGN_SHA128:
                return CryptHelper.sha128(strWithKey).toUpperCase();
            default:
                throw CustomException.create("不支持的签名类型：" + signType);
        }
    }

    public static boolean verify(String sign, String signType, String appKey, Object obj, List<String> excepts, Map<String, String> replaceMap) {
        switch (signType.toUpperCase()) {
            case SIGN_MD5:
            case SIGN_SHA256:
            case SIGN_SHA512:
            case SIGN_SHA384:
            case SIGN_SHA:
            case SIGN_SHA128:
            case SIGN_HMAC_SHA256:
                return sign(signType, appKey, obj, excepts, replaceMap).equalsIgnoreCase(sign);
            case SIGN_SHA256_RSA:
            case SIGN_RSA2:
                return verifySHA256WithRSA(ClazzHelper.getSortedFieldsVal(obj, excepts, replaceMap, true), appKey, sign);
            case SIGN_RSA:
            case SIGN_SHA1_RSA:
                return verifySHA1WithRSA(ClazzHelper.getSortedFieldsVal(obj, excepts, replaceMap, true), appKey, sign);
            default:
                throw CustomException.create("不支持的签名类型：" + signType);
        }
    }

    //仅限某些情况下
    public static boolean verify(String sign, String signType, String plainText) {
        return verify(sign, signType, plainText, null);
    }

    /**
     * @param sign
     * @param signType
     * @param plainText
     * @return
     */
    public static boolean verify(String sign, String signType, String plainText, String appKey) {
        switch (signType.toUpperCase()) {
            case SIGN_MD5:
            case SIGN_SHA256:
            case SIGN_SHA512:
            case SIGN_SHA384:
            case SIGN_SHA:
            case SIGN_SHA128:
            case SIGN_HMAC_SHA256:
                return sign(signType, plainText, appKey).equalsIgnoreCase(sign);
            case SIGN_SHA256_RSA:
            case SIGN_RSA2:
                return verifySHA256WithRSA(plainText, appKey, sign);
            case SIGN_RSA:
            case SIGN_SHA1_RSA:
                return verifySHA1WithRSA(plainText, appKey, sign);
            default:
                throw CustomException.create("不支持的签名类型：" + signType);
        }
    }

    private static String byteArrayToHexString(byte[] b) {
        StringBuilder hs = new StringBuilder();
        String stmp;
        for (int n = 0; b != null && n < b.length; n++) {
            stmp = Integer.toHexString(b[n] & 0XFF);
            if (stmp.length() == 1)
                hs.append('0');
            hs.append(stmp);
        }
        return hs.toString().toLowerCase();
    }

    /**
     * sha256_HMAC加密
     *
     * @param message 消息
     * @param secret  秘钥
     * @return 加密后字符串
     */
    public static String doHMacSHA256(String message, String secret) {
        try {
            Mac hMacSha256 = Mac.getInstance(HMAC_SHA256);
            SecretKeySpec secretKey = new SecretKeySpec(secret.getBytes(CHARSET), HMAC_SHA256);
            hMacSha256.init(secretKey);
            //byte[] bytes = hMacSha256.doFinal(message.getBytes(CHARSET));
            //hash = Base64.encodeBase64String(bytes);
            return byteArrayToHexString(hMacSha256.doFinal(message.getBytes(CHARSET)));
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    public static String signSHA256WithRSA(String message, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA).sign(message, privateKey);
    }
    public static String signSHA256WithRSA(String message, char[] privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA).sign(message, String.valueOf(privateKey));
    }

    public static String signSM3WithSM2(String message, char[] privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SM3SM2).sign(message, String.valueOf(privateKey));
    }

    @Deprecated
    public static String signSHA1WithRSA(String message, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA128RSA).sign(message, privateKey);
    }

    @Deprecated
    public static String signMD5WithRSA(String message, char[] privateKey, String charset, boolean base64When76Enter) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.MD5RSA).sign(message, String.valueOf(privateKey), charset, base64When76Enter);
    }

    public static boolean verifySHA256WithRSA(String message, String publicKey, String sign) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA).verify(message, publicKey, sign);
    }

    public static boolean verifySHA256WithRSA(String message, char[] publicKey, String sign) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA).verify(message, String.valueOf(publicKey), sign);
    }

    public static boolean verifySM3WithSM2(String message, char[] publicKey, String sign) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SM3SM2).verify(message, String.valueOf(publicKey), sign);
    }

    @Deprecated
    public static boolean verifySHA1WithRSA(String message, String publicKey, String sign) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA128RSA).verify(message, publicKey, sign);
    }

    @Deprecated
    public static boolean verifyMD5WithRSA(String message, char[] publicKey, String sign, String charset) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.MD5RSA).verify(message, String.valueOf(publicKey), sign, charset);
    }

    public static String signSha256(String appKey, Object obj) {
        return signSha256(appKey, obj, "");
    }


    public static String signSha256(String appKey, Object obj, String excepts) {
        return signSha256(appKey, obj, excepts == null ? null : Arrays.asList(excepts.split(",")));
    }

    public static String signSha256(String appKey, Object obj, List<String> excepts) {
        return signSha256(appKey, obj, excepts, null);
    }

    public static String signSha256(String appKey, Object obj, List<String> excepts, Map<String, String> replaceMap) {
        return sign(SIGN_SHA256, appKey, obj, excepts, replaceMap);
    }


    public static String signMd5(String appKey, Object obj) {
        return signMd5(appKey, obj, "");
    }


    public static String signMd5(String appKey, Object obj, String excepts) {
        return signMd5(appKey, obj, excepts == null ? null : Arrays.asList(excepts.split(",")));
    }

    public static String signMd5(String appKey, Object obj, List<String> excepts) {
        return signMd5(appKey, obj, excepts, null);
    }

    /**
     * @param appKey
     * @param obj
     * @param excepts    不需要加入签名的属性字段
     * @param replaceMap 有些属性字段需要替换，使用别名
     * @return
     */
    public static String signMd5(String appKey, Object obj, List<String> excepts, Map<String, String> replaceMap) {
        return sign(SIGN_MD5, appKey, obj, excepts, replaceMap);
    }


    /**
     * @param template
     * @param appKey
     * @param obj
     * @return
     */
    public static String signMd5(String template, String appKey, Object obj) {
        Map<Object, Object> map = ClazzHelper.KV(obj);
        return Md5.MD5Encode(TemplateUtil.renderStr(template, map) + "&key=" + appKey).toUpperCase();
    }
}
