package cn.pengh.crypt.asymmetric;

import javax.crypto.Cipher;
import javax.crypto.NullCipher;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;

/**
 *
 * ssh key的类型有四种，分别是dsa、rsa、 ecdsa、ed25519。根据数学特性，这四种类型又可以分为两大类，dsa/rsa是一类，ecdsa/ed25519是一类，后者算法更先进。
 * dsa因为安全问题，已不再使用了。ecdsa因为政治原因和技术原因，也不推荐使用。
 * rsa是目前兼容性最好的，应用最广泛的key类型，在用ssh-keygen工具生成key的时候，默认使用的也是这种类型。不过在生成key时，如果指定的key size太小的话，也是有安全问题的，推荐key size是3072或更大。
 * ed25519是目前最安全、加解密速度最快的key类型，由于其数学特性，它的key的长度比rsa小很多，优先推荐使用。它目前唯一的问题就是兼容性，即在旧版本的ssh工具集中可能无法使用。
 *
 * https://zhuanlan.zhihu.com/p/524180490
 *
 * 推荐使用bouncycastle库实现 ECC ECDSA
 * static {
 *         Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
 *     }
 * @author Created by pengh
 * @datetime 2021/6/11 11:22
 */
public class SHA256withECDSA extends SHA256WithRSA implements IAsymmetricEncryptor {

    @Override
    protected String getSignType() {
        return AsymmetricFactory.Algorithm.SHA256ECDSA;
    }

    @Override
    protected String getAsymmetricKeyType() {
        return AsymmetricFactory.AsymmetricAlgorithm.EC;
    }
    @Override
    protected String getAsymmetricCipherAlgorithm() {
        return AsymmetricFactory.AsymmetricAlgorithm.EC;
    }

    protected Cipher getPubEnCipher(PublicKey pubKey2) throws Exception {
        ECPublicKey pubKey = (ECPublicKey) pubKey2;

        Cipher cipher = new NullCipher();
        cipher.init(Cipher.ENCRYPT_MODE, pubKey, new ECPublicKeySpec(pubKey.getW(), pubKey.getParams()).getParams());
        return cipher;
    }

    @Override
    protected Cipher getPriDeCipher(PrivateKey priKey2) throws Exception {
        ECPrivateKey priKey = (ECPrivateKey) priKey2;

        Cipher cipher = new NullCipher();
        cipher.init(Cipher.DECRYPT_MODE, priKey, new ECPrivateKeySpec(priKey.getS(), priKey.getParams()).getParams());
        return cipher;
    }
}
