package cn.pengh.crypt.asymmetric;

import cn.pengh.exception.CustomException;

import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * @author Created by pengh
 * @datetime 2021/6/10 10:05
 */
public abstract class AbstractAsymmetricEncryptor implements IAsymmetricEncryptor {
    abstract protected String getSignType(); //非对称加密算法类型，如SHA256WithRSA

    abstract protected String getAsymmetricKeyType(); //RSA,EC

    abstract protected String getAsymmetricCipherAlgorithm(); //RSA,EC

    protected int getMaxDecryptBlockSize() {
        return -1; //no use
    }
    protected int getMaxEncryptBlockSize() {
        return -1; //no use
    }

    protected String getCipherProvider() {
        return null;
    }

    @Override
    public String encrypt(String plainText, String publicKey, String charset) {
        try {
            return encryptIt(plainText, publicKey, charset);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public String decrypt(String cipherText, String privateKey, String charset) {
        try {
            return decryptIt(cipherText, privateKey, charset);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public String sign(String message, String privateKey, String charset, boolean base64When76Enter) {
        try {
            return signIt(message, privateKey, charset, base64When76Enter);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public boolean verify(String message, String publicKey, String sign, String charset) {
        try {
            return verifyIt(message, publicKey, sign, charset);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }


    protected String encryptIt(String plainText, String publicKey, String charset) throws Exception {
        PublicKey pubKey = getPublicKeyFromX509(getAsymmetricKeyType(), publicKey.getBytes(charset));

        byte[] encryptedData = encryptIt(plainText.getBytes(charset), pubKey);
        return new String(encryptedData, charset);
    }

    protected byte[] encryptIt(byte[] data, PublicKey pubKey) throws Exception {
        int maxEncrypt = getMaxEncryptBlockSize();
        Cipher cipher = getPubEnCipher(pubKey);
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            int inputLen = data.length;

            int offSet = 0;
            byte[] cache;
            int i = 0;
            // 对数据分段加密
            while (inputLen - offSet > 0) {
                if (inputLen - offSet > maxEncrypt) {
                    cache = cipher.doFinal(data, offSet, maxEncrypt);
                } else {
                    cache = cipher.doFinal(data, offSet, inputLen - offSet);
                }
                out.write(cache, 0, cache.length);
                i++;
                offSet = i * maxEncrypt;
            }
            return base64Encode(out.toByteArray());
        } catch (Exception e) {
            throw e;
        }
    }

    protected String decryptIt(String cipherText, String privateKey, String charset) throws Exception {
        if (cipherText == null) {
            throw CustomException.create("cipher密文不能为空");
        }
        PrivateKey priKey = getPrivateKeyFromPKCS8(getAsymmetricKeyType(), privateKey.getBytes(charset));
        byte[] decryptedData = decryptIt(base64Decode(cipherText.getBytes(charset)), priKey);
        return new String(decryptedData, charset);
    }

    protected byte[] decryptIt(byte[] encryptedData, PrivateKey priKey) throws Exception {
        int maxDecrypt = getMaxDecryptBlockSize();
        Cipher cipher = getPriDeCipher(priKey);
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            int inputLen = encryptedData.length;
            int offSet = 0;
            byte[] cache;
            int i = 0;
            // 对数据分段解密
            while (inputLen - offSet > 0) {
                if (inputLen - offSet > maxDecrypt) {
                    cache = cipher.doFinal(encryptedData, offSet, maxDecrypt);
                } else {
                    cache = cipher.doFinal(encryptedData, offSet, inputLen - offSet);
                }
                out.write(cache, 0, cache.length);
                i++;
                offSet = i * maxDecrypt;
            }
            return out.toByteArray();
        } catch (Exception e) {
            throw e;
        }
    }

    protected Cipher getPubEnCipher(PublicKey pubKey) throws Exception {
        Cipher cipher = getCipher();
        cipher.init(Cipher.ENCRYPT_MODE, pubKey);
        return cipher;
    }

    protected Cipher getPriDeCipher(PrivateKey priKey) throws Exception {
        Cipher cipher = getCipher();
        cipher.init(Cipher.DECRYPT_MODE, priKey);
        return cipher;
    }
    private Cipher getCipher() throws Exception {
        String p = getCipherProvider();
        return p == null ? Cipher.getInstance(getAsymmetricCipherAlgorithm()) : Cipher.getInstance(getAsymmetricCipherAlgorithm(), p);
    }
    private Signature getSignature() throws Exception {
        String p = getCipherProvider();
        return p == null ? Signature.getInstance(getSignType()) : Signature.getInstance(getSignType(), p);
    }

    protected String signIt(String message, String privateKey, String charset, boolean base64When76Enter) throws Exception {
        PrivateKey priKey = getPrivateKeyFromPKCS8(getAsymmetricKeyType(), privateKey.getBytes(charset));
        Signature signature = getSignature();
        signature.initSign(priKey);
        signature.update(message.getBytes(charset));

        return new String(base64Encode(signature.sign(), base64When76Enter));
    }



    protected boolean verifyIt(String message, String publicKey, String sign, String charset) throws Exception {
        PublicKey pubKey = getPublicKeyFromX509(getAsymmetricKeyType(), publicKey.getBytes(charset));
        Signature signature = getSignature();
        signature.initVerify(pubKey);
        signature.update(message.getBytes(charset));

        return signature.verify(base64Decode(sign.getBytes(charset)));
    }

    public static PrivateKey getPrivateKeyFromPKCS8(String algorithm, byte[] priKey) throws Exception {
        /*try (InputStream ins = new ByteArrayInputStream(priKey)) {
            if (ins == null || StringUtil.isEmpty(algorithm)) {
                return null;
            }

            KeyFactory keyFactory = KeyFactory.getInstance(algorithm);

            byte[] encodedKey = FileUtil.ReadFile(ins);

            return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(base64Decode(encodedKey)));
        }*/
        return KeyFactory.getInstance(algorithm).generatePrivate(getPKCS8PrivateKey(priKey));
    }

    protected static PKCS8EncodedKeySpec getPKCS8PrivateKey(byte[] priKey) {
        return new PKCS8EncodedKeySpec(base64Decode(priKey));
    }

    public static PublicKey getPublicKeyFromX509(String algorithm, byte[] pubKey) throws Exception {
        /*try (InputStream ins = new ByteArrayInputStream(pubKey);
             InputStreamReader reader = new InputStreamReader(ins)) {
            KeyFactory keyFactory = KeyFactory.getInstance(algorithm);

            StringWriter writer = new StringWriter();
            StreamUtil.io(reader, writer);

            byte[] encodedKey = writer.toString().getBytes();

            return keyFactory.generatePublic(new X509EncodedKeySpec(base64Decode(encodedKey)));
        }*/
        return KeyFactory.getInstance(algorithm).generatePublic(getX509PublicKey(pubKey));
    }

    protected static X509EncodedKeySpec getX509PublicKey(byte[] pubKey) {
        return new X509EncodedKeySpec(base64Decode(pubKey));
    }


    /*public static String base64EncodeStr(byte[] data) {
        //return java.util.Base64.getEncoder().encodeToString(data);
        //return org.apache.commons.codec.binary.Base64.encodeBase64String(data);
        return cn.pengh.crypt.Base64.encode(data);
    }*/

    public static byte[] base64Encode(byte[] data) {
        //return java.util.Base64.getEncoder().encode(data);
        //return cn.pengh.crypt.Base64.encodeWithoutEnter(data).getBytes(StandardCharsets.UTF_8);
        //return org.apache.commons.codec.binary.Base64.encodeBase64(data, false);
        return base64Encode(data, false);
    }

    public static byte[] base64Encode(byte[] data, boolean base64When76Enter) {
        //return cn.pengh.crypt.Base64.encodeWithoutEnter(data).getBytes(StandardCharsets.UTF_8);
        return org.apache.commons.codec.binary.Base64.encodeBase64(data, base64When76Enter);
    }

    /*public static byte[] base64Decode(String src) {
        //return java.util.Base64.getDecoder().decode(src);
        return org.apache.commons.codec.binary.Base64.decodeBase64(src);
        //return cn.pengh.crypt.Base64.decode(src);
    }*/

    public static byte[] base64Decode(byte[] src) {
        //return java.util.Base64.getDecoder().decode(src);
        //return cn.pengh.crypt.Base64.decode(new String(src));
        return org.apache.commons.codec.binary.Base64.decodeBase64(src);
    }


    public static final String[] genRsaPriAndPubKey() {
        return genRsaPriAndPubKey(2048);
    }

    /**
     * 空密码的公私玥
     *
     * @param keySize 2048, 4096...
     * @return [priKey, pubKey]
     */
    public static final String[] genRsaPriAndPubKey(int keySize) {
        KeyPairGenerator keyPairGenerator = null;
        try {
            keyPairGenerator = KeyPairGenerator.getInstance(AsymmetricFactory.AsymmetricAlgorithm.RSA);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        keyPairGenerator.initialize(keySize, new SecureRandom());
        KeyPair keyPair = keyPairGenerator.generateKeyPair();

        //JcaPKCS8Generator gen1 = new JcaPKCS8Generator(keyPair.getPrivate(), null);
        //PemObject obj1 = gen1.generate();

        //Log.getSlf4jLogger().debug("private key: {}", new String(base64Encode(keyPair.getPrivate().getEncoded())));
        //Log.getSlf4jLogger().debug("public key: {}", new String(base64Encode(keyPair.getPublic().getEncoded())));

        return new String[]{new String(base64Encode(keyPair.getPrivate().getEncoded())),
                new String(base64Encode(keyPair.getPublic().getEncoded()))};
    }
}
