package cn.pengh.core.log4j.appender;

import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.appender.AppenderLoggingException;
import org.apache.logging.log4j.core.config.Property;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.layout.SerializedLayout;
import org.apache.logging.log4j.core.util.Booleans;

import java.io.Serializable;
import java.net.InetAddress;
import java.util.Properties;

/**
 * @author Created by pengh
 * @datetime 2017/5/18 14:33
 */
@Plugin(name = "Kafka", category = "Core", elementType = "appender", printObject = true)
public final class KafkaAppender extends AbstractAppender {

    private KafkaProducer<String, String> producer = null;
    private String topic;
    private boolean syncsend;

    protected KafkaAppender(String name, Filter filter, Layout<? extends Serializable> layout, boolean ignoreExceptions,
                            KafkaProducer<String, String> producer, String topic, boolean syncsend) {
        super(name, filter, layout, ignoreExceptions);
        this.producer = producer;
        this.topic = topic;
        this.syncsend = syncsend;
    }

    @PluginFactory
    public static KafkaAppender createAppender(@PluginAttribute("name") final String name,
                                               @PluginElement("Filter") final Filter filter,
                                               @PluginAttribute("ignoreExceptions") final String ignore,
                                               @PluginAttribute("topic") final String topic,
                                               @PluginAttribute("enable") String enable,
                                               @PluginAttribute("syncsend") String syncsend,
                                               @PluginElement("Layout") Layout<? extends Serializable> layout,
                                               @PluginElement("Properties") final Property[] properties) {

        boolean ignoreExceptions = Booleans.parseBoolean(ignore, true);
        boolean enableKafka = Booleans.parseBoolean(enable, true);
        boolean sync = Booleans.parseBoolean(syncsend, false);

        KafkaProducer producer = null;

        Properties props = new Properties();


        props.put("client.id", "log4jProducer");
        props.put("key.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        props.put("value.serializer", "org.apache.kafka.common.serialization.StringSerializer");

        for (Property property : properties) {
            LOGGER.debug("k:{}, v{}",property.getName(), property.getValue());
            props.put(property.getName(), property.getValue());
        }

        if(enableKafka)
            producer = new KafkaProducer<String, String>(props);

        if(layout == null) {
            layout = SerializedLayout.createLayout();
        }

        return new KafkaAppender(name, filter, layout, ignoreExceptions, producer , topic, sync);

    }

    @Override
    public final void stop() {
        super.stop();
        if (producer != null) {
            producer.close();;
        }
    }

    public void append(LogEvent event) {
        try {
            if (producer == null) {
                return;
            }
            String data = getLayout().toSerializable(event).toString().replaceAll("\\s+$","");
            //add location ip
            data = System.getProperty("user.name")+"@"+ InetAddress.getLocalHost().getHostAddress().toString()+" "+data;

            //System.out.println(data);
            if(syncsend)
                producer.send(new ProducerRecord<String, String>(topic,data));
            else
                producer.send(new ProducerRecord<String, String>(topic,data),new LogCallBack(System.currentTimeMillis(), data));

        } catch (final Exception e) {
            e.printStackTrace();
            LOGGER.error("Unable to write to kafka for appender [{}].",  this.getName(), e);
            throw new AppenderLoggingException("Unable to write to kafka in appender: " + e.getMessage(), e);
        } finally {
        }
    }


    class LogCallBack implements Callback {

        private final long startTime;
        private final String message;

        public LogCallBack(long startTime,  String message) {
            this.startTime = startTime;
            this.message = message;
        }

        public void onCompletion(RecordMetadata metadata, Exception exception) {
            long elapsedTime = System.currentTimeMillis() - startTime;
            if (metadata != null) {
                LOGGER.debug(
                        "message(" + message + ") sent to partition(" + metadata.partition() +
                                "), " +
                                "offset(" + metadata.offset() + ") in " + elapsedTime + " ms");
            } else {
                exception.printStackTrace();
            }
        }
    }

}
