package cn.pengh.core.data.req;

import cn.pengh.helper.ClazzHelper;
import cn.pengh.helper.CryptHelper;
import cn.pengh.helper.SignHelper;
import cn.pengh.library.Log;
import cn.pengh.util.DateUtil;
import cn.pengh.util.StringUtil;

import java.util.*;

/**
 * @author Created by pengh
 * @datetime 2023/6/26 08:49
 */
public interface ISignRequest {
    List<Object> EXCEPTS = new ArrayList<Object>() {
        private static final long serialVersionUID = -8364936642243658505L;

        {
            add("sign");
            add("signType");
        }
    };

    String getNonce();

    String getTimestamp();

    String getSignType();

    String getSign();

    default List<Object> excludeFields() {
        return EXCEPTS;
    }

    default Map<Object, Object> replaceFieldsMap() {
        return new HashMap<>();
    }

    default String notNullSignType() {
        return StringUtil.isEmpty(getSignType()) ? SignHelper.SIGN_MD5 : getSignType();
    }

    /**
     * 其中，json转string再转map，效率最低；自写反射还行，如ClazzHelper；内省Introspector实现，效率较高 <br>
     * 子类可重写，推荐使用spring的BeanMap + TreeMap，有优化，自带缓存 <br>
     * 以key排序 <br>
     * 之前使用此，约1.2ms，ClazzHelper.getSortedFieldsVal(this, EXCEPTS, null, true); <br>
     * 改成BeanMap，约0.06ms，约20倍 <br>
     *
     * @return key自然排序，使用=和&拼接的kv字符串
     */
    default String sortedBeforeSign() {
        /*org.springframework.cglib.beans.BeanMap beanMap = org.springframework.cglib.beans.BeanMap.create(this);
        TreeMap<String, Object> map = new TreeMap<String, Object>() {
            private static final long serialVersionUID = 5534585705909129278L;

            {
                Object replacedKey, val;
                Map<String, String> replaceFieldsMap = replaceFieldsMap();
                for (Iterator it = beanMap.keySet().iterator(); it.hasNext(); ) {
                    Object key = it.next();
                    if (excludeFields().contains(key) || (val = beanMap.get(key)) == null) {
                        continue;
                    }
                    key = replaceFieldsMap == null
                            || replaceFieldsMap.size() == 0
                            || (replacedKey = replaceFieldsMap.get(key)) == null ? key : replacedKey;
                    put(key.toString(), val);
                }
            }
        };*/

        Map<Object, Object> map = ClazzHelper.KV(this, excludeFields(), replaceFieldsMap(), true, true);
        StringBuilder sb = new StringBuilder();
        int len1 = map.size() - 1, i = 0;
        for (Map.Entry<Object, Object> en : map.entrySet()) {
            sb.append(en.getKey()).append("=").append(en.getValue());
            if (i++ != len1) {
                sb.append("&");
            }
        }
        return sb.toString();
    }

    default String sign(String key) {
        return sign(sortedBeforeSign(), key, notNullSignType());
    }

    default String sign(String key, String signType) {
        return sign(sortedBeforeSign(), key, signType);
    }

    default String sign(String beforeSignStr, String key, String signType) {
        if (SignHelper.SIGN_RSA2.equals(signType.toUpperCase())) {
            return CryptHelper.sha256WithRsa(beforeSignStr, key);
        }
        return signWithHash(beforeSignStr, key, signType);
    }

    default boolean validSign(String key) {
        return validSign(notNullSignType(), key);
    }

    default boolean validSign(String signType, String key) {
        return validSign2(signType, key, null);
    }

    // 兼容新旧key并存期间缓存生效问题 2023.6.8
    default boolean validSign2(String key1, String key2) {
        return validSign2(notNullSignType(), key1, key2);
    }

    // 兼容新旧key并存期间缓存生效问题 2023.6.8
    // 建议一般key1为新
    default boolean validSign2(String signType, String key1, String key2) {
        String signStr = getSign();
        //Log.getSlf4jLogger().trace("{},{},{}", signStr, signType, getNonce());
        if (StringUtil.isEmpty(signStr) || StringUtil.isEmpty(signType) || StringUtil.isEmpty(getNonce())) {
            return false;
        }

        if (StringUtil.isEmpty(key1)) {
            return StringUtil.isNotEmpty(key2) && validSignWithHash(sortedBeforeSign(), key2, signType, signStr);
        }
        String beforeSignStr;
        return validSignWithHash((beforeSignStr = sortedBeforeSign()), key1, signType, signStr)
                || (StringUtil.isNotEmpty(key2) && validSignWithHash(beforeSignStr, key2, signType, signStr));
    }

    default boolean validSignWithTime(String signType, String key, int minutes) {
        return validSignWithTime2(signType, key, null, minutes);
    }

    default boolean validSignWithTime2(String signType, String key1, String key2, int minutes) {
        if (!validSign2(signType, key1, key2)) {
            return false;
        }

        //校验timestamp，比如 时间前后浮动30分钟
        //10位的加0、16位或其他位的忽略

        long millis = minutes * 1000 * 60, curr = System.currentTimeMillis(), time = Long.valueOf(getTimestamp().length() == 13 ? getTimestamp() : getTimestamp() + "000");

        boolean ok = !(time + millis < curr || curr + millis < time);
        if (!ok) {
            Log.getSlf4jLogger().trace("{} minutes time valid failed. req time: {}", minutes, DateUtil.getLocalDateTimeByUnixLong(time / 1000));
        }
        return ok;
    }

    default boolean validSignWithTime(String key, int minutes) {
        return validSignWithTime(notNullSignType(), key, minutes);
    }

    default boolean validSignWithTime2(String key1, String key2, int minutes) {
        return validSignWithTime2(notNullSignType(), key1, key2, minutes);
    }

    /**
     * 单向HASH加密
     *
     * @param beforeSignStr
     * @param key
     * @param signType      不区分大小写，目前仅支持
     * @return
     */
    static String signWithHash(String beforeSignStr, String key, String signType) {
        signType = signType.toUpperCase();
        String strWithKey = withHashKey(beforeSignStr, key);
        if (SignHelper.SIGN_SHA256.equals(signType)) {
            return CryptHelper.sha256(strWithKey);
        } else if (SignHelper.SIGN_MD5.equals(signType)) {
            return CryptHelper.md5(strWithKey);
        } else if (SignHelper.SIGN_SHA512.equals(signType)) {
            return CryptHelper.sha512(strWithKey);
        } else if (SignHelper.SIGN_SHA384.equals(signType)) {
            return CryptHelper.sha384(strWithKey);
        } else if (SignHelper.SIGN_SHA.equals(signType)) {
            return CryptHelper.sha128(strWithKey);
        }
        return null;
    }

    static boolean validSignWithHash(String beforeSignStr, String key, String signType, String signStr) {
        if (SignHelper.SIGN_RSA2.equals(signType.toUpperCase())) {
            return SignHelper.verifySHA256WithRSA(beforeSignStr, key, signStr);
        }
        return validSignWithHash(signWithHash(beforeSignStr, key, signType), signStr);
    }

    static boolean validSignWithHash(String validSign, String signStr) {
        return validSign != null && validSign.equals(signStr) || validSign.equals(signStr.toLowerCase());
    }

    static String withHashKey(String beforeSignStr, String key) {
        Log.getSlf4jLogger().trace("{}&key={}", beforeSignStr, StringUtil.addMaskStarPre(key));
        return beforeSignStr + "&key=" + key;
    }
}
