package cn.pengh.pbase.core;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

/**
 * 参考使用ConcurrentHashMap实现高效缓存框架
 * https://blog.csdn.net/zxfryp909012366/article/details/82881217
 * @author Created by pengh
 * @datetime 2018/12/29 21:03
 */
public abstract class CacheComputable<K,V> {
    protected ConcurrentHashMap<K, Future<V>> cache = new ConcurrentHashMap<>();
    public abstract V compute(K k);

    /**
     * 这里通过调用ConcurrentHashMap::putIfAbsent方法
     * 假设两个线程传入了同样的参数，并且都创建了一个FutureTask对象，
     * 一个线程获得了cache的执行权限执行了cache::putIfAbsent()方法，并且返回了一个null到局部变量f中，
     * 此时另一个线程也会调用cache::putIfAbsent()方法，由于第一个线程已经将相关键值对存入到cache中了，
     * 那么第二个线程将获得第一个线程创建的FutureTask对象，并且将其替换给当前线程中的局部变量f，并且其判断f不为null，那么其会调用f::get()方法，
     * 而此时第一个线程正在执行FutureTask::run方法，如果其已经计算完成，那么其会返回结果给第一个线程，而第二个线程是直接执行的FutureTask::get方法，
     * 如果第一个线程执行完成，那么第二个线程将直接获取结果返回，如果第一个线程没有执行完成，那么第二个线程将等待第一个线程执行完成后再返回结果。
     *
     * 这里对compute方法使用while循环的目的是，
     * 当某个线程在执行结果的时候，其余的线程需要等待该线程执行完成，
     * 如果其余的线程由于某些原因等待被打断，那么通过while循环其会继续进入等待状态，从而得到执行结果。
     * 而对于某个执行计算结果的线程而言，如果其计算过程被取消或失败，那么对于缓存而言这就造成了缓存污染，
     * 因为存入的是FutureTask对象，而不是真正的结果，如果计算的线程被取消，那么实际上FutureTask::get方法将一直无法获取到值，但是cache中对应键的值也不是null的，这就是缓存污染的根源。
     * 代码中通过调用FutureTask::get方法时捕获CancellationException来捕捉执行计算的线程计算被取消或失败的情况，
     * 当出现这种情况的时候就从cache中将相应的FutureTask对象给移除掉，并且通过while循环也可以是后来进入的线程再次执行run方法从而得到计算结果
     *
     * 同步阻塞同一时间所有线程
     * @param k
     * @return
     */
    public V run(K k) {
        while (true) {
            Future<V> f = cache.get(k);

            if (f == null) {
                FutureTask<V> ft = new FutureTask<>(() -> { return compute(k);});
                f = cache.putIfAbsent(k, ft);

                if (f == null) {
                    f = ft;
                    ft.run();
                }
            }

            try {
                return f.get();
            } catch (CancellationException e) {
                cache.remove(k, f);
            } catch (Exception e) {
                e.printStackTrace();
                return null;
            }
        }
    }
}