package cn.pengh.crypt;

import cn.pengh.exception.CustomException;
import org.apache.commons.codec.binary.Base64;

import javax.crypto.Cipher;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * need jdk1.8+
 * @author Created by pengh
 * @datetime 2019-03-27 09:50
 */
public class RSA {
    private static final String KEY_PAIR_RSA = "RSA";
    private static final String SIGNATURE_MD5 = "MD5WithRSA";
    private static final String SIGNATURE_SHA2 = "SHA256withRSA";
    private static final int KEY_LENGTH = 2048;
    private static final int KEY_LENGTH_MIN = 1024;
    private static final String CHARSET = StandardCharsets.UTF_8.name();

    public static KeyPair generateKeyPair() throws Exception {
        return generateKeyPair(KEY_LENGTH);
    }

    /**
     * 获取公私密钥对数组串
     * {pub, pri}
     * @return
     */
    public static String[] generateKey() {
        try {
            KeyPair pair = generateKeyPair();
            String pub = base64Encode(pair.getPublic().getEncoded());
            String pri = base64Encode(pair.getPrivate().getEncoded());

            return new String[]{pub, pri};
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    public static KeyPair generateKeyPair(int n) throws Exception {
        if (n < KEY_LENGTH_MIN) {
            throw CustomException.create("RSA模值的长度需大于"+KEY_LENGTH_MIN+"位，建议"+KEY_LENGTH);
        }

        KeyPairGenerator generator = KeyPairGenerator.getInstance(KEY_PAIR_RSA);
        generator.initialize(n, new SecureRandom());

        return generator.generateKeyPair();
    }

    /*public static KeyPair getKeyPairFromKeyStore() throws Exception {
        //Generated with:
        //  keytool -genkeypair -alias mykey -storepass s3cr3t -keypass s3cr3t -keyalg RSA -keystore keystore.jks

        InputStream ins = Rsa.class.getResourceAsStream("/keystore.jks");

        KeyStore keyStore = KeyStore.getInstance("JCEKS");
        keyStore.load(ins, "s3cr3t".toCharArray());   //Keystore password
        KeyStore.PasswordProtection keyPassword =       //Key password
                new KeyStore.PasswordProtection("s3cr3t".toCharArray());

        KeyStore.PrivateKeyEntry privateKeyEntry = (KeyStore.PrivateKeyEntry) keyStore.getEntry("mykey", keyPassword);

        java.security.cert.Certificate cert = keyStore.getCertificate("mykey");
        PublicKey publicKey = cert.getPublicKey();
        PrivateKey privateKey = privateKeyEntry.getPrivateKey();

        return new KeyPair(publicKey, privateKey);
    }*/

    /**
     * 公钥加密
     * @param plainText 明文
     * @param publicKey
     * @return
     * @throws Exception
     */
    public static String encrypt(String plainText, PublicKey publicKey) throws Exception {
        return encrypt(plainText, publicKey, CHARSET);
    }
    public static String encrypt(String plainText, PublicKey publicKey, String charset) throws Exception {
        Cipher encryptCipher = Cipher.getInstance(KEY_PAIR_RSA);
        encryptCipher.init(Cipher.ENCRYPT_MODE, publicKey);

        byte[] cipherText = encryptCipher.doFinal(plainText.getBytes(charset));

        return base64Encode(cipherText);
    }

    /**
     * 私钥解密
     * @param cipherText 密文
     * @param privateKey
     * @return
     * @throws Exception
     */
    public static String decrypt(String cipherText, PrivateKey privateKey) throws Exception {
        return decrypt(cipherText, privateKey, CHARSET);
    }
    public static String decrypt(String cipherText, PrivateKey privateKey, String charset) throws Exception {
        byte[] bytes = base64Decode(cipherText);

        Cipher decriptCipher = Cipher.getInstance(KEY_PAIR_RSA);
        decriptCipher.init(Cipher.DECRYPT_MODE, privateKey);

        return new String(decriptCipher.doFinal(bytes), charset);
    }

    /**
     * 签名，sha256-rsa
     *
     * @param plainText 明文
     * @param privateKey 私钥
     * @return
     * @throws Exception
     */
    public static String sign(String plainText, String privateKey) {
        return signStr(plainText, privateKey, SIGNATURE_SHA2);
    }

    public static String sign(String plainText, String privateKey, String charset) {
        return signStr(plainText, privateKey, SIGNATURE_SHA2, charset);
    }

    public static String sign(String plainText, PrivateKey privateKey) throws Exception {
        return signIt(plainText, privateKey, SIGNATURE_SHA2);
    }

    public static String signWithMD5(String plainText, String privateKey) {
        return signStr(plainText, privateKey, SIGNATURE_MD5);
    }

    public static String signWithMD5(String plainText, PrivateKey privateKey) throws Exception {
        return signIt(plainText, privateKey, SIGNATURE_MD5);
    }

    /**
     * 公钥验证签名
     * @param plainText 明文
     * @param signature 签名串
     * @param publicKey 公钥
     * @return
     */
    public static boolean verify(String plainText, String signature, String publicKey) {
        return verifyStr(plainText, signature, publicKey, SIGNATURE_SHA2);
    }

    public static boolean verify(String plainText, String signature, String publicKey, String charset) {
        return verifyStr(plainText, signature, publicKey, SIGNATURE_SHA2, charset);
    }

    public static boolean verify(String plainText, String signature, PublicKey publicKey) throws Exception {
        return verifyIt(plainText, signature, publicKey, SIGNATURE_SHA2);
    }
    public static boolean verifyWithMD5(String plainText, String signature, PublicKey publicKey) throws Exception {
        return verifyIt(plainText, signature, publicKey, SIGNATURE_MD5);
    }
    public static boolean verifyWithMD5(String plainText, String signature, String publicKey) {
        return verifyStr(plainText, signature, publicKey, SIGNATURE_MD5);
    }


    /**
     * 签名，可选类型有SHA256withRSA、MD5WithRSA等
     * @param plainText
     * @param privateKey
     * @param signatureTp
     * @return
     */
    public static String signStr(String plainText, String privateKey, String signatureTp) {
        return signStr(plainText, privateKey, signatureTp, CHARSET);
    }
    public static String signStr(String plainText, String privateKey, String signatureTp, String charset) {
        try {
            return signIt(plainText, KeyFactory.getInstance(KEY_PAIR_RSA).generatePrivate(new PKCS8EncodedKeySpec(base64Decode(privateKey))), signatureTp, charset);
        } catch (Exception e){
            e.printStackTrace();
            return null;
        }

    }

    /**
     * 验证签名，可选类型有SHA256withRSA、MD5WithRSA等
     * @param plainText
     * @param signature
     * @param publicKey
     * @param signatureTp
     * @return
     */
    public static boolean verifyStr(String plainText, String signature, String publicKey, String signatureTp) {
        return verifyStr(plainText, signature, publicKey, signatureTp, CHARSET);
    }

    public static boolean verifyStr(String plainText, String signature, String publicKey, String signatureTp, String charset) {
        try {
            return verifyIt(plainText, signature, KeyFactory.getInstance(KEY_PAIR_RSA).generatePublic(new X509EncodedKeySpec(base64Decode(publicKey))), signatureTp, charset);
        } catch (Exception e){
            e.printStackTrace();
            return false;
        }

    }

    private static String signIt(String plainText, PrivateKey privateKey, String signatureTp) throws Exception {
        return signIt(plainText, privateKey, signatureTp, CHARSET);
    }
    //signatureTp in MD5WithRSA,SHA256withRSA
    private static String signIt(String plainText, PrivateKey privateKey, String signatureTp, String charset) throws Exception {
        Signature privateSignature = Signature.getInstance(signatureTp);
        privateSignature.initSign(privateKey);
        privateSignature.update(plainText.getBytes(charset));

        byte[] signature = privateSignature.sign();

        return base64Encode(signature);
        //return Base64.getEncoder().encodeToString(signature);
    }
    private static boolean verifyIt(String plainText, String signature, PublicKey publicKey, String signatureTp) throws Exception {
        return verifyIt(plainText, signature, publicKey, signatureTp, CHARSET);
    }
    private static boolean verifyIt(String plainText, String signature, PublicKey publicKey, String signatureTp, String charset) throws Exception {
        Signature publicSignature = Signature.getInstance(signatureTp);
        publicSignature.initVerify(publicKey);
        publicSignature.update(plainText.getBytes(charset));

        byte[] signatureBytes = base64Decode(signature);

        return publicSignature.verify(signatureBytes);
    }

    private static String base64Encode(byte[] data) {
        //return java.util.Base64.getEncoder().encodeToString(data);
        return Base64.encodeBase64String(data);
        //return Base64.encode(data);
    }
    private static byte[] base64Decode(String src) {
        //return java.util.Base64.getDecoder().decode(src);
        return Base64.decodeBase64(src);
        //return Base64.decode(src);
    }
}