package cn.pengh.core.concurrent;

import cn.pengh.util.CurrencyUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author Created by pengh
 * @datetime 2020/6/17 11:00
 */
public class ExecutorUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(ExecutorUtil.class);

    public static final int POOL_SIZE = CurrencyUtil.multiply(Runtime.getRuntime().availableProcessors(), 1.2).intValue() + 1;
    public static final int TIMEOUT = 3000; //ms
    public static final int QUEUE_CAPACITY = 500000; //50w


    public static ThreadPoolExecutor getExecutor() {
        return getExecutor(POOL_SIZE);
    }

    public static ThreadPoolExecutor getExecutor(int coreSize) {
        return getExecutor(coreSize, QUEUE_CAPACITY, "TASK");
    }

    public static ThreadPoolExecutor getExecutor(int queueCapacity, String threadName) {
        return getExecutor(POOL_SIZE, queueCapacity, threadName);
    }

    public static ThreadPoolExecutor getExecutor(String threadName) {
        return getExecutor(QUEUE_CAPACITY, threadName);
    }

    public static ThreadPoolExecutor getExecutor(int coreSize, int queueCapacity, String threadName) {
        return getExecutor(coreSize, queueCapacity, TIMEOUT, threadName);
    }

    // https://www.cnblogs.com/wjqhuaxia/p/11762150.html
    // 类似ActiveMq，最大努力执行任务型
    // 当触发拒绝策略时，在尝试N ms时间重新将任务塞进任务队列，超时还没成功时，就抛出异常
    public static ThreadPoolExecutor getExecutor(int coreSize, int queueCapacity, int timeout, String threadName) {
        return new ThreadPoolExecutor(coreSize, coreSize, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(queueCapacity), NamedThreadFactory.create(threadName), (r, e) -> {
            try {
                if (e.getQueue().offer(r, timeout, TimeUnit.MILLISECONDS)) {
                    return;
                }
            } catch (InterruptedException e1) {
                throw new RejectedExecutionException("Interrupted waiting for BrokerService.worker");
            }
            String msg = String.format("Thread pool is EXHAUSTED!" +
                            " Thread Name: %s, Pool Size: %d (active: %d, core: %d, max: %d, largest: %d), Task: %d (completed: %d)," +
                            " Executor status:(isShutdown:%s, isTerminated:%s, isTerminating:%s)",
                    threadName, e.getPoolSize(), e.getActiveCount(), e.getCorePoolSize(), e.getMaximumPoolSize(), e.getLargestPoolSize(),
                    e.getTaskCount(), e.getCompletedTaskCount(), e.isShutdown(), e.isTerminated(), e.isTerminating());
            LOGGER.warn(msg);
            throw new RejectedExecutionException("Timed Out " + timeout + "ms while attempting to enqueue Task.");
        });
    }
}
