package cn.pengh.http;

import cn.pengh.helper.ClazzHelper;
import cn.pengh.util.CurrencyUtil;
import okhttp3.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 依赖 "com.squareup.okhttp3:okhttp:$okhttp_version",
 *
 * @author Created by pengh
 * @datetime 2020/6/12 11:02
 */
public class OkHttpUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(OkHttpUtil.class);
    private static final String CHARSET = "utf-8";
    private static final String MEDIA_TYPE_JSON = "application/json";
    private static final String MEDIA_TYPE_FORM = "application/x-www-form-urlencoded";
    private static final MediaType CONTENT_TYPE_JSON_UTF8 = MediaType.parse(MEDIA_TYPE_JSON + "; charset=" + CHARSET);
    private static final MediaType CONTENT_TYPE_FORM_UTF8 = MediaType.parse(MEDIA_TYPE_FORM + "; charset=" + CHARSET);

    private static Map<Integer, OkHttpClient> okHttpClientMap = new ConcurrentHashMap<>();
    private static final int INSTANCE_MAX = 10; //本地map里面最多10个okHttpClient实例，当大于则清空所有
    private static final int TIMEOUT = 5000; //默认请求超时5s
    private static final int CONNECT_TIMEOUT_FACTOR = 5; //连接超时，一般 == 超时/5，最低100ms
    private static final int CONNECT_TIMEOUT_MIN = 100; //连接超时，最低100ms

    private static class OkHttpClientBuilderLazyHolder {
        private static final OkHttpClient.Builder INSTANCE = new OkHttpClient.Builder();
    }

    private static OkHttpClient getInstance(int timeout) {
        OkHttpClient okHttpClient = okHttpClientMap.get(timeout);
        int connectTimeout = timeout / CONNECT_TIMEOUT_FACTOR; //连接超时，一般 == 超时/5，最低100ms
        connectTimeout = connectTimeout == 0 ? CONNECT_TIMEOUT_MIN : connectTimeout;

        if (okHttpClient == null) {
            if (okHttpClientMap.size() > INSTANCE_MAX) {
                okHttpClientMap.clear();
            }
            OkHttpClientBuilderLazyHolder.INSTANCE
                    .retryOnConnectionFailure(true) //默认true，connect失败时重试一次
                    .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
                    .readTimeout(timeout, TimeUnit.MILLISECONDS)
                    .writeTimeout(timeout, TimeUnit.MILLISECONDS);

            okHttpClient = OkHttpClientBuilderLazyHolder.INSTANCE.build();
            okHttpClientMap.put(timeout, okHttpClient);
        }
        return okHttpClient;
    }

    public static String post(String url, Object obj) {
        return post(url, obj, null);
    }

    public static String post(String url, Object obj, int timeout) {
        return post(url, obj, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String post(String url, Object obj, HttpRequest.HttpRequestConfig config) {
        return postFormMap(url, ClazzHelper.KV(obj), config);
    }

    public static String post(String url, Map<String, String> param) {
        return post(url, param, TIMEOUT);
    }

    public static String post(String url, Map<String, String> param, int timeout) {
        return post(url, param, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String get(String url) {
        return get(url, HttpRequest.HttpRequestConfig.createDefault());
    }

    public static String postJson(String url, String json) {
        return postJson(url, json, TIMEOUT);
    }

    public static String postJson(String url, String json, int timeout) {
        return postJson(url, json, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String postJson(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return post(url, json, getMediaTypeJson(config), config);
    }

    public static String post(String url, Map<String, String> param, HttpRequest.HttpRequestConfig config) {
        return postFormMap(url, ClazzHelper.KV(param), config);
    }

    public static String postFormMap(String url, Map<Object, Object> param, HttpRequest.HttpRequestConfig config) {
        //LOGGER.debug("Params: {}", param);
        /*FormBody.Builder formBodyBuilder = new FormBody.Builder(Charset.forName(config == null ? "utf-8" : config.getCharset()));
        param.forEach((k, v) -> formBodyBuilder.add(k, v));
        return doHttp(new Request.Builder().url(url).post(formBodyBuilder.build()), config);*/
        //解决中文乱码问题
        StringBuilder sb = new StringBuilder();
        /*param.forEach((k, v) -> sb.append(k).append("=").append(v).append("&"));
        String body = sb.toString().replaceAll("(.*)&$", "$1");
        LOGGER.debug("Params: {}", body);*/

        FormBody.Builder formBodyBuilder = new FormBody.Builder(Charset.forName(config == null ? CHARSET : config.getCharset()));
        try {
            param.forEach((k, v) -> formBodyBuilder.add(k.toString(), v.toString()));
        } catch (Exception e) {
            e.printStackTrace();
        }

        FormBody fb = formBodyBuilder.build();
        for (int i = 0; i < fb.size(); i++) {
            sb.append(fb.encodedName(i)).append("=").append(fb.encodedValue(i)).append("&");
        }
        String body = sb.toString().replaceAll("(.*)&$", "$1");
        LOGGER.debug("Params: {}", body);
        return post(url, body, getMediaTypeForm(config), config);
    }

    private static MediaType getMediaTypeForm(HttpRequest.HttpRequestConfig config) {
        return config == null || CHARSET.equalsIgnoreCase(config.getCharset()) ? CONTENT_TYPE_FORM_UTF8 : MediaType.parse(MEDIA_TYPE_FORM + "; charset=" + config.getCharset());
    }

    private static MediaType getMediaTypeJson(HttpRequest.HttpRequestConfig config) {
        return config == null || CHARSET.equalsIgnoreCase(config.getCharset()) ? CONTENT_TYPE_JSON_UTF8 : MediaType.parse(MEDIA_TYPE_JSON + "; charset=" + config.getCharset());
    }


    public static String putJson(String url, String json) {
        return putJson(url, json, null);
    }

    public static String putJson(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return doHttp(new Request.Builder().url(url).put(RequestBody.create(getMediaTypeJson(config), json)), config);
    }

    public static String post(String url, String body, MediaType mediaType, HttpRequest.HttpRequestConfig config) {
        return doHttp(new Request.Builder().url(url).post(RequestBody.create(mediaType, body)), config);
    }

    public static String get(String url, HttpRequest.HttpRequestConfig config) {
        return doHttp(new Request.Builder().url(url).get(), config);
    }

    public static String get(String url, Map<String, String> headers) {
        return get(url, HttpRequest.HttpRequestConfig.createDefault().setHeaders(headers));
    }


    private static String doHttp(Request.Builder requestBuilder, HttpRequest.HttpRequestConfig config) {
        long start = System.nanoTime();

        int timeout = TIMEOUT;
        if (config != null) {
            timeout = config.getTimeout();
            Map<String, String> headers = config.getHeaders();
            if (headers != null) {
                headers.forEach((k, v) -> requestBuilder.addHeader(k, v));
            }
        }

        Request request = requestBuilder.build();
        LOGGER.debug("{} {} {}. Timeout {}ms", request.method(), request.url(), request.body() == null ? "" : request.body().contentType(), timeout);
        //LOGGER.debug("Params: {}", JsonUtil.toJSONString(request.body()));
        try {
            Response response = getInstance(timeout).newCall(request).execute();
            String res = response.body().string();
            LOGGER.debug("{}", response.headers().toString().replace("\n", " "));
            LOGGER.debug("Response {}: {}", response.code(), res);
            return res;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            LOGGER.debug("Time elapsed: {}s", CurrencyUtil.divide(System.nanoTime() - start, 1e9, 6));
        }

    }
}
