package cn.pengh.helper;

import cn.pengh.crypt.*;
import cn.pengh.crypt.asymmetric.AbstractAsymmetricEncryptor;
import cn.pengh.crypt.asymmetric.AsymmetricFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;


/**
 * Default CHARSET = "UTF-8";
 * base64依赖 commons-codec:commons-codec:$commons_codec_version
 *
 * @author Created by pengh
 * @datetime 2021/10/9 09:57
 */
public class CryptHelper {
    /**
     * 推荐使用自带salt的bcrypt，以替代md5、sha256等
     *
     * @param plainText
     * @return
     */
    public static String bcrypt(String plainText) {
        return BCrypt.hashpw(plainText, BCrypt.gensalt());
    }

    public static boolean bcryptVerify(String plainText, String cipherText) {
        return BCrypt.checkpw(plainText, cipherText);
    }

    /*
    推荐使用 sha256及以上
     */
    public static String md5(String plainText) {
        return Md5.MD5Encode(plainText);
    }

    // 使用guava
    public static String md5File(File file) {
        com.google.common.io.ByteSource byteSource = com.google.common.io.Files.asByteSource(file);
        try {
            return byteSource.hash(com.google.common.hash.Hashing.md5()).toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    // 使用commons.codec
    public static String md5File(String filePath) {
        try (FileInputStream fis = new FileInputStream(filePath)) {
            return org.apache.commons.codec.digest.DigestUtils.md5Hex(fis);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static String sha256File(File file) {
        return file == null ? null : sha256File(file.getAbsolutePath());
    }


    public static String sha256File(String filePath) {
        try (FileInputStream fis = new FileInputStream(filePath)) {
            return org.apache.commons.codec.digest.DigestUtils.sha256Hex(fis);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /*
    推荐使用 sha256及以上
     */
    @Deprecated
    public static String sha128(String plainText) {
        return SHA.shaEncrypt(plainText);
    }

    public static String sha256(String plainText) {
        return SHA.sha256Encrypt(plainText);
    }

    public static String sha384(String plainText) {
        return SHA.sha384Encrypt(plainText);
    }

    public static String sha512(String plainText) {
        return SHA.sha512Encrypt(plainText);
    }

    /*
    同sha256WithRsa
    ssh-keygen -t rsa -f ~/data/test.pem -b 2048 -m PKCS8 -N ''
    openssl rsa -pubout -in ~/data/test.pem -out ~/data/tmp.pub

    cat ~/data/test.pem
    cat ~/data/tmp.pub
     */
    public static String rsa2(String plainText, String publicKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.RSA2).encrypt(plainText, publicKey);
    }

    public static String rsa2Decrypt(String cipherText, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.RSA2).decrypt(cipherText, privateKey);
    }

    public static String sha256WithRsaOAEP(String plainText, String publicKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA_OAEP).encrypt(plainText, publicKey);
    }

    public static String sha256WithRsaOAEPDecrypt(String cipherText, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA_OAEP).decrypt(cipherText, privateKey);
    }

    public static String sha256WithRsa(String plainText, String publicKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA).encrypt(plainText, publicKey);
    }

    public static String sha256WithRsaDecrypt(String cipherText, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA256RSA).decrypt(cipherText, privateKey);
    }

    public static String sha512WithRsa(String plainText, String publicKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA512RSA).encrypt(plainText, publicKey);
    }

    public static String sha512WithRsaDecrypt(String cipherText, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA512RSA).decrypt(cipherText, privateKey);
    }

    public static String sha512WithRsaOAEP(String plainText, String publicKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA512RSA_OAEP).encrypt(plainText, publicKey);
    }

    public static String sha512WithRsaOAEPDecrypt(String cipherText, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA512RSA_OAEP).decrypt(cipherText, privateKey);
    }

    public static String[] genRsa2PriAndPubKey() {
        return AbstractAsymmetricEncryptor.genRsaPriAndPubKey(2048);
    }



    @Deprecated
    /*
    推荐使用 sha256WithRsa2或者rsa2
     */
    public static String sha128WithRsa(String plainText, String publicKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA128RSA).encrypt(plainText, publicKey);
    }

    @Deprecated
    /*
    推荐使用 sha256WithRsa2Decrypt
     */
    public static String sha128WithRsaDecrypt(String cipherText, String privateKey) {
        return AsymmetricFactory.getByName(AsymmetricFactory.Algorithm.SHA128RSA).decrypt(cipherText, privateKey);
    }

    public static int murmurhash32(String plainText) {
        return com.google.common.hash.Hashing.murmur3_32_fixed().hashString(plainText, StandardCharsets.UTF_8).asInt();
    }

    public static int murmurhash32Unsigned(String plainText) {
        int value = com.google.common.hash.Hashing.murmur3_32_fixed().hashString(plainText, StandardCharsets.UTF_8).asInt();
        return value >= 0 ? value : value & Integer.MAX_VALUE;
    }

    /**
     * 有负数
     *
     * @param plainText
     * @return
     */
    public static long murmurhash128(String plainText) {
        return com.google.common.hash.Hashing.murmur3_128().hashString(plainText, StandardCharsets.UTF_8).asLong();
    }

    /**
     * 转成无符号
     *
     * @param plainText
     * @return
     */
    public static long murmurhash128Unsigned(String plainText) {
        long value = com.google.common.hash.Hashing.murmur3_128().hashString(plainText, StandardCharsets.UTF_8).asLong();
        return value >= 0 ? value : value & Long.MAX_VALUE;
    }

    // 一般用于替代MD5
    public static String murmurhash128Str(String plainText) {
        return com.google.common.hash.Hashing.murmur3_128().hashString(plainText, StandardCharsets.UTF_8).toString();
    }

    public static String base62(String plainText) {
        return new String(Base62.createInstance().encode(plainText.getBytes(StandardCharsets.UTF_8)), StandardCharsets.UTF_8);
    }

    public static String murmurhash128Encode62(String plainText) {
        return encode62(murmurhash128Unsigned(plainText));
    }

    public static String murmurhash32Encode62(String plainText) {
        return encode62(murmurhash32Unsigned(plainText));
    }

    public static String encode62(long n) {
        return Encode10.encode62(n);
    }

    public static String encode63(long n) {
        return Encode10.encode63(n);
    }

    public static String encode64(long n) {
        return Encode10.encode64(n);
    }
}
