package cn.pengh.util;

import cn.pengh.library.Log;
import cn.pengh.pbase.core.CacheComputable;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.basic.DateConverter;
import com.thoughtworks.xstream.converters.javabean.JavaBeanConverter;
import com.thoughtworks.xstream.core.util.QuickWriter;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.io.naming.NoNameCoder;
import com.thoughtworks.xstream.io.xml.DomDriver;
import com.thoughtworks.xstream.io.xml.PrettyPrintWriter;
import com.thoughtworks.xstream.io.xml.XppDriver;

import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;

/**
 * 单例
 * 缓存、FutureTask、ConcurrentHashMap、putIfAbsent
 * @author pengh
 * @Date 2016年9月28日 下午5:56:02
 */
public class XstreamUtil {
	private transient XStream xstream;
	private transient XStream xstreamCDATA;
    private transient XStreamUnMarshal xStreamUnMarshalCache;

	private XstreamUtil() {
        Log.debug("--XStreamUtil singleton init...");
        initXStream();

        //初始化 xml反序列化成java bean的xstream 缓存实例
        xStreamUnMarshalCache = new XStreamUnMarshal();
    }

	//初始化 java序列化成xml的xstream实例
	private void initXStream() {
		xstream = new XStream(new XppDriver(new NoNameCoder()));
		after(xstream);

		xstreamCDATA = new XStream(new XppDriver(new NoNameCoder()) {
			@Override
			public HierarchicalStreamWriter createWriter(Writer out) {
				return new PrettyPrintWriter(out) {
					@Override
					public void startNode(String name, Class clazz) {
						super.startNode(name, clazz);
					}
					@Override
					public String encodeNode(String name) {
						return name;
					}
					@Override
					protected void writeText(QuickWriter writer, String text) {
						writer.write("<![CDATA[");
						writer.write(text);
						writer.write("]]>");
					}
				};
			}
		});
		after(xstreamCDATA);
	}
	private void after(XStream xstream) {
		//xstream.registerConverter(new JavaBeanConverter(xstream.getMapper()), XStream.PRIORITY_VERY_LOW);
		xstream.registerConverter(new DateConverter(TimeZone.getTimeZone("GMT+8")), XStream.PRIORITY_NORMAL);
		xstream.autodetectAnnotations(true);
		xstream.aliasSystemAttribute(null, "class");//去掉 class 属性
	}

	//初始化 xml反序列化成java bean的xstream实例
	private XStream initXStreamUnmarshal() {
        XStream xstreamUnmarshal =  new XStream(new DomDriver());
		XStream.setupDefaultSecurity(xstreamUnmarshal);
		xstreamUnmarshal.ignoreUnknownElements();
		return xstreamUnmarshal;
	}

	private static class LazyHolder {
		private static XstreamUtil INSTANCE = new XstreamUtil();
	}
	public static XstreamUtil getInstance() {
		return LazyHolder.INSTANCE;
	}


	public static <T> T fromXML(String xml, Class<T> clazz) {
        return (T)getInstance().xStreamUnMarshalCache.run(clazz).fromXML(xml);
	}

	private class XStreamUnMarshal extends CacheComputable<Class<?>, XStream> {
        @Override
        public XStream compute(Class<?> clazz) {
            Log.getLogger().debug("--XStreamUtil unMarshal cache init <{}>...", clazz.getName());
            XStream x = getInstance().initXStreamUnmarshal();
            x.allowTypes(withDeclaredClasses(clazz));
            x.processAnnotations(clazz);

            /*if (cache.size() > 102400) {
                cache.clear();
            }*/
            return x;
        }
    }

	public static String toXml(Object object){
		return toXml(object,false);
	}
	
	public static String toXml(Object object, final boolean needCDATA){
		return toXml(object,needCDATA,true,null,false);
	}

	public static String toXml(Object object, String clazzAlias){
		return toXml(object,false,false,clazzAlias,false);
	}
	public static String toXml(Object object, final boolean needCDATA,final boolean pretty,String clazzAlias) {
		return toXml(object, needCDATA, pretty, clazzAlias, false);
	}
    public static String toXml(Object object, final boolean needCDATA,String clazzAlias) {
        return toXml(object, needCDATA, false, clazzAlias, false);
    }
    public static String toXml(Object object, final boolean needCDATA,final boolean pretty,String clazzAlias,boolean getter) {
        return toXml(needCDATA ? getInstance().xstreamCDATA : getInstance().xstream, object, pretty, clazzAlias, getter);
    }

	private static String toXml(XStream xstream, Object object,final boolean pretty,String clazzAlias,boolean getter) {
		//https://github.com/x-stream/xstream/blob/master/xstream/src/test/com/thoughtworks/xstream/converters/javabean/JavaBeanConverterTest.java
		if (getter)
			xstream.registerConverter(new JavaBeanConverter(xstream.getMapper()), XStream.PRIORITY_VERY_LOW);

		if(StringUtil.isNotEmpty(clazzAlias))
			xstream.alias(clazzAlias, object.getClass());//以@XStreamAlias("clazzAlias")为准
		return pretty ? xstream.toXML(object) : xstream.toXML(object).replaceAll(">\\s+<", "><"); 
	}

	private static Class[] withDeclaredClasses(Class<?> clazz) {
		List<Class<?>> list = new ArrayList<Class<?>>(){
			private static final long serialVersionUID = 6042388860779880244L; {
				add(clazz);
				for(Class<?> s: clazz.getDeclaredClasses()) {
					add(s);
				}
			}
		};
		return list.toArray(new Class[list.size()]);
	}

}




