package cn.pengh.util;

import cn.pengh.exception.CustomException;
import cn.pengh.io.json.MaskDataConfig;
import cn.pengh.io.json.MaskJacksonSerializer;
import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ClassUtils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.List;

/**
 * 依次适配，Jackson、FastJson、Gson
 *
 * @author Created by pengh
 * @datetime 2021/6/4 10:59
 */
public class JsonUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(JsonUtil.class);
    private static final String CLAZZ_JACKSON = "com.fasterxml.jackson.databind.ObjectMapper";
    private static final String CLAZZ_FASTJSON = "com.alibaba.fastjson.JSON";
    private static final String CLAZZ_GSON = "com.google.gson.Gson";

    // 依赖spring-core的ClassUtils来判断
    private static boolean hasJsonClazz(String clazzName) {
        try {
            return ClassUtils.isPresent(clazzName, JsonUtil.class.getClassLoader());
        } catch (Exception e) {
            LOGGER.error(e.getMessage());
            return false;
        }
    }

    private static <T> JsonInternalFacade<T> getInstance() {
        if (hasJsonClazz(CLAZZ_JACKSON)) {
            return JacksonFacade.getInstance();
        } else if (hasJsonClazz(CLAZZ_FASTJSON)) {
            return FastJsonFacade.getInstance();
        } else if (hasJsonClazz(CLAZZ_GSON)) {
            return GsonFacade.getInstance();
        }
        throw CustomException.create("无JSON依赖适配器，如Jackson、FastJson、Gson");
    }

    public static String toJson(Object object) {
        try {
            return getInstance().toJson(object);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * @param object
     * @param sensitiveFields
     * @return
     */
    public static String toJsonWithSensitive(Object object, List<String> sensitiveFields) {
        try {
            if (hasJsonClazz(CLAZZ_JACKSON)) {
                return JacksonFacade.getInstance().toJsonWithSensitive(object, sensitiveFields);
            }
            return toJson(object);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static String toJsonWithSensitive(Object object) {
        try {
            if (hasJsonClazz(CLAZZ_JACKSON)) {
                return JacksonFacade.getInstance().toJsonWithSensitive(new MaskDataConfig<>(object));
            }
            return toJson(object);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static String toJsonWithSensitive(MaskDataConfig config) {
        try {

            if (hasJsonClazz(CLAZZ_JACKSON)) {
                return JacksonFacade.getInstance().toJsonWithSensitive(config);
            }
            return toJson(config.getData());
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static <T> T fromJson(String str, Class<T> clazz) {
        try {
            return str == null ? null : getInstance().fromJson(str, clazz);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static <T> List<T> fromJsonArray(String str, Class<T> clazz) {
        try {
            return str == null ? null : getInstance().fromJsonToArray(str, clazz);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }


    public interface SensitiveJsonInternalFacade<T> {
        String toJsonWithSensitive(Object object, List<String> sensitiveFields) throws Throwable;
        String toJsonWithSensitive(MaskDataConfig config) throws Throwable;

        String toJsonWithIgnoreSensitiveFields(Object object, List<String> sensitiveFields);
    }

    public interface JsonInternalFacade<T> {
        String toJson(Object object) throws Throwable;

        <T> T fromJson(String str, Class<T> clazz) throws Throwable;

        <T> List<T> fromJsonToArray(String str, Class<T> clazz) throws Throwable;
    }

    public static class JacksonFacade<T> implements JsonInternalFacade<T>, SensitiveJsonInternalFacade<T> {
        private JacksonFacade() {
            LOGGER.debug("init Jackson..");
        }

        private static final class LazyHolder {
            private static final JacksonFacade INSTANCE = new JacksonFacade<>();
        }

        public static JacksonFacade getInstance() {
            return JacksonFacade.LazyHolder.INSTANCE;
        }

        private static ObjectMapper objectMapper;
        private static ObjectMapper sensitiveObjectMapper;

        private static ObjectMapper getJacksonInstance() {
            if (objectMapper == null) {
                LOGGER.debug("init Jackson ObjectMapper");
                objectMapper = new ObjectMapper();
                objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            }
            return objectMapper;
        }

        private static ObjectMapper getSensitiveJacksonInstance() {
            if (sensitiveObjectMapper == null) {
                LOGGER.debug("init sensitive Jackson ObjectMapper");
                sensitiveObjectMapper = new ObjectMapper();
                sensitiveObjectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                sensitiveObjectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);

                SimpleModule module = new SimpleModule();
                module.addSerializer(MaskDataConfig.class, new MaskJacksonSerializer(MaskDataConfig.class));
                sensitiveObjectMapper.registerModule(module);
                //sensitiveObjectMapper.addMixIn(Object.class, IgnoreSensitiveFieldsMixin.class);
            }
            return sensitiveObjectMapper;
        }

        @JsonIgnoreProperties(value = {"password", "secret", "token", "price"})
        private static class IgnoreSensitiveFieldsMixin {
        }

        @Override
        public String toJson(Object object) throws Throwable {
            return getJacksonInstance().writeValueAsString(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) throws Throwable {
            return getJacksonInstance().readValue(str, clazz);
        }


        @Override
        public String toJsonWithSensitive(Object object, List<String> sensitiveFields) throws Throwable {
            return getSensitiveJacksonInstance().writeValueAsString(new MaskDataConfig<>(object, sensitiveFields));
        }

        @Override
        public String toJsonWithSensitive(MaskDataConfig config) throws Throwable {
            return getSensitiveJacksonInstance().writeValueAsString(config);
        }

        @Override
        public String toJsonWithIgnoreSensitiveFields(Object object, List<String> sensitiveFields) {
            return null;
        }

        @Override
        public <T> List<T> fromJsonToArray(String str, Class<T> clazz) throws Throwable {
            return getJacksonInstance().readValue(str, getJacksonInstance().getTypeFactory().constructParametricType(List.class, clazz));
        }
    }

    public static class FastJsonFacade<T> implements JsonInternalFacade<T> {
        private FastJsonFacade() {
            LOGGER.debug("init FastJson..");
        }

        private static final class LazyHolder {
            private static final FastJsonFacade INSTANCE = new FastJsonFacade<>();
        }

        public static FastJsonFacade getInstance() {
            return FastJsonFacade.LazyHolder.INSTANCE;
        }

        @Override
        public String toJson(Object object) {
            return JSON.toJSONString(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) {
            return JSON.parseObject(str, clazz);
        }

        @Override
        public <T> List<T> fromJsonToArray(String str, Class<T> clazz) throws Throwable {
            return JSON.parseArray(str, clazz);
        }
    }

    public static class GsonFacade<T> implements JsonInternalFacade<T> {
        private GsonFacade() {
            LOGGER.debug("init Gson..");
        }

        private static final class LazyHolder {
            private static final GsonFacade INSTANCE = new GsonFacade<>();
        }

        private static Gson gson;

        private static Gson getGsonInstance() {
            if (gson == null) {
                LOGGER.debug("init Gson");
                gson = new GsonBuilder() //gson = new Gson();
                        .setDateFormat(DateUtil.LOCALE_FORMAT)
                        //.serializeNulls()
                        .create();
            }
            return gson;
        }

        public static GsonFacade getInstance() {
            return GsonFacade.LazyHolder.INSTANCE;
        }

        @Override
        public String toJson(Object object) {
            return getGsonInstance().toJson(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) {
            return getGsonInstance().fromJson(str, clazz);
        }

        @Override
        public <T> List<T> fromJsonToArray(String str, Class<T> clazz) throws Throwable {
            return getGsonInstance().fromJson(str, new ParameterizedTypeImpl(clazz));
        }

        private class ParameterizedTypeImpl implements ParameterizedType {
            Class clazz;

            public ParameterizedTypeImpl(Class clazz) {
                this.clazz = clazz;
            }

            @Override
            public Type[] getActualTypeArguments() {
                return new Type[]{clazz};
            }

            @Override
            public Type getRawType() {
                return List.class;
            }

            @Override
            public Type getOwnerType() {
                return null;
            }
        }
    }
}