package cn.pengh.util;

import cn.pengh.exception.CustomException;
import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ClassUtils;

/**
 * 依次适配，Jackson、FastJson、Gson
 * @author Created by pengh
 * @datetime 2021/6/4 10:59
 */
public class JsonUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(JsonUtil.class);
    private static final String CLAZZ_JACKSON = "com.fasterxml.jackson.databind.ObjectMapper";
    private static final String CLAZZ_FASTJSON = "com.alibaba.fastjson.JSON";
    private static final String CLAZZ_GSON = "com.google.gson.Gson";

    // 依赖spring-core的ClassUtils来判断
    private static boolean hasJsonClazz(String clazzName) {
        try {
            return ClassUtils.isPresent(clazzName, JsonUtil.class.getClassLoader());
        } catch (Exception e) {
            LOGGER.error(e.getMessage());
            return false;
        }
    }

    private static <T> JsonInternalFacde<T> getInstance() {
        if (hasJsonClazz(CLAZZ_JACKSON)) {
            return JacksonFacde.getInstance();
        } else if (hasJsonClazz(CLAZZ_FASTJSON)) {
            return FastJsonFacde.getInstance();
        } else if (hasJsonClazz(CLAZZ_GSON)) {
            return GsonFacde.getInstance();
        }
        throw CustomException.create("无JSON依赖适配器，如Jackson、FastJson、Gson");
    }

    public static String toJson(Object object) {
        try {
            return getInstance().toJson(object);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static <T> T fromJson(String str, Class<T> clazz) {
        try {
            return str == null ? null : getInstance().fromJson(str, clazz);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    private interface JsonInternalFacde<T> {
        String toJson(Object object) throws Throwable;
        <T> T fromJson(String str, Class<T> clazz) throws Throwable;
    }

    public static class JacksonFacde<T> implements JsonInternalFacde<T> {
        private JacksonFacde() {
            LOGGER.debug("init Jackson..");
        }
        private static final class LazyHolder {
            private static final JacksonFacde INSTANCE = new JacksonFacde<>();
        }
        public static JacksonFacde getInstance() {
            return JacksonFacde.LazyHolder.INSTANCE;
        }
        private static ObjectMapper objectMapper;
        private static ObjectMapper getJacksonInstance() {
            if (objectMapper == null) {
                LOGGER.debug("init Jackson ObjectMapper");
                objectMapper = new ObjectMapper();
                objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            }
            return objectMapper;
        }
        @Override
        public String toJson(Object object) throws Throwable {
            return getJacksonInstance().writeValueAsString(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) throws Throwable  {
            return getJacksonInstance().readValue(str, clazz);
        }
    }

    public static class FastJsonFacde<T> implements JsonInternalFacde<T> {
        private FastJsonFacde() {
            LOGGER.debug("init FastJson..");
        }
        private static final class LazyHolder {
            private static final FastJsonFacde INSTANCE = new FastJsonFacde<>();
        }
        public static FastJsonFacde getInstance() {
            return FastJsonFacde.LazyHolder.INSTANCE;
        }
        @Override
        public String toJson(Object object) {
            return JSON.toJSONString(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz)  {
            return JSON.parseObject(str, clazz);
        }
    }

    public static class GsonFacde<T> implements JsonInternalFacde<T> {
        private GsonFacde() {
            LOGGER.debug("init Gson..");
        }
        private static final class LazyHolder {
            private static final GsonFacde INSTANCE = new GsonFacde<>();
        }
        private static Gson gson;
        private static Gson getGsonInstance() {
            if (gson == null) {
                LOGGER.debug("init Gson");
                gson = new GsonBuilder() //gson = new Gson();
                        .setDateFormat(DateUtil.LOCALE_FORMAT)
                        //.serializeNulls()
                        .create();
            }
            return gson;
        }
        public static GsonFacde getInstance() {
            return GsonFacde.LazyHolder.INSTANCE;
        }
        @Override
        public String toJson(Object object) {
            return getGsonInstance().toJson(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) {
            return getGsonInstance().fromJson(str, clazz);
        }
    }
}