package cn.pengh.util;

import cn.pengh.exception.CustomException;
import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ClassUtils;

/**
 * 自主适配
 * @author Created by pengh
 * @datetime 2021/6/4 10:59
 */
public class JsonUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(JsonUtil.class);
    private static final String CLAZZ_JACKSON = "com.fasterxml.jackson.databind.ObjectMapper";
    private static final String CLAZZ_FASTJSON = "com.alibaba.fastjson.JSON";
    private static final String CLAZZ_GSON = "com.google.gson.Gson";

    private static boolean hasJsonClazz(String clazzName) {
        return ClassUtils.isPresent(clazzName, JsonUtil.class.getClassLoader());
    }

    private static <T> JsonInternal<T> getInstance() {
        if (hasJsonClazz(CLAZZ_JACKSON)) {
            return JacksonInternal.getInstance();
        } else if (hasJsonClazz(CLAZZ_FASTJSON)) {
            return FastJsonInternal.getInstance();
        } else if (hasJsonClazz(CLAZZ_GSON)) {
            return GsonInternal.getInstance();
        }
        throw CustomException.create("无JSON依赖适配器，如Jackson、FastJson、Gson");
    }

    public static String toJson(Object object) {
        try {
            return getInstance().toJson(object);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static <T> T fromJson(String str, Class<T> clazz) {
        try {
            return str == null ? null : getInstance().fromJson(str, clazz);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    private interface JsonInternal<T> {
        String toJson(Object object) throws Throwable;
        <T> T fromJson(String str, Class<T> clazz) throws Throwable;
    }

    private static class JacksonInternal<T> implements JsonInternal<T> {
        private JacksonInternal() {
            LOGGER.debug("init Jackson..");
        }
        private static final class LazyHolder {
            private static final JacksonInternal INSTANCE = new JacksonInternal<>();
        }
        private static JacksonInternal getInstance() {
            return JacksonInternal.LazyHolder.INSTANCE;
        }
        private static ObjectMapper objectMapper;
        public static ObjectMapper getJacksonInstance() {
            if (objectMapper == null) {
                LOGGER.debug("init Jackson ObjectMapper");
                objectMapper = new ObjectMapper();
                objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            }
            return objectMapper;
        }
        @Override
        public String toJson(Object object) throws Throwable {
            return getJacksonInstance().writeValueAsString(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) throws Throwable  {
            return getJacksonInstance().readValue(str, clazz);
        }
    }

    private static class FastJsonInternal<T> implements JsonInternal<T> {
        private FastJsonInternal() {
            LOGGER.debug("init FastJson..");
        }
        private static final class LazyHolder {
            private static final FastJsonInternal INSTANCE = new FastJsonInternal<>();
        }
        private static FastJsonInternal getInstance() {
            return FastJsonInternal.LazyHolder.INSTANCE;
        }
        @Override
        public String toJson(Object object) {
            return JSON.toJSONString(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz)  {
            return JSON.parseObject(str, clazz);
        }
    }

    private static class GsonInternal<T> implements JsonInternal<T> {
        private GsonInternal() {
            LOGGER.debug("init Gson..");
        }
        private static final class LazyHolder {
            private static final GsonInternal INSTANCE = new GsonInternal<>();
        }
        private static GsonInternal getInstance() {
            return GsonInternal.LazyHolder.INSTANCE;
        }
        @Override
        public String toJson(Object object) {
            return new Gson().toJson(object);
        }

        @Override
        public <T> T fromJson(String str, Class<T> clazz) {
            return new Gson().fromJson(str, clazz);
        }
    }
}