package cn.pengh.util;

import cn.pengh.exception.CustomException;
import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ClassUtils;

/**
 * 自主适配
 * @author Created by pengh
 * @datetime 2021/6/4 10:59
 */
public class JsonUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(JsonUtil.class);
    private static final String CLAZZ_JACKSON = "com.fasterxml.jackson.databind.ObjectMapper";
    private static final String CLAZZ_FASTJSON = "com.alibaba.fastjson.JSON";
    private static final String CLAZZ_GSON = "com.google.gson.Gson";

    private static boolean hasJsonClazz(String clazzName) {
        return ClassUtils.isPresent(clazzName, JsonUtil.class.getClassLoader());
    }

    private static <T> Json<T> getInstance() {
        if (hasJsonClazz(CLAZZ_JACKSON)) {
            return new JacksonInternal<>();
        } else if (hasJsonClazz(CLAZZ_FASTJSON)) {
            LOGGER.debug("init FastJson");
            return new FastJsonInternal<>();
        } else if (hasJsonClazz(CLAZZ_GSON)) {
            LOGGER.debug("init Gson");
            return new GsonInternal<>();
        }
        throw CustomException.create("无JSON适配器，如Jackson、FastJson、Gson");
    }

    public static String toJSONString(Object object) {
        try {
            return getInstance().writeValueAsString(object);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    public static <T> T parseObject(String str, Class<T> clazz) {
        try {
            return str == null ? null : getInstance().readValue(str, clazz);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    private interface Json<T> {
        String writeValueAsString(Object object) throws Throwable;
        <T> T readValue(String str, Class<T> clazz) throws Throwable;
    }
    private static class JacksonInternal<T> implements Json<T> {
        private static ObjectMapper objectMapper;
        public static ObjectMapper getInstance() {
            if (objectMapper == null) {
                LOGGER.debug("init jackson ObjectMapper");
                objectMapper = new ObjectMapper();
                objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            }
            return objectMapper;
        }
        @Override
        public String writeValueAsString(Object object) throws Throwable {
            return getInstance().writeValueAsString(object);
        }

        @Override
        public <T> T readValue(String str, Class<T> clazz) throws Throwable  {
            return getInstance().readValue(str, clazz);
        }
    }

    private static class FastJsonInternal<T> implements Json<T> {
        @Override
        public String writeValueAsString(Object object) {
            return JSON.toJSONString(object);
        }

        @Override
        public <T> T readValue(String str, Class<T> clazz)  {
            return JSON.parseObject(str, clazz);
        }
    }

    private static class GsonInternal<T> implements Json<T> {
        @Override
        public String writeValueAsString(Object object) {
            return new Gson().toJson(object);
        }

        @Override
        public <T> T readValue(String str, Class<T> clazz) throws Throwable  {
            return new Gson().fromJson(str, clazz);
        }
    }
}