package cn.pengh.http;

import cn.pengh.helper.ClazzHelper;
import cn.pengh.util.CurrencyUtil;
import okhttp3.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 依赖 "com.squareup.okhttp3:okhttp:$okhttp_version",
 *
 * @author Created by pengh
 * @datetime 2020/6/12 11:02
 */
public class OkHttpUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(OkHttpUtil.class);

    private static Map<Integer, OkHttpClient> okHttpClientMap = new ConcurrentHashMap<>();
    private static final int TIMEOUT = 5000; //默认5s

    private static class OkHttpClientBuilderLazyHolder {
        private static final OkHttpClient.Builder INSTANCE = new OkHttpClient.Builder();
    }

    private static OkHttpClient getInstance(int timeout) {
        OkHttpClient okHttpClient = okHttpClientMap.get(timeout);
        int connectTimeout = timeout / 5; //连接超时，默认/5即可，最低100ms
        connectTimeout = connectTimeout == 0 ? 100 : connectTimeout;

        if (okHttpClient == null) {
            if (okHttpClientMap.size() > 10) {
                okHttpClientMap.clear();
            }
            OkHttpClientBuilderLazyHolder.INSTANCE
                    .retryOnConnectionFailure(true) //默认true，connect失败时重试一次
                    .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
                    .readTimeout(timeout, TimeUnit.MILLISECONDS)
                    .writeTimeout(timeout, TimeUnit.MILLISECONDS);

            okHttpClient = OkHttpClientBuilderLazyHolder.INSTANCE.build();
            okHttpClientMap.put(timeout, okHttpClient);
        }
        return okHttpClient;
    }

    public static String post(String url, Object obj) {
        return post(url, ClazzHelper.getKVMap(obj));
    }

    public static String post(String url, Map<String, String> param) {
        return post(url, param, TIMEOUT);
    }

    public static String post(String url, Map<String, String> param, int timeout) {
        return post(url, param, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String get(String url) {
        return get(url, null);
    }

    public static String postJson(String url, String json) {
        return postJson(url, json, TIMEOUT);
    }

    public static String postJson(String url, String json, int timeout) {
        return postJson(url, json, HttpRequest.HttpRequestConfig.createDefault().setTimeout(timeout));
    }

    public static String postJson(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return doHttp(new Request.Builder().url(url).post(RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json)), config);
    }

    public static String post(String url, Map<String, String> param, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", param);
        FormBody.Builder formBodyBuilder = new FormBody.Builder();
        param.forEach((k, v) -> formBodyBuilder.add(k, v));
        return doHttp(new Request.Builder().url(url).post(formBodyBuilder.build()), config);
    }

    public static String putJson(String url, String json) {
        return putJson(url, json, null);
    }
    public static String putJson(String url, String json, HttpRequest.HttpRequestConfig config) {
        LOGGER.debug("Params: {}", json);
        return doHttp(new Request.Builder().url(url).put(RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json)), config);
    }

    public static String get(String url, HttpRequest.HttpRequestConfig config) {
        return doHttp(new Request.Builder().url(url).get(), config);
    }


    private static String doHttp(Request.Builder requestBuilder, HttpRequest.HttpRequestConfig config) {
        long start = System.nanoTime();

        int timeout = TIMEOUT;
        if (config != null) {
            timeout = config.getTimeout();
            Map<String, String> headers = config.getHeaders();
            if (headers != null) {
                headers.forEach((k, v) -> requestBuilder.addHeader(k, v));
            }
        }

        Request request = requestBuilder.build();
        LOGGER.debug("{} {} Timeout {}ms", request.method(), request.url().toString(), timeout);
        //LOGGER.debug("Params: {}", JsonUtil.toJSONString(request.body()));
        try {
            Response response = getInstance(timeout).newCall(request).execute();
            String res = response.body().string();
            LOGGER.debug("{}", response.headers().toString().replace("\n", " "));
            LOGGER.debug("Response {}: {}", response.code(), res);
            return res;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            LOGGER.debug("Time elapsed: {}s", CurrencyUtil.divide(System.nanoTime() - start, 1e9, 6));
        }

    }
}
