package cn.pengh.util;

import cn.pengh.exception.CustomException;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.*;

public class DateUtil {
    private static final Map<String, Integer> monthQuarterMap = new HashMap<String, Integer>();
    public static final String YEAR_FORMAT = "yyyy";
    public static final String MONTH_FORMAT = "yyyyMM";
    public static final String DAY_FORMAT = "yyyyMMdd";
    public static final String DAY_FORMAT_DASH = "yyyy-MM-dd";
    public static final String LOCALE_FORMAT = "yyyy-MM-dd HH:mm:ss";
    public static final String TIME_FORMAT = "yyyyMMddHHmmss";
    public static final String FORMAT_HHmmss = "HHmmss";
    public static final String FORMAT_HHmmssSSS = "HHmmssSSS";
    public static final String FORMAT_LOG = "yy/MM/dd HH:mm:ss.SSS";

    public static final DateTimeFormatter DT_LOCALE_FORMAT = DateTimeFormatter.ofPattern(LOCALE_FORMAT);
    public static final DateTimeFormatter DT_TIME_FORMAT = DateTimeFormatter.ofPattern(TIME_FORMAT);
    public static final DateTimeFormatter DT_DAY_FORMAT = DateTimeFormatter.ofPattern(DAY_FORMAT);
    public static final DateTimeFormatter DT_DAY_FORMAT_DASH = DateTimeFormatter.ofPattern(DAY_FORMAT_DASH);
    public static final DateTimeFormatter DT_DAY_FORMAT_YYMMDD = DateTimeFormatter.ofPattern("yyMMdd");
    public static final DateTimeFormatter DT_MONTH_FORMAT = DateTimeFormatter.ofPattern(MONTH_FORMAT);
    public static final DateTimeFormatter DT_FORMAT_LOG = DateTimeFormatter.ofPattern(FORMAT_LOG);

    private static final int NOON = 12;

    static {
        monthQuarterMap.put("01", 1);
        monthQuarterMap.put("02", 1);
        monthQuarterMap.put("03", 1);
        monthQuarterMap.put("04", 2);
        monthQuarterMap.put("05", 2);
        monthQuarterMap.put("06", 2);
        monthQuarterMap.put("07", 3);
        monthQuarterMap.put("08", 3);
        monthQuarterMap.put("09", 3);
        monthQuarterMap.put("10", 4);
        monthQuarterMap.put("11", 4);
        monthQuarterMap.put("12", 4);
    }

    public static Date toDate(LocalDate localDate) {
        return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    public static Date toDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    public static LocalDate toLocalDate(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }

    public static LocalDateTime toLocalDateTime(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
    }

    public static String toLocaleTime() {
        return LocalDateTime.now().format(DT_LOCALE_FORMAT);
    }

    public static String toLocaleTime(String date) {
        return toLocaleTime(date, DAY_FORMAT);
    }

    public static String toLocaleTime(String date, String format) {
        return LocalDateTime.parse(date, DateTimeFormatter.ofPattern(format)).format(DT_LOCALE_FORMAT);
    }


    public static int today() {
        return Integer.valueOf(getCurrDay());
    }

    public static int nextDay(int day) {
        return Integer.valueOf(getNextDay(day));
    }


    public static int nextDay(int today, int day) {
        return Integer.valueOf(getNextDay(today, day));
    }

    public static String getCurrDay() {
        return LocalDate.now().format(DT_DAY_FORMAT);
    }

    public static String getCurrDayYyMMdd() {
        return LocalDate.now().format(DT_DAY_FORMAT_YYMMDD);
    }


    public static String getCurrDay(String format) {
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern(format));
    }

    public static String getYesterdayDate(String date) {
        return getNextDay(date, -1);
    }

    /**
     * 当月 201407
     *
     * @return
     */
    public static String getCurrMonth() {
        return getCurrMonth(MONTH_FORMAT);
    }

    public static String getCurrMonth(String format) {
        return LocalDate.now().format(DateTimeFormatter.ofPattern(format));
    }

    /**
     * 上一月 201406
     *
     * @return
     */
    public static String getLastMonth() {
        return getLastMonth(MONTH_FORMAT);
    }

    public static String getLastMonthYear() {
        return getLastMonth(YEAR_FORMAT);
    }

    /**
     * 上一月
     *
     * @param format
     * @return
     */
    public static String getLastMonth(String format) {
        return LocalDate.now().plusMonths(-1).format(DateTimeFormatter.ofPattern(format));
    }

    /**
     * 获取某月的下一月
     *
     * @param currMonth
     * @param format
     * @return
     */
    private static String getNextMonth(String currMonth, String format, boolean isNext) {
        return LocalDate.parse(currMonth, DT_MONTH_FORMAT).plusMonths(isNext ? 1 : -1).format(DateTimeFormatter.ofPattern(format));
    }

    public static String getNextMonth(String currMonth) {
        return getNextMonth(currMonth, MONTH_FORMAT, true);
    }

    public static String getNextMonth() {
        return LocalDate.now().plusMonths(1).format(DT_MONTH_FORMAT);
    }

    public static String getNextMonth(int month, String format) {
        return LocalDate.now().plusMonths(month).format(DateTimeFormatter.ofPattern(format));
    }

    public static String getNextMonth(int month) {
        return getNextMonth(month, DAY_FORMAT);
    }

    public static int nextMonth(int month) {
        return Integer.valueOf(getNextMonth(month));
    }

    public static String getPrevMonth(String currMonth) {
        return getNextMonth(currMonth, MONTH_FORMAT, false);
    }

    public static String getPrevMonth() {
        return getLastMonth();
    }

    /**
     * @param currDay yyyyMMdd
     * @param format
     * @param day
     * @return
     */
    public static String getNextDay(String currDay, String format, int day) {
        return LocalDate.parse(currDay, DT_DAY_FORMAT).plusDays(day).format(DateTimeFormatter.ofPattern(format));
    }

    public static String getNextDay(String currDay, int day) {
        return getNextDay(currDay, DAY_FORMAT, day);
    }

    public static String getNextDay(int currDay, int day) {
        return getNextDay(currDay + "", day);
    }


    /**
     * useage getNextDay(1) or getNextDay(-1)
     *
     * @param day
     * @return
     */
    public static String getNextDay(int day, String format) {
        return LocalDateTime.now().plusDays(day).format(DateTimeFormatter.ofPattern(format));
    }

    public static String getNextDate(String currDay, int day) {
        return getNextDay(currDay, day);
    }

    public static String getLastDate(String currDay, int day) {
        return getNextDay(currDay, -day);
    }

    public static String getNextDay(int day) {
        return getNextDay(day, DAY_FORMAT);
    }

    public static Date getNextMin(int min) {
        return toDate(LocalDateTime.now().plusMinutes(min));
    }


    /**
     * useage getNextDate(1) or getNextDate(-1)
     *
     * @param day
     * @return
     */
    public static Date getNextDate(int day) {
        return toDate(LocalDate.now().plusDays(day));
        //return new DateTime().plusDays(day).toDate();
    }

    public static Date getNextDate(Date date, int day) {
        return toDate(LocalDateTime.now().plusDays(day));
        //return new DateTime(date).plusDays(day).toDate();
    }

    public static Date getNextDateByStr(String date, int day) {
        return toDate(LocalDate.parse(date, DT_DAY_FORMAT).plusDays(day));
        //return new DateTime(str2Date(date)).plusDays(day).toDate();
    }

    private static String getNextMonthDay(String currDay, String format, boolean isNext) {
        return LocalDate.parse(currDay, DT_DAY_FORMAT).plusMonths(isNext ? 1 : -1).format(DateTimeFormatter.ofPattern(format));
    }

    public static String getNextMonthDay(String currDay, String format) {
        return getNextMonthDay(currDay, format, true);
    }

    public static String getNextMonthDay(String currDay) {
        return getNextMonthDay(currDay, DAY_FORMAT);
    }

    public static String getNextMonthDay() {
        //return getNextMonthDay(getCurrDay());
        return LocalDate.now().plusMonths(1).format(DT_DAY_FORMAT);
    }

    public static String getPrevMonthDay(String currDay, String format) {
        return getNextMonthDay(currDay, format, false);
    }

    public static String getPrevMonthDay(String currDay) {
        return getNextMonthDay(currDay, DAY_FORMAT);
    }

    public static String getPrevMonthDay() {
        return getNextMonthDay(getCurrDay());
    }

    /**
     * Date -> String
     *
     * @param date
     * @return
     */
    public static String Date2Str(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime().format(DT_DAY_FORMAT);
    }

    /**
     * Date -> String
     *
     * @param date
     * @return
     */
    public static String Date2Str(Date date, String fmt) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime().format(DateTimeFormatter.ofPattern(fmt));
    }

    /**
     * String -> Date
     *
     * @param ds
     * @return
     */
    public static Date str2Date(String ds) {
        if (ds.length() != 8) {
            throw CustomException.create("格式不对(yyyyMMdd)");
        }
        return str2Date(ds, DAY_FORMAT);
    }

    /**
     * 指定格式，String -> Date
     *
     * @param ds
     * @param format yyyyMMdd yyyy-MM-dd yyyy/MM/dd
     * @return
     */
    public static Date str2Date(String ds, String format) {
        if (format.indexOf("M") < 0) {
            ds += "0101";
            format += "MMdd";
        } else if (format.indexOf("d") < 0) {
            ds += "01";
            format += "dd";
        }
        //System.out.println("ds = [" + ds + "], format = [" + format + "]");
        return toDate(LocalDate.parse(ds, DateTimeFormatter.ofPattern(format)));
    }


    /**
     * 获取当前月份前num个月的yyyyMM
     *
     * @param num
     * @return
     */
    public static String getPreviousMonthYyyyMM(String yyyyMM, int num) {
        return LocalDate.parse(yyyyMM, DT_MONTH_FORMAT).plusMonths(num).format(DT_MONTH_FORMAT);
    }

    public static String getDateStrYyyyMM(Date date) {
        return Date2Str(date, MONTH_FORMAT);
    }

    public static Date getDateByUnixLong(Long unixTime) {
        return unixTime == null || unixTime == 0L ? null : getDateByUnixStr(unixTime + "");
    }

    public static Date getDateByUnixStr(String unixTime) {
        return StringUtil.isEmpty(unixTime) ? null : new Date(1000 * Long.parseLong(unixTime));
    }

    public static LocalDateTime getLocalDateTimeByUnixStr(String unixTime) {
        return toLocalDateTime(getDateByUnixStr(unixTime));
    }

    public static LocalDateTime getLocalDateTimeByUnixLong(Long unixTime) {
        return toLocalDateTime(getDateByUnixLong(unixTime));
    }

    public static Date getDateByYyyyMM(String yyyyMM) {
        return toDate(LocalDate.parse(yyyyMM, DT_MONTH_FORMAT));
    }

    /**
     * 获取当前所属季度
     *
     * @param
     * @return
     */
    public static int getQuarterByMM(String MM) {
        return monthQuarterMap.get(MM);
    }

    /**
     * 返回上一月所在季度
     *
     * @return
     */
    public static int getLastQuarter() {
        return getQuarterByMM(getLastMonth("MM"));
    }

    public static boolean isSameMonth(String preMonth, String month) {
        return preMonth.substring(4, 6).equals(month.substring(4, 6));
    }

    /**
     * 获取本月第一天
     *
     * @param
     * @return
     */
    public static String getMonthFirstDay(String yyyyMMdd) {
        return LocalDate.parse(yyyyMMdd, DT_DAY_FORMAT).with(TemporalAdjusters.firstDayOfMonth()).format(DT_DAY_FORMAT);
    }

    /**
     * 获取日期的最后一天
     *
     * @param
     * @return
     */
    public static String getMonthLastDay(String yyyyMMdd) {
        return LocalDate.parse(yyyyMMdd, DT_DAY_FORMAT).with(TemporalAdjusters.lastDayOfMonth()).format(DT_DAY_FORMAT);
    }

    /**
     * 获取本月第一天
     *
     * @param
     * @return
     */
    public static String getMonthFirstDayYyyyMM(String month) {
        return getMonthFirstDay(month + "01");
    }

    /**
     * 获取本月最后一天
     *
     * @param
     * @return
     */
    public static String getMonthLastDayYyyyMM(String month) {
        return getMonthLastDay(month + "01");
    }

    public static int getMonthLastDayInt(String month) {
        String lastDay = getMonthLastDayYyyyMM(month);
        return Integer.valueOf(lastDay.substring(6, 8));
    }


    /**
     * 获取下月第一天
     */
    public static String getNextMonthFirstDayYyyyMM(String month) {
        return getNextMonthFirstDay(month + "01");
    }

    /**
     * 获取下月第一天
     */
    public static String getNextMonthFirstDay(String dateStr) {
        return LocalDate.parse(dateStr, DT_DAY_FORMAT).plusMonths(1).with(TemporalAdjusters.firstDayOfMonth()).format(DT_DAY_FORMAT);
    }

    /**
     * @param beginMonth
     * @param endMonth
     * @return
     */
    public static List<String> getTotalMonthList(String beginMonth, String endMonth) {
        List<String> tMtList = new ArrayList<String>();
        LocalDate beginLocalDate = LocalDate.parse(beginMonth + "01", DT_DAY_FORMAT);
        LocalDate endLocalDate = LocalDate.parse(endMonth + "01", DT_DAY_FORMAT);
        while (beginLocalDate.compareTo(endLocalDate) <= 0) {
            tMtList.add(beginLocalDate.format(DT_MONTH_FORMAT));
            beginLocalDate = beginLocalDate.plusMonths(1);
        }
        return tMtList;
    }

    /**
     * 获取月最后一天或者deadline中最小的那天
     *
     * @param month
     * @param deadline
     * @return
     */
    public static String getMonthLastDayOrDeadlineElier(String month, String deadline) {
        String lastDay = getMonthLastDayYyyyMM(month);
        return lastDay.compareTo(deadline) > 0 ? deadline : lastDay;
    }

    /**
     * 获取年月信息通过yyyyMMdd
     *
     * @param yyyyMMdd
     * @return
     */
    public static String getMonthByDate(String yyyyMMdd) {
        return yyyyMMdd.substring(0, 6);
    }

    public static String getMonthFirstDayOrBeginDateLater(String month, String beginDate) {
        String firstDate = getMonthFirstDayYyyyMM(month);
        return firstDate.compareTo(beginDate) < 0 ? beginDate : firstDate;
    }

    public static boolean isAftenoonHr(int hr) {
        return hr >= NOON;
    }

    public static boolean isMorningHr(int hr) {
        return hr <= NOON;
    }

    //00:00 -- 08:59
    public static boolean isEarlyMorning() {
        return isHourBefore(9);
    }

    //几点之前
    public static boolean isHourBefore(int hour) {
        int h = LocalDateTime.now().getHour();
        return 0 <= h && h < hour;
    }

    //几分之前
    public static boolean isMinuteBefore(int min) {
        int m = LocalDateTime.now().getMinute();
        return 0 <= m && m < min;
    }

    /**
     * isInPerHour(3) ==> return true when current hour in(0,3,6,9,12,15,18,21)
     * isInPerHour(10) ==> return true when current hour in(0,10,20)
     *
     * @param
     * @return
     */
    public static boolean isInGapHour(int gap) {
        if (gap < 1 || gap > 24)
            return false;
        List<Integer> list = new ArrayList<Integer>();
        for (int i = 0; i < 24; i++) {
            list.add(i);
            i += gap - 1;
        }
        //Log.info(list.toString());
        return list.contains(LocalDateTime.now().getHour());
    }


    public static boolean isSmallLater(int workLateMinute) {
        return workLateMinute < 10;
    }

    public static boolean isBigLater(int workLateMinute) {
        return workLateMinute >= 10 && workLateMinute < 60;
    }


    public static int yearGap(String dateBegin) {
        return yearGap(dateBegin, getCurrDay());
    }

    /**
     * 日期格式写死
     *
     * @param dateBegin 20150505
     * @param dateEnd   20150505
     * @return
     */
    public static int yearGap(String dateBegin, String dateEnd) {
        if (StringUtil.isEmpty(dateBegin) || StringUtil.isEmpty(dateEnd) || dateEnd.compareTo(dateBegin) <= 0)
            return 0;
        String dif = (Integer.valueOf(dateEnd) - Integer.valueOf(dateBegin)) + "";
        if (dif.length() > 4)
            dif = dif.substring(0, dif.length() - 4);
        else
            return 0;
        int age = Integer.valueOf(dif);
        return age < 0 ? 0 : age;
    }

    //must date like 20150505
    public static String format(String date) {
        return format(date, 0);
    }

    /**
     * @param date       --> 20150505
     * @param formatType 0 --> 20150505
     *                   1 --> 2015-05-05
     *                   2 --> 2015/05/05
     *                   3 --> 2015.05.05
     *                   4 --> 2015年05月05日
     * @return
     */
    public static String format(String date, int formatType) {
        if (date.length() < 6)
            return date;
        switch (formatType) {
            case 0:
                return date;
            case 4:
                return date.substring(0, 4) + "年" + date.substring(4, 6) + "月" + (date.length() < 8 ? "" : date.substring(6, 8) + "日");
            default:
                return getFormat(date, formatType);
        }
    }

    public static String formatInt(int date, int formatType) {
        return format(String.valueOf(date), formatType);
    }

    private static String getFormat(String day, int type) {
        return day.substring(0, 4) + FORMAT_STYLE_MAP.get(type) + day.substring(4, 6)
                + (day.length() < 8 ? "" : FORMAT_STYLE_MAP.get(type) + day.substring(6, 8));
    }

    //4 == 3 / 0.75
    private static Map<Integer, String> FORMAT_STYLE_MAP = new HashMap<Integer, String>(4) {
        private static final long serialVersionUID = 4215529268333155347L;

        {
            put(1, "-");
            put(2, "/");
            put(3, ".");
        }
    };

    public static long getDaysBetween(Date startDate, Date endDate) {
        return getDaysBetween(toLocalDate(startDate), toLocalDate(endDate));
    }

    public static long getDaysBetween(String startDate, String endDate) {
        return getDaysBetween(LocalDate.parse(startDate, DT_DAY_FORMAT), LocalDate.parse(endDate, DT_DAY_FORMAT));
    }

    /**
     * 包含起始、不包含截止日期
     *
     * @param startDate 20120201
     * @param endDate   20120301
     * @return
     */
    public static long getDaysBetween(LocalDate startDate, LocalDate endDate) {
        long count = 0;
        while (startDate.compareTo(endDate) < 0) {
            count++;
            startDate = startDate.plusDays(1);
        }
        return count;
    }

}
