package cn.pengh.library;

import cn.pengh.util.FileUtil;
import cn.pengh.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * @author Created by pengh
 * @datetime 2018/8/28 15:18
 */
public class ConfigReaderHolder {
	private final static Logger logger = LoggerFactory.getLogger(ConfigReaderHolder.class);

	private ConfigReaderHolder() { }
	private static class ConfigReaderSingletonLazyHolder {
		private static volatile ConfigReaderHolder INSTANCE = new ConfigReaderHolder();
	}

	/**
	 * 初始化
	 * @param env 环境：development、production、uat
	 * @param root 配置文件存放根目录
	 * @param customConfig 自定义配置
	 * @param cache 缓存，默认false 即不使用缓存，每次都是加载文件
	 * @return
	 */
	public static ConfigReaderHolder getInstance(String env, String root, boolean cache, ICustomConfig customConfig) {
		ConfigReaderHolder c = ConfigReaderSingletonLazyHolder.INSTANCE;
		c.setEnvironment(env);
		c.setRoot(root);
		c.cache = cache;
		c.setCustomConfig(customConfig);
		ConfigReaderSingletonLazyHolder.INSTANCE = c;

		logger.debug("ConfigReaderHolder init, env:{}, rootSub:{}, customConfig:{}. root: {}", env, root, customConfig, c.ROOT);
		return c;
	}

	public static ConfigReaderHolder getInstance(String env, String root, boolean cache) {
		return getInstance(env, root, cache, null);
	}

	public static ConfigReaderHolder getInstance(String env, String root) {
		return getInstance(env, root, false);
	}

	public static ConfigReaderHolder getInstance() {
		return ConfigReaderSingletonLazyHolder.INSTANCE;
	}

	public static final class ENVIRONMENT {
		public static final String DEVELOPMENT = "development";
		public static final String PRODUCTION = "production";
		public static final String DEV = "dev";
		public static final String PRO = "pro";
		public static final String UAT = "uat";
	}

	public static final class PROP {
		public static final String FUIOU = "fuiou";
		public static final String PERSISTENCE = "persistence";
	}

	public static boolean isProductionEnvironment() {
		return isProductionEnvironment;
	}
	public static final String CONFIG_ENVIRONMENT = "config.environment";

	private String PROP_SUFFIX = ".properties";
	private String ROOT = Thread.currentThread().getContextClassLoader().getResource("").getPath();
	private String ROOT_RESOURCES = ROOT.replaceAll("(.*\\/)(classes)\\/?$","$1resources");
	private String ROOT_SUB = "";
	private boolean cache = false;


	private static final Map<String, Properties> propMap = new HashMap<String, Properties>();
	//TODO
	private static final Map<String, Long> propFileLastTimeMap = new HashMap<String, Long>();
	private ICustomConfig customConfig;
	
	private static boolean isProductionEnvironment;

	
	private void setEnvironment(String environment) {
		if (StringUtil.isEmpty(environment))
			return;
		PROP_SUFFIX = "-"+environment+PROP_SUFFIX;
		isProductionEnvironment = ENVIRONMENT.PRODUCTION.equals(environment);
	}
	private void setRoot(String root) {
		ROOT_SUB = "/"+root+"/";
		ROOT += "/"+ROOT_SUB;
		ROOT_RESOURCES += "/"+ROOT_SUB;
	}
	private void setCustomConfig(ICustomConfig rc) {
		customConfig = rc;
	}

	/**
	 * @param propName
	 */
	private void loadProperties(String propName) {
		Properties prop = null;
		if(cache) {
			prop = propMap.get(propName);

			if (prop == null) {
				prop = new Properties();
			} else {
				return;
			}
		} else {
			prop = new Properties();
		}

		InputStream fis = null;
		try {
			try {
				//fis = new FileInputStream(ROOT + propName + PROP_SUFFIX);
				try {
					fis = new FileInputStream(ROOT + propName + PROP_SUFFIX);
				} catch (Exception e) {
					//Log.getLogger().error("{}。 try getResourceAsStream()", e.getMessage());
					//e.printStackTrace();
					//src/main/resources的目录下时，只能使用Class.getResourceAsStream()方法来加载
					//Log.debug(ROOT_SUB + propName + PROP_SUFFIX);
					fis = ConfigReaderHolder.class.getResourceAsStream(ROOT_SUB + propName + PROP_SUFFIX);
				}
			} catch (Exception e) {
				//Log.error(e.getMessage()+"。try ROOT_RESOURCES");
				//e.printStackTrace();
				fis = new FileInputStream(ROOT_RESOURCES +propName + PROP_SUFFIX);//为了本地测试

			}


			//
			prop.load(fis);
			propMap.put(propName,prop);
		} catch (Exception e) {
			Log.error(e.getMessage());
			throw new RuntimeException("Read "+propName+" fail:"+e.getMessage(),e);
		} finally{
			try {
				if(fis != null)
					fis.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	/**
	 * @param propName
	 * @param key
	 */
	public String getConfig(String propName, String key) {
		loadProperties(propName);		
		Properties prop = propMap.get(propName);
		if(!prop.containsKey(key)){
			Log.error("Prop ["+propName+"]no this key:"+key);
			return null;
			//throw new RuntimeException("Prop ["+propName+"]no this key:"+key);
		}
		return prop.getProperty(key);
	}
	
	public void setConfig(String propName, String key, String val) {
		FileUtil.ModPropFile(ROOT+propName+PROP_SUFFIX,key, val);
	}
	
	public String getConfig(String key) {
		return getConfig(PROP.FUIOU, key);
	}
	
	//自定义配置信息
	public final String getCustomConfig(String key){
		if(customConfig == null)
			throw new RuntimeException("请初始化cn.pengh.library.ConfigReader.ICustomConfig接口类");
		String conf = null;
		try {
			conf = customConfig.getCustomConfig(key);
		} catch (Exception e) {
			//e.printStackTrace();
			logger.error("----------ConfigReader::getCustomConfig is null: {}, error msg: {}",key,e.getMessage());
		}		
		return conf;
	}
	
	
	/**
	 * 	加个类com.fuiou.ready.common.config.FuiouKeyLoadConfig
		public class FuiouKeyLoadConfig implements ICustomConfig {		
			@Override
			public String getRemoteConfig(String key) {
				return SysKeyLoaderUtil.loadSysKey(ConfigReader.getConfig(Consts.UNION_KEY_URL),Consts.UNION_PLAT_ID, key);
			}			
		}
	 * spring配置文件
	 * <bean class="cn.pengh.library.ConfigReader">
        	<property name="environment" value="development"/>
        	<property name="root" value="consts"/>
        	<property name="customConfig">
	        	<bean class="com.fuiou.ready.common.config.FuiouKeyLoadConfig"/>
	        </property>
    	</bean>    	
	 * 只需这样使用就可以了ConfigReader.getRemoteConfig(busiCd)
	 * @author penghcn
	 * @created 2016年2月17日下午2:24:05
	 */
	public interface ICustomConfig {
		String getCustomConfig(String key);
	}
	
}
