package cn.pengh.util;

import cn.pengh.crypt.Base64;
import cn.pengh.io.UnicodeReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * @author penghcn
 * @created 2014年11月14日下午1:53:51
 */
public class FileUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtil.class);

    public static final int BUFFER_LENGTH_64K = 1024 * 64; //默认64k内存
    public static final int BUFFER_LENGTH_2M = 2097152; //2M内存
    public static final String CHARSET_DEFAULT = "utf-8";
    private final static double S_B = 1024d, S_KB = 1048576d, S_MB = 1073741824d, S_GB = 1099511627776d, S_TB = 1125899906842624d, S_PB = 1152921504606846976d;

    // 关键代码 执行序列化和反序列化 进行深度拷贝
    public static <T> List<T> deepCopy(List<T> src) {
        try (ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
             ByteArrayInputStream byteIn = new ByteArrayInputStream(byteOut.toByteArray());
             ObjectOutputStream out = new ObjectOutputStream(byteOut);
             ObjectInputStream in = new ObjectInputStream(byteIn)) {

            out.writeObject(src);
            return (List<T>) in.readObject();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    // String sql = GetFileContents(new org.springframework.core.io.ClassPathResource("/consts/txn.sql.selects").getInputStream())
    public static String GetFileContents(InputStream in) {
        StringBuilder sb = new StringBuilder();
        //scan=new BufferedReader(new InputStreamReader(new FileInputStream(filePath)));
        try (BufferedReader scan = new BufferedReader(new UnicodeReader(in, CHARSET_DEFAULT))) {
            while (true) {
                String line = scan.readLine();
                if (line == null) {
                    break;
                }
                sb.append(line);
            }
            return sb.toString();
        } catch (Exception e) {
            e.printStackTrace();
            throw new IllegalArgumentException("无法获取文件");
        }
    }

    public static String GetFileContents(String filePath) {
        try {
            return GetFileContents(new FileInputStream(filePath));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            throw new IllegalArgumentException("无法获取文件");
        }
    }

    public static boolean WriteContentsToFile(String filePath, String content) {
        File file = new File(filePath);
        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
            changeFolderPermission(file.getParentFile());
        }
        try {
            if (!file.exists()) {
                file.createNewFile();
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }

        try (FileOutputStream fos = new FileOutputStream(file)) {
            LOGGER.info("WriteStringToFile: {}", filePath);
            fos.write(content.getBytes());
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    public static byte[] ReadFile(String root) {
        try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(root));
             ByteArrayOutputStream out = new ByteArrayOutputStream(BUFFER_LENGTH_64K)) {

            LOGGER.info("Available bytes: {}", formatFileSize(in.available()));

            byte[] temp = new byte[BUFFER_LENGTH_64K];
            int size = 0;
            while ((size = in.read(temp)) != -1) {
                out.write(temp, 0, size);
            }
            in.close();

            byte[] content = out.toByteArray();
            LOGGER.info("Readed bytes count: {}", formatFileSize(content.length));
            return content;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    public static void ModPropFile(String path, String key, String val) {
        ModUniqueSingleLine(path, key, key + "=" + val);
    }

    public static void ModUniqueSingleLine(String path, String lineStart, String line) {
        StringBuilder sb = new StringBuilder();
        try (BufferedReader in = new BufferedReader(new FileReader(path));
             OutputStreamWriter out = new OutputStreamWriter(new FileOutputStream(path))) {
            String str = null;
            while ((str = in.readLine()) != null) {
                if (str.startsWith(lineStart)) {
                    sb.append(line);
                } else {
                    sb.append(str);
                }
                sb.append(System.getProperty("line.separator"));
            }
            in.close();
            out.write(sb.toString());
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    /**
     * 递归删除文件夹
     *
     * @param path
     */
    public static void DirDelte(String path) {
        File dirFile = null;
        try {
            dirFile = new File(path);

            if ((dirFile.exists() && dirFile.isDirectory())) {
                for (File f : dirFile.listFiles()) {
                    if (f.isDirectory()) {
                        DirDelte(f.getAbsolutePath());
                    } else {
                        f.delete();
                    }
                }

            }
            dirFile.delete();
        } catch (Exception e) {
            e.printStackTrace();
        }
        LOGGER.trace("Done...递归删除文件夹：{}", path);
    }

    public static void FileDelEmpty(String root) {
        removeEmptyDirs(visitAllDirs(new File(root)));
    }

    //得到某一目录下的所有文件夹
    private static List<File> visitAllDirs(File root) {
        List<File> list = new ArrayList<File>();
        visitAllDirs(root, list);
        return list;
    }

    //递归获取
    private static void visitAllDirs(File root, List<File> list) {
        File[] dirs = root.listFiles();
        if (dirs == null) {
            return;
        }
        for (File dir : dirs) {
            if (dir.isDirectory()) {
                list.add(dir);
            }
            visitAllDirs(dir, list);
        }
    }

    private static void removeEmptyDirs(List<File> dirs) {
        for (File dir : dirs) {
            if (!(dir.isDirectory() && (dir.list() == null || dir.list().length <= 0))) {
                continue;
            }
            LOGGER.debug("Empty Dir {}", dir.getAbsolutePath());
            dir.delete();
        }
    }

    public static boolean FileCopy(String fs, String fd) {
        return FileCopy(new File(fs), new File(fd));
    }

    public static boolean FileCopy(String fs, File fd) {
        return FileCopy(new File(fs), fd);
    }

    public static boolean FileCopy(File fs, String fd) {
        return FileCopy(fs, new File(fd));
    }

    /**
     * 复制文件
     *
     * @param fs
     * @param fd
     */
    public static boolean FileCopy(File fs, File fd) {
        long st = System.nanoTime();
        if (!fd.getParentFile().exists()) {
            LOGGER.debug("--create ParentFile: {}", fd.getParentFile().getAbsolutePath());
            fd.getParentFile().mkdirs();
            changeFolderPermission(fd.getParentFile());
        }
        try (FileInputStream fin = new FileInputStream(fs);
             FileOutputStream out = new FileOutputStream(fd)) {
            byte[] buffer = new byte[BUFFER_LENGTH_2M];

            while (true) {
                int ins = fin.read(buffer);
                if (ins == -1) {
                    fin.close();
                    out.flush();
                    out.close();
                    break;
                } else {
                    out.write(buffer, 0, ins);
                }
            }
            LOGGER.debug("--Spends: {}ms", CurrencyUtil.divide(System.nanoTime() - st, 1e6));
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public static boolean FileCopy(InputStream fin, String fd) {
        return FileCopy(fin, new File(fd));
    }

    public static boolean FileCopy(InputStream fin, File fd) {
        if (!fd.getParentFile().exists()) {
            LOGGER.debug("--create ParentFile: {}", fd.getParentFile().getAbsolutePath());
            fd.getParentFile().mkdirs();
            changeFolderPermission(fd.getParentFile());
        }
        try {
            return FileCopy(fin, new FileOutputStream(fd));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return false;
        }
    }

    public static boolean FileCopy(InputStream fin, OutputStream fout) {
        long st = System.nanoTime();
        try {
            byte[] buffer = new byte[BUFFER_LENGTH_2M];

            while (true) {
                int ins = fin.read(buffer);
                if (ins == -1) {
                    fin.close();
                    fout.flush();
                    fout.close();
                    break;
                } else {
                    fout.write(buffer, 0, ins);
                }
            }
            LOGGER.debug("--Spends: {}ms", CurrencyUtil.divide(System.nanoTime() - st, 1e6));
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (fin != null) {
                    fin.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                if (fout != null) {
                    fout.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

    }

    private static void changeFolderPermission(File dirFile) {
        Set<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
        perms.add(PosixFilePermission.OWNER_READ);
        perms.add(PosixFilePermission.OWNER_WRITE);
        perms.add(PosixFilePermission.OWNER_EXECUTE);
        perms.add(PosixFilePermission.GROUP_READ);
        perms.add(PosixFilePermission.GROUP_WRITE);
        perms.add(PosixFilePermission.GROUP_EXECUTE);
        //chmod 770
        try {
            Path path = Paths.get(dirFile.getAbsolutePath());
            Files.setPosixFilePermissions(path, perms);
        } catch (Exception e) {
            LOGGER.error("Change folder {} permission failed.", dirFile.getAbsolutePath());
        }
    }

    public static String formatFileSize(long fileSize) {
        if (fileSize > Long.MAX_VALUE || fileSize < S_B) {
            return fileSize + "B";
        } else if (fileSize < S_KB) {
            return CurrencyUtil.convert(fileSize / S_B, 3).toString() + "K";
        } else if (fileSize < S_MB) {
            return CurrencyUtil.convert(fileSize / S_KB, 3).toString() + "M";
        } else if (fileSize < S_GB) {
            return CurrencyUtil.convert(fileSize / S_MB, 3).toString() + "G";
        } else if (fileSize < S_TB) {
            return CurrencyUtil.convert(fileSize / S_GB, 3).toString() + "T";
        } else if (fileSize < S_PB) {
            return CurrencyUtil.convert(fileSize / S_TB, 3).toString() + "P";
        } else {
            return CurrencyUtil.convert(fileSize / S_PB, 3).toString() + "E";
        }
    }

    public static String getImageBase64(File imgFile) {
        try {
            return getImageBase64(ImageIO.read(imgFile));
        } catch (Throwable e) {
            return null;
        }

    }

    public static String getImageBase64(String imgFile) {
        return getImageBase64(new File(imgFile));
    }

    public static String getImageBase64(BufferedImage img) {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
            ImageIO.write(img, "png", baos);
            byte[] bytes = baos.toByteArray();
            return Base64.encode(bytes).trim();
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static File unzip(String fileName) {
        return unzip(fileName, null);
    }

    public static File unzip(String fileName, String destDir) {
        File file = new File(fileName);
        //默认解压在当前目录下的同名文件夹中
        if (destDir == null) {
            destDir = file.getParentFile().getAbsolutePath() + "/" + file.getName().replaceAll("(.*).zip$", "$1");
            LOGGER.debug("default unzip dir: {}", destDir);
        }
        File destDirFile = new File(destDir);
        if (!destDirFile.exists()) {
            destDirFile.mkdirs();
        }
        try {
            ZipFile zip = new ZipFile(file, Charset.forName("GBK")); //中文乱码问题
            for (Enumeration entries = zip.entries(); entries.hasMoreElements(); ) {
                ZipEntry entry = (ZipEntry) entries.nextElement();
                String outPath = (destDir + "/" + entry.getName()).replaceAll("\\*", "/"); //转成linux
                LOGGER.debug("unzip file: {}", outPath);

                File outFile = new File(outPath);
                if (outFile.isDirectory()) {
                    continue;
                }

                if (!outFile.getParentFile().exists()) {
                    outFile.getParentFile().mkdirs();
                }

                try (InputStream in = zip.getInputStream(entry);
                     BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(outFile))) {
                    byte[] buffer = new byte[FileUtil.BUFFER_LENGTH_2M];
                    int len;
                    while ((len = in.read(buffer)) != -1) {
                        out.write(buffer, 0, len);
                    }
                }
            }

            return destDirFile;
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }
}
