package cn.payingcloud.umf;

import cn.payingcloud.umf.model.AccessToken;
import cn.payingcloud.umf.model.Payment;
import cn.payingcloud.umf.model.PaymentMethod;
import cn.payingcloud.umf.model.Refund;
import cn.payingcloud.umf.util.HttpUtils;
import cn.payingcloud.umf.util.JsonUtils;
import cn.payingcloud.umf.util.SignUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

public class UmfClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(UmfClient.class);
    private final UmfProfile profile;
    private final PrivateKey privateKey;
    private final X509Certificate certificate;
    private final TokenProvider tokenProvider;
    private boolean verifyCallbackSign = true;
    private boolean verifyResponseSign = false;

    public UmfClient(UmfProfile profile, PrivateKey privateKey, X509Certificate certificate, TokenProvider tokenProvider) {
        Validate.notNull(profile, "profile must not be null");
        Validate.notNull(privateKey, "privateKey must not be null");
        Validate.notNull(certificate, "certificate must not be null");
        Validate.notNull(tokenProvider, "tokenProvider must not be null");
        this.profile = profile;
        this.privateKey = privateKey;
        this.certificate = certificate;
        this.tokenProvider = tokenProvider;
    }

    public void setVerifyCallbackSign(boolean verifyCallbackSign) {
        this.verifyCallbackSign = verifyCallbackSign;
    }

    public void setVerifyResponseSign(boolean verifyResponseSign) {
        this.verifyResponseSign = verifyResponseSign;
    }

    public static AccessToken getAccessToken(UmfProfile profile, String clientId, String clientSecret) {
        String url = profile.getUrl() + "/oauth/authorize";
        Map<String, String> params = new HashMap<>();
        params.put("client_id", clientId);
        params.put("client_secret", clientSecret);
        params.put("grant_type", "client_credentials");
        String json = JsonUtils.toJson(params);
        String response = HttpUtils.doPost(url, json);
        return JsonUtils.fromJson(response, AccessToken.class);
    }

    public PlaceOrderResponse placeOrder(Payment payment) {
        payment.encrypt(certificate);
        String url = profile.getUrl() + "/payments/payment";
        PlaceOrderResponse response = post(url, payment, PlaceOrderResponse.class);
        if (response.successful() && isBankCardPay(payment)) {
            payment.setId(response.getPaymentId());
            return verifyCode(payment);
        }
        return response;
    }

    public PlaceOrderResponse execute(Payment payment) {
        Validate.notNull(payment, "payment can't be null");
        String url = profile.getUrl() + "/payments/payment/" + payment.getId() + "/execute";
        return post(url, payment, PlaceOrderResponse.class);
    }

    public QueryOrderResponse queryOrder(String paymentId) {
        Validate.notNull(paymentId, "paymentId can't be null");
        String url = profile.getUrl() + "/payments/payment/" + paymentId;
        return get(url, QueryOrderResponse.class);
    }

    public RefundResponse refund(String paymentId, Refund refund) {
        Validate.notNull(paymentId, "paymentId can't be null");
        String url = profile.getUrl() + "/payments/payment/" + paymentId + "/refund";
        return post(url, refund, RefundResponse.class);
    }

    public QueryRefundResponse queryRefund(String refundId) {
        Validate.notNull(refundId, "refundId can't be null");
        String url = profile.getUrl() + "/payments/refund/" + refundId;
        return get(url, QueryRefundResponse.class);
    }

    public QueryTransactionResponse queryTransaction(Date date) {
        String merDate = DateFormatUtils.format(date, "yyMMdd");
        String url = profile.getUrl() + "/payments/?mer_date=" + merDate;
        return get(url, QueryTransactionResponse.class);
    }

    public <T extends CallbackRequest> T resolveCallbackRequest(String sign, String data, Class<T> tClass) {
        LOGGER.debug("Resolving Callback Request, Sign: {}, Json: {}", sign, data);
        if (verifyCallbackSign && !SignUtils.verify(certificate, sign, data)) {
            throw new UmfException("Failed to pass callback sign verification");
        }
        return JsonUtils.fromJson(data, tClass);
    }

    public String generateCallbackResponseData(CallbackRequest callback) {
        return "{\"meta\":{\"ret_code\":\"0000\",\"ret_msg\":[\"SUCCESS\"]}}";
    }

    private PlaceOrderResponse verifyCode(Payment payment) {
        Validate.notNull(payment, "payment can't be null");
        String url = profile.getUrl() + "/payments/payment/" + payment.getId() + "/verify";
        return post(url, payment, PlaceOrderResponse.class);
    }

    private <T extends Response> T get(String url, Class<T> tClass) {
        LOGGER.debug("Send Request, Url: {}", url);
        String response = HttpUtils.doGet(url, tokenProvider.getToken());
        LOGGER.debug("Response Json: {} ", response);
        T result = JsonUtils.fromJson(response, tClass);
        if (verifyResponseSign && result.successful()) { // 只在请求成功时需要验证签名
            SignUtils.verify(certificate, result.getMeta().getSign(), response);
        }
        return result;
    }

    private <T extends Response> T post(String url, Object obj, Class<T> tClass) {
        String json = JsonUtils.toJson(obj);
        LOGGER.debug("Send Request, Url: {}, Json: {}", url, json);
        String sign = SignUtils.sign(privateKey, json);
        String response = HttpUtils.doPost(url, json, tokenProvider.getToken(), sign);
        LOGGER.debug("Response Json: {} ", response);
        T result = JsonUtils.fromJson(response, tClass);
        if (verifyResponseSign && result.successful()) { // 只在请求成功时需要验证签名
            SignUtils.verify(certificate, result.getMeta().getSign(), response);
        }
        return result;
    }

    private boolean isBankCardPay(Payment payment) {
        return PaymentMethod.CREDIT_CARD == payment.getPayer().getPaymentMethod() || PaymentMethod.DEBIT_CARD == payment.getPayer().getPaymentMethod();
    }
}
