package cn.payingcloud.umf.util;

import org.apache.commons.codec.digest.DigestUtils;

import java.io.*;
import java.nio.charset.Charset;
import java.util.Base64;

/**
 * @author YQ.Huang
 */
public class StreamUtils {
    private StreamUtils() {
    }

    private final static int BUFFER_SIZE = 4096;

    public static InputStream loadResource(String name) {
        return StreamUtils.class.getResourceAsStream(name);
    }

    /**
     * Base64 encode the InputStream and calculate MD5 digest
     *
     * @param in InputStream to be encoded.
     * @return Array of two String, 0: base64, 1: md5
     */
    public static String[] base64(InputStream in) {
        String[] result = new String[2];
        byte[] bytes = copyToByteArray(in);
        result[0] = Base64.getEncoder().encodeToString(bytes);
        result[1] = DigestUtils.md5Hex(bytes);
        return result;
    }

    /**
     * Copy the contents of the given InputStream into a new byte array.
     * Leaves the stream open when done.
     *
     * @param in the stream to copy from
     * @return the new byte array that has been copied to
     */
    public static byte[] copyToByteArray(InputStream in) {
        if (in == null) throw new IllegalArgumentException("No InputStream specified");
        ByteArrayOutputStream out = new ByteArrayOutputStream(BUFFER_SIZE);
        copy(in, out);
        return out.toByteArray();
    }

    /**
     * Copy the contents of the given InputStream into a String.
     * Leaves the stream open when done.
     *
     * @param in      the InputStream to copy from
     * @param charset charset to use
     * @return the String that has been copied to
     */
    public static String copyToString(InputStream in, Charset charset) {
        if (in == null) throw new IllegalArgumentException("No InputStream specified");
        if (charset == null) throw new IllegalArgumentException("No Charset specified");
        try {
            StringBuilder out = new StringBuilder();
            InputStreamReader reader = new InputStreamReader(in, charset);
            char[] buffer = new char[BUFFER_SIZE];
            int bytesRead;
            while ((bytesRead = reader.read(buffer)) != -1) {
                out.append(buffer, 0, bytesRead);
            }
            return out.toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Copy the contents of the given byte array to the given OutputStream.
     * Leaves the stream open when done.
     *
     * @param in  the byte array to copy from
     * @param out the OutputStream to copy to
     */
    public static void copy(byte[] in, OutputStream out) {
        if (in == null) throw new IllegalArgumentException("No input byte array specified");
        if (out == null) throw new IllegalArgumentException("No OutputStream specified");
        try {
            out.write(in);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Copy the contents of the given String to the given output OutputStream.
     * Leaves the stream open when done.
     *
     * @param in      the String to copy from
     * @param charset the Charset
     * @param out     the OutputStream to copy to
     */
    public static void copy(String in, Charset charset, OutputStream out) {
        if (in == null) throw new IllegalArgumentException("No input byte array specified");
        if (out == null) throw new IllegalArgumentException("No OutputStream specified");
        if (charset == null) throw new IllegalArgumentException("No Charset specified");
        try {
            Writer writer = new OutputStreamWriter(out, charset);
            writer.write(in);
            writer.flush();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Copy the contents of the given InputStream to the given OutputStream.
     * Leaves both streams open when done.
     *
     * @param in  the InputStream to copy from
     * @param out the OutputStream to copy to
     * @return the number of bytes copied
     */
    public static int copy(InputStream in, OutputStream out) {
        try {
            if (in == null) throw new IllegalArgumentException("No InputStream specified");
            if (out == null) throw new IllegalArgumentException("No OutputStream specified");
            int byteCount = 0;
            byte[] buffer = new byte[BUFFER_SIZE];
            int bytesRead;
            while ((bytesRead = in.read(buffer)) != -1) {
                out.write(buffer, 0, bytesRead);
                byteCount += bytesRead;
            }
            out.flush();
            return byteCount;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
