package cn.patterncat.cache.component.key;

import org.springframework.cache.Cache;
import org.springframework.cache.caffeine.CaffeineCache;
import org.springframework.cache.concurrent.ConcurrentMapCache;
import org.springframework.data.redis.cache.RedisCache;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * https://github.com/ipalbeniz/spring-boot-caches-endpoint/blob/master/src/main/java/com/demo/autoconfiguration/cachekeys/CacheKeysExtractorFactory.java
 */
public class CacheKeysExtractorFactory {

    private static final Map<Class, CacheKeysExtractor> KEYS_EXTRACTORS_BY_CACHE_CLASS = new ConcurrentHashMap<>();

    private CacheKeysExtractorFactory() {
        // Avoid instantiation
    }

    public static CacheKeysExtractor getKeyExtractor(Cache cache) {
        return KEYS_EXTRACTORS_BY_CACHE_CLASS.computeIfAbsent(cache.getClass(),
                k -> createCacheKeysExtractor(cache));
    }

    private static CacheKeysExtractor createCacheKeysExtractor(Cache cache) {
        if (cache instanceof ConcurrentMapCache) {
            return new ConcurrentMapCacheKeysExtractor();
        } else if (cache instanceof CaffeineCache) {
            return new CaffeineCacheKeysExtractor();
        } else if (cache instanceof RedisCache) {
            return new RedisCacheKeysExtractor();
        } else {
            return new NoOpCacheKeysExtractor();
        }
    }
}
