package cn.patterncat.cache.component;

import cn.patterncat.rest.ApiResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.endpoint.annotation.DeleteOperation;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.boot.actuate.endpoint.annotation.Selector;
import org.springframework.boot.actuate.endpoint.web.WebEndpointResponse;
import org.springframework.boot.actuate.endpoint.web.annotation.EndpointWebExtension;
import org.springframework.data.util.Pair;

import java.util.List;

/**
 * 此类适用于修改原始的CacheKeysEndpoint自动暴露的rest api，但是好像有时候覆盖会出错
 * CacheKeysEndpointWebExtension映射跟CacheKeysEndpoint的requestPredicate的path都一样，导致delete请求直接调用了CacheKeysEndpoint的deleteCacheEntryByKey方法而非CacheKeysEndpointWebExtension里头的方法
 *
 * 唯一的差别在org/springframework/web/servlet/mvc/condition/ProducesRequestCondition.java
 * WebExtension的produces为[application/vnd.spring-boot.actuator.v2+json || application/json]
 * 自动生成的produces为[]
 * NOTE 因而请求的时候需要带上 header "Accept: application/json"
 *
 * org/springframework/boot/actuate/endpoint/web/servlet/AbstractWebMvcEndpointHandlerMapping.java
 * org/springframework/boot/actuate/endpoint/web/annotation/WebEndpointDiscoverer.java
 * https://github.com/spring-projects/spring-boot/issues/13261
 *
 * Created by cat on 2019-03-12.
 */
@EndpointWebExtension(endpoint = CacheKeysEndpoint.class)
public class CacheKeysEndpointWebExtension {

    private static final Logger LOGGER = LoggerFactory.getLogger(CacheKeysEndpointWebExtension.class);

    private CacheKeysEndpoint delegate;

    public CacheKeysEndpointWebExtension(CacheKeysEndpoint cacheKeysEndpoint) {
        this.delegate = cacheKeysEndpoint;
    }

    @ReadOperation
    public WebEndpointResponse getCacheKeys(@Selector String cacheManagerName, @Selector String cacheName) {
        try {
            Object result = delegate.getCacheKeys(cacheManagerName,cacheName);
            return new WebEndpointResponse(ApiResult.success(result), WebEndpointResponse.STATUS_OK);
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(),ex);
            return new WebEndpointResponse(ApiResult.fail(ex.getMessage()),WebEndpointResponse.STATUS_BAD_REQUEST);
        }
    }

    @ReadOperation
    public WebEndpointResponse getValueByKey(@Selector String cacheManagerName, @Selector String cacheName, @Selector String key){
        try{
            Object result = delegate.getValueByKey(cacheManagerName,cacheName,key);
            //NOTE 不知道为啥object mapper返回{}
            return new WebEndpointResponse(ApiResult.success(result), WebEndpointResponse.STATUS_OK);
        }catch (Exception ex){
            LOGGER.error(ex.getMessage(),ex);
            return new WebEndpointResponse(ApiResult.fail(ex.getMessage()),WebEndpointResponse.STATUS_BAD_REQUEST);
        }
    }

    @DeleteOperation
    public WebEndpointResponse deleteCacheEntryByKey(@Selector String cacheManagerName, @Selector String cacheName, @Selector String key){
        try{
            delegate.deleteCacheEntryByKey(cacheManagerName,cacheName,key);
            return new WebEndpointResponse(ApiResult.success(),WebEndpointResponse.STATUS_OK);
        }catch (Exception ex){
            LOGGER.error(ex.getMessage(),ex);
            return new WebEndpointResponse(ApiResult.fail(ex.getMessage()),WebEndpointResponse.STATUS_OK);
        }
    }
}
