package org.springframework.cloud.sleuth.instrument.web;

import brave.ErrorParser;
import brave.SpanCustomizer;
import brave.http.HttpAdapter;
import brave.http.HttpServerParser;
import cn.patterncat.tracing.ExtraTracingProperties;

import javax.servlet.http.HttpServletResponse;

/**
 * SleuthHttpServerParser
 * Created by cat on 2019-01-23.
 */
public class ExtraSleuthHttpServerParser extends HttpServerParser {

    private static final String STATUS_CODE_KEY = "http.status";

    private static final String NO_RESPONSE_STATUS = "-1";

    private final ExtraSleuthHttpClientParser clientParser;

    private final ErrorParser errorParser;

    ExtraSleuthHttpServerParser(TraceKeys traceKeys, ExtraTracingProperties properties, ErrorParser errorParser) {
        //note 这里是否有必要再new，直接共用一个ExtraSleuthHttpClientParser
        this.clientParser = new ExtraSleuthHttpClientParser(traceKeys,properties);
        this.errorParser = errorParser;
    }

    @Override
    protected ErrorParser errorParser() {
        return this.errorParser;
    }

    @Override
    protected <Req> String spanName(HttpAdapter<Req, ?> adapter, Req req) {
        return this.clientParser.spanName(adapter, req);
    }

    @Override
    public <Req> void request(HttpAdapter<Req, ?> adapter, Req req,
                              SpanCustomizer customizer) {
        this.clientParser.request(adapter, req, customizer);
    }

    @Override
    public <Resp> void response(HttpAdapter<?, Resp> adapter, Resp res, Throwable error,
                                SpanCustomizer customizer) {
        if (res == null) {
            error(null, error, customizer);
            return;
        }
        Integer httpStatus = adapter.statusCode(res);
        if (httpStatus == null) {
            customizer.tag(STATUS_CODE_KEY, NO_RESPONSE_STATUS);
            error(httpStatus, error, customizer);
            return;
        }
        if (httpStatus == HttpServletResponse.SC_OK && error != null) {
            // Filter chain threw exception but the response status may not have been set
            // yet, so we have to guess.
            customizer.tag(STATUS_CODE_KEY,
                    String.valueOf(HttpServletResponse.SC_INTERNAL_SERVER_ERROR));
        }
        // only tag valid http statuses
        else {
            customizer.tag(STATUS_CODE_KEY, String.valueOf(httpStatus));
        }
        error(httpStatus, error, customizer);
    }
}
