package cn.patterncat.tracing;

import cn.patterncat.tracing.helper.MaskHelper;
import com.google.common.collect.Sets;
import lombok.Getter;
import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.http.MediaType;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Created by cat on 2018-12-05.
 */
@Getter
@Setter
@ConfigurationProperties(prefix = "extra.tracing")
public class ExtraTracingProperties {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExtraTracingProperties.class);

    private boolean enabled = true;

    private boolean traceServletRequestBody = true;

    private List<String> printableMediaSubTypes = new ArrayList<String>(){{
        add(MediaType.APPLICATION_FORM_URLENCODED.getSubtype()); // x-www-form-urlencoded
        add(MediaType.APPLICATION_JSON.getSubtype());  //json
        add(MediaType.APPLICATION_XML.getSubtype()); // xml
//        add(MediaType.TEXT_HTML.getSubtype()); //html, sac接口返回的是text/html
//        add(MediaType.TEXT_MARKDOWN.getSubtype()); //markdown
        add(MediaType.TEXT_PLAIN.getSubtype()); //plain
//        add(MediaType.TEXT_XML.getSubtype()); //xml
    }};

    private int maxPrintBodyBytes = 2048;

    private String servletRequestQueryTag = "http.query";

    private String servletRequestBodyTag = "http.body";

    private boolean traceSpringMvcUrlPattern = false;

    private String springMvcUrlPatternTag = "http.urlPattern";

    private boolean traceRestTemplateRequestBody = true;

    private String httpRequestBodyTag = "http.body";

    private String httpRequestQueryTag = "http.query";

    private boolean extraServletHeaderEnabled = true;

    private String tagMultiHeaderSeparator = ",";

    //tag name --> header name, if multi header name maps to one tag, use first non empty value in order
    private Map<String,String> traceTagHeaderMappings = new HashMap<>();

    private boolean maskEnabled = false;

    private List<String> maskUrlPaths = new ArrayList<>();

    private List<String> maskFields = new ArrayList<String>(){{
        add("password");
    }};

    private String maskReplacement = "******";

    private Pattern maskPattern = MaskHelper.buildMaskPattern(Sets.newHashSet("password"));

    public void setMaskFields(List<String> maskFields) {
        this.maskFields = maskFields;
        maskPattern = MaskHelper.buildMaskPattern(maskFields.stream().collect(Collectors.toSet()));
    }

    public String mask(String data, Supplier<String> urlPathSupplier){
        if(!isMaskEnabled()){
            return data;
        }

        if(StringUtils.isEmpty(data)){
            return data;
        }

        try{
            String urlPath = urlPathSupplier.get();
            if(getMaskUrlPaths().contains(urlPath)){
                String masked = MaskHelper.maskQueryString(data,getMaskPattern(),MaskHelper.DEFAULT_REPLACE_GROUP,getMaskReplacement());
                return masked;
            }
        }catch (Exception e){
            LOGGER.warn(e.getMessage(),e);
        }

        return data;
    }
}
